/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.EventListener;
import java.util.EventObject;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ulcjava.base.application.IApplication;
import com.ulcjava.base.application.ULCPollingTimer;
import com.ulcjava.base.application.event.ActionEvent;
import com.ulcjava.base.application.event.IActionListener;
import com.ulcjava.base.development.DevelopmentRunner;
import com.ulcjava.base.server.ULCSession;


/**
 * ULC application base class that defines a simple lifecycle: <code>initialize</code>,<code>startup</code>,<code>ready</code>,<code>shutdown</code>.
 * <p>
 * The Application <code>start</code> method calls three methods: first <code>initialize()</code>, that subclasses
 * may overwrite to process any initializations that have to be done before the GUI is created. As second method
 * <code>startup()</code> is called. Subclasses must implement this to create and show the GUI. As third
 * <code>ready()</code> is called in the next roundtrip that is triggered by a {@link ULCPollingTimer} after the GUI
 * has been created on the client side.
 * <p>
 * To terminate the Application, use the Action named by the constant <code>Application.QUIT_ACTION</code>.
 * Alternatively call <code>exit()</code> directly form the code. The exit method then asks all registered
 * {@link IExitListener}s if the application can be terminated. If no exit listener interrupts the termination
 * <code>com.ulcjava.base.application.ApplicationContext.terminate()</code> is called. This will then call
 * <code>stop()</code> which wraps <code>shutdown()</code>. <code>shutdown()</code> should be overwritten by
 * subclasses to perform any cleanup processing.
 * <p>
 * The {@link ApplicationContext} provides access to the {@link ResourceManager} and {@link ActionManager}.
 * <p>
 * {@link SingleFrameApplication} extends this class to provide an easy to use jump start application class.
 * 
 * @see SingleFrameApplication
 * @see ApplicationContext
 */
public abstract class Application extends AbstractBean implements IApplication, Serializable {

    /**
     * Interface for classes that get called before the application is terminated.
     * 
     * @see Application#addExitListener(com.ulcjava.applicationframework.application.Application.IExitListener)
     * @see Application#exit(EventObject)
     */
    public interface IExitListener extends EventListener, Serializable {
        /**
         * The application calls this method on each of its registered {@code IExitListener} until one of them returns
         * <code>false</code>. If one of them does return <code>false</code>, the exit process is canceled
         * immediately. Classes implementing this interface can implement <code>canExit(EventObject)</code> to perform
         * some checks, for example if there are any unsaved edits.
         * 
         * @param event that triggered the termination. Can be <code>null</code>.
         * @return <code>true</code> if the termination of the application can continue, <code>false</code> to
         *         cancel the termination.
         */
        boolean canExit(EventObject event);

        /**
         * The application calls this method on each of its registered {@code IExitListener} immediately before the
         * application gets terminated. Classes implementing this interface can implement
         * <code>willExit(EventObject)</code> to perform some cleanup tasks s for example properly closing open files.
         * But beware, the application is going to be terminated afterwards in any case, Exceptions thrown in the method
         * will only be logged.
         * 
         * @param event that triggered the termination. Can be <code>null</code>.
         */
        void willExit(EventObject event);
    }

    private static final Logger LOG = Logger.getLogger(Application.class.getName());
    /**
     * Name of the Action that calls exit to terminate the application.
     */
    public static final String QUIT_ACTION = "quit";
    private final List<IExitListener> fExitListenerList;
    private ApplicationContext fContext;

    /**
     * Creates an Application instance.
     */
    public Application() {
        fExitListenerList = new ArrayList<IExitListener>();
    }

    /**
     * Factory method that creates the {@link ApplicationContext} for the application. Subclasses may overwrite this to
     * create a subclass of {@link ApplicationContext}.
     * 
     * @return the {@link ApplicationContext} created.
     */
    protected ApplicationContext createApplicationContext() {
        return new ApplicationContext(this);
    }

    /**
     * Each {@link ULCSession} has exactly one instance of the Application that is running. This method provides an
     * ubiquitous access to it.
     * 
     * @return the application instance for the current ULCSession.
     */
    public static synchronized Application getInstance() {
        return (Application)com.ulcjava.base.application.ApplicationContext.getApplication();
    }

    /**
     * Calls {@code initialize()} and then {@code startup()}. At the end a {@link ULCPollingTimer} is started to
     * trigger the next roundtrip to call {@code ready()}.
     */
    public final void start() {
        initialize();
        startup();
        ULCPollingTimer readyTrigger = new ULCPollingTimer(10, new IActionListener() {

            public void actionPerformed(ActionEvent event) {
                ready();
            }

        });
        readyTrigger.setRepeats(false);
        readyTrigger.start();
    }

    /**
     * Convenience method to run the application in development environment. subclasses can write a one line main method :
     * <br>
     * <br>
     * <code>run(args);</code>
     * 
     * @param args from main method
     */
    public static void run(String[] args) {
        String launcherClass = "unkown";
        StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
        for (int i = 0; i < stackTraceElements.length; i++) {
            StackTraceElement stackTraceElement = stackTraceElements[i];
            String methodName = stackTraceElement.getMethodName();
            if ("run".equals(methodName)) {
                launcherClass = stackTraceElements[i + 1].getClassName();
                break;
            }

        }
        Class<?> applicationClass;
        try {
            applicationClass = Class.forName(launcherClass);
            DevelopmentRunner.setApplicationClass(applicationClass);
            DevelopmentRunner.main(args);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
    }

    /**
     * Generic main method that starts the Application that is configured in ULCApplicationConfiguration.xml in the
     * {@link DevelopmentRunner}.
     * 
     * @param args command line arguments.
     */
    public static void main(String[] args) throws Exception {
        DevelopmentRunner.main(args);
    }

    /**
     * Calls shutdown(). Subclasses can overwrite shutdown to add code that cleans up before terminating the
     * application. Any Exception thrown by shutdown() are caught and logged.
     */
    public final void stop() {
        try {
            shutdown();
        } catch (Exception e) {
            LOG.log(Level.WARNING, "Exception in shutdown()", e);
        }
    }

    /**
     * @see com.ulcjava.base.application.IApplication#handleMessage(java.lang.String)
     */
    public void handleMessage(String message) {
    }

    /**
     * @see com.ulcjava.base.application.IApplication#activate()
     */
    public void activate() {
    }

    /**
     * @see com.ulcjava.base.application.IApplication#passivate()
     */
    public void passivate() {
    }

    /**
     * @see com.ulcjava.base.application.IApplication#pause()
     */
    public void pause() {
    }

    /**
     * @see com.ulcjava.base.application.IApplication#resume()
     */
    public void resume() {
    }


    /**
     * Calls {@code exit(EventObject)} with a <code>null EventObject</code>.
     * 
     * @see #exit(EventObject)
     */
    public void exit() {
        exit(null);
    }

    /**
     * Calls <code>canExit()</code> on all registered {@link IExitListener}. If none returns false, the listeners
     * <code>willExit()</code> method is called. At the end
     * <code>com.ulcjava.base.application.ApplicationContext.terminate()</code> is called to terminate the
     * application.
     * 
     * @param event the Event, if any, that triggered the termination. Can be <code>null</code>.
     */
    public void exit(EventObject event) {
        for (IExitListener exitListener : fExitListenerList) {
            if (!exitListener.canExit(event)) {
                return;
            }
        }

        for (IExitListener exitListener : fExitListenerList) {
            try {
                exitListener.willExit(event);
            } catch (Exception e) {
                LOG.log(Level.WARNING, "Exception in ExitListener.willExit()", e);
            }
        }
        com.ulcjava.base.application.ApplicationContext.terminate();
    }

    /**
     * Standard action to quit the application. The name is defined by the constant <code>Application.QUIT_ACTION</code>.
     * Use <code>getContext().getActionMap().get(Application.QUIT_ACTION)</code> to retrieve the action. Quit just
     * calls exit with the {@link ActionEvent} as parameter.
     * 
     * @param e action event that has triggered the action.
     */
    @Action(name = QUIT_ACTION)
    public void quit(ActionEvent e) {
        exit(e);
    }

    /**
     * Adds the given {@link IExitListener} to the list of exit listeners.
     * 
     * @param listener to add.
     * @see IExitListener
     * @see #exit(EventObject)
     */
    public void addExitListener(IExitListener listener) {
        fExitListenerList.add(listener);
    }

    /**
     * Removes the given {@link IExitListener} from the list of exit listeners.
     * 
     * @param listener to add.
     * @see IExitListener
     * @see #exit(EventObject)
     */
    public void removeExitListener(IExitListener listener) {
        fExitListenerList.remove(listener);
    }

    /**
     * @return the registered {@link IExitListener}s.
     */
    public IExitListener[] getExitListeners() {
        return fExitListenerList.toArray(new IExitListener[fExitListenerList.size()]);
    }

    /**
     * @return the {@link ApplicationContext} of the application.
     */
    public final ApplicationContext getContext() {
        if (fContext == null) {
            fContext = createApplicationContext();
        }
        return fContext;
    }

    /**
     * Hides the given view. This method delegates to <code>view.hide()</code>.
     * 
     * @param view to hide.
     * @see View#hide()
     */
    public void hide(View view) {
        view.hide();
    }

    /**
     * Shows the given view. This method delegates to <code>view.show()</code>.
     * 
     * @param view to show.
     * @see View#show()
     */
    public void show(View view) {
        view.show();
    }

    /**
     * Subclasses may implement this method to perform initializations that have to be done before the GUI is created in
     * <code>startup()</code>.
     */
    protected void initialize() {
    }

    /**
     * Subclasses must implement this to create and show the GUI. This method is called from <code>start()</code>
     * after <code>initialize()</code>.
     * 
     * @see #show(View)
     */
    protected abstract void startup();

    /**
     * Invoked using a {@link ULCPollingTimer} to immediately start the next roundtrip after the GUI has been created on
     * the client side.
     */
    protected void ready() {
    }

    /**
     * Invoked by {@code stop()} before the application is stopped. Subclasses should overwrite to add code that cleans
     * up before terminating the application. This is the really last code that is executed from the application, for
     * the current {@code ULCSession}.
     */
    protected void shutdown() {
    }

}
