/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import org.apache.commons.beanutils.PropertyUtils;

import com.ulcjava.base.application.AbstractAction;
import com.ulcjava.base.application.IAction;
import com.ulcjava.base.application.event.ActionEvent;
import com.ulcjava.base.application.util.KeyStroke;
import com.ulcjava.base.application.util.ULCIcon;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Action class that gets created for &#064;Action annotated methods and made accessible by an
 * {@link ApplicationActionMap} created with the {@link ActionManager}. The {@code ApplicationAction} uses a
 * {@link ResourceMap} to initialize its properties, the key is defined by the action's name followed by ".Action." and
 * the property, The action's name is set to either the name of the annotated method or by the annotation's name
 * parameter.
 * <p>
 * The resources are mapped to the {@code IAction} properties as follows : <table border>
 * <tr>
 * <th>Resource Key</th>
 * <th>IAction constant</th>
 * </tr>
 * <tr>
 * <td><code><i>actionName</i>.Action.text</code></td>
 * <td><code>IAction.NAME</code></td>
 * </tr>
 * <tr>
 * <td><code><i>actionName</i>.Action.shortDescription</code></td>
 * <td><code>IAction.SHORT_DESCRIPTION</code></td>
 * </tr>
 * <tr>
 * <td><code><i>actionName</i>.Action.longDescription</code></td>
 * <td><code>IAction.LONG_DESCRIPTION</code></td>
 * </tr>
 * <tr>
 * <td><code><i>actionName</i>.Action.accelerator</code></td>
 * <td><code>IAction.ACCELERATOR_KEY</code></td>
 * </tr>
 * <tr>
 * <td><code><i>actionName</i>.Action.icon</code></td>
 * <td><code>IAction.SMALL_ICON</code></td>
 * </tr>
 * <tr>
 * <td><code><i>actionName</i>.Action.command</code></td>
 * <td><code>IAction.ACTION_COMMAND_KEY</code></td>
 * </tr>
 * <tr>
 * <td><code><i>actionName</i>.Action.mnemonic</code></td>
 * <td><tt>IAction.MNEMONIC_KEY</code></td>
 * </tr>
 * </table><br>
 * The MNEMONIC_KEY can also be set by using an '&' in the <i>text</i> property, this overrides a value that is set with
 * the mnemonic resource. The following example sets the mnemonic for the quit action to 'E':
 * 
 * <pre>
 * quit.Action.text=&amp;Exit
 * quit.Action.mnemonic=X
 * </pre>
 * 
 * @see Action
 * @see IAction
 */
public class ApplicationAction extends AbstractAction implements PropertyChangeListener, Serializable {
    private static final String ACTION_PREFIX = ".Action.";

    private static final String ACTION_TEXT = ACTION_PREFIX + "text";
    private static final String ACTION_SHORT_DESCRIPTION = ACTION_PREFIX + "shortDescription";
    private static final String ACTION_LONG_DESCRIPTION = ACTION_PREFIX + "longDescription";
    private static final String ACTION_ACCELERATOR = ACTION_PREFIX + "accelerator";
    private static final String ACTION_ICON = ACTION_PREFIX + "icon";
    private static final String ACTION_COMMAND = ACTION_PREFIX + "command";
    private static final String ACTION_MNEMONIC = ACTION_PREFIX + "mnemonic";


    private static final Logger LOG = Logger.getLogger(Application.class.getName());

    private final ApplicationActionMap fApplicationActionMap;
    private final ResourceMap fResourceMap;
    private final String fActionName;
    private transient Method fActionMethod;
    private final String fEnabledProperty;
    private final Object fActionTarget;

    private transient Method fEnabledPropertyReadMethod;
    private transient Method fEnabledPropertyWriteMethod;

    /**
     * Creates an ApplicationAction with the given name. The action executes the given method on the given
     * {@link ApplicationActionMap}'s actionObject when <code>actionPerformed</code> is called. The given
     * enabledProperty is used to dynamically enable and disable the action. If the resourceMap is not <code>null</code>,
     * it is used to set the values in the action's property map.
     * 
     * @param applicationActionMap the ApplicationActionMap this action is being constructed for. Must not be
     *            <code>null</code>.
     * @param resourceMap initial Action properties are loaded from this ResourceMap.
     * @param actionName the name of the &#064;Action. Must not be <code>null</code>.
     * @param actionMethod actionPerformed calls this method. Must not be <code>null</code>.
     * @param enabledProperty name of the enabling property. Must be a boolean property on the
     *            <code>applicationActionMap</code>'s actionsObject and have property change support.
     */
    public ApplicationAction(ApplicationActionMap applicationActionMap, ResourceMap resourceMap, String actionName, Method actionMethod, String enabledProperty) {
        if (applicationActionMap == null) {
            throw new IllegalArgumentException("applicationActionMap must not be null");
        }
        if (actionName == null) {
            throw new IllegalArgumentException("actionName must not be null");
        }
        if (actionMethod == null) {
            throw new IllegalArgumentException("actionMethod must not be null");
        }

        fApplicationActionMap = applicationActionMap;
        fActionTarget = applicationActionMap.getActionsObject();
        fResourceMap = resourceMap;
        fActionName = actionName;
        fActionMethod = actionMethod;
        fEnabledProperty = enabledProperty;
        if (fEnabledProperty != null) {
            installEnabledBinding();
        } else {
            fEnabledPropertyReadMethod = null;
            fEnabledPropertyWriteMethod = null;
        }
        initProperties();
    }

    private void installEnabledBinding() {
        Object actionsObject = getApplicationActionMap().getActionsObject();
        installMethods(actionsObject);
        installPCL(actionsObject);

    }

    private void installPCL(Object actionsObject) {
        Class<?> actionsClass = actionsObject.getClass();
        Method addPCLMethod = null;
        try {
            addPCLMethod = actionsClass.getMethod("addPropertyChangeListener", PropertyChangeListener.class);
            addPCLMethod.invoke(actionsObject, this);
        } catch (Exception e) {
            enablingPropertyError("PropertyChangeListener could not be installed", e);
        }
    }

    private void installMethods(Object actionsObject) {
        PropertyDescriptor enablingProperty = null;
        try {
            enablingProperty = PropertyUtils.getPropertyDescriptor(actionsObject, getEnabledProperty());

        } catch (Exception e) {
            enablingPropertyError("Enabling property could not be accessed", e);
        }
        if (enablingProperty != null) {
            if (isBoolean(enablingProperty.getPropertyType())) {
                Method enablingPropertyReadMethod = PropertyUtils.getReadMethod(enablingProperty);
                if (enablingPropertyReadMethod == null) {
                    enablingPropertyError("Enabling property must be readable");
                }
                setEnabledPropertyReadMethod(enablingPropertyReadMethod);
                setEnabledPropertyWriteMethod(PropertyUtils.getWriteMethod(enablingProperty));
            } else {
                enablingPropertyError("Enabling property must be of Type Boolean");
            }
        }
    }

    private void enablingPropertyError(String message) {
        enablingPropertyError(message, null);
    }

    private void enablingPropertyError(String message, Throwable cause) {
        throw new IllegalArgumentException(message, cause);
    }

    private boolean isBoolean(Class<?> propertyType) {
        return propertyType == Boolean.class || propertyType == Boolean.TYPE;
    }

    /**
     * @see com.ulcjava.base.application.AbstractAction#setEnabled(boolean)
     */
    @Override
    public void setEnabled(boolean enabled) {
        Method enabledPropertyWriteMethod = getEnabledPropertyWriteMethod();
        if (enabledPropertyWriteMethod == null) {
            if (getEnabledPropertyReadMethod() == null) {
                super.setEnabled(enabled);
            } else {
                throw new Error("Enabled Property should not be set - because enabling Property is read only");
            }
        } else {
            Object actionsObject = getApplicationActionMap().getActionsObject();
            try {
                enabledPropertyWriteMethod.invoke(actionsObject, enabled);
            } catch (Exception e) {
                String msg = String.format("%s.%s(%s) failed", actionsObject.getClass(), enabledPropertyWriteMethod,
                        enabled);
                throw new Error("Error on setting enabled property : " + msg, e);
            }
        }
    }

    /**
     * @see com.ulcjava.base.application.AbstractAction#isEnabled()
     */
    @Override
    public boolean isEnabled() {
        Method enabledPropertyReadMethod = getEnabledPropertyReadMethod();
        if (enabledPropertyReadMethod == null) {
            return super.isEnabled();
        } else {
            Object actionsObject = getApplicationActionMap().getActionsObject();
            try {
                return (Boolean)enabledPropertyReadMethod.invoke(actionsObject);
            } catch (Exception e) {
                String msg = String.format("%s.%s() failed", actionsObject.getClass(), enabledPropertyReadMethod);
                throw new Error("Error on getting enabled property : " + msg, e);
            }
        }
    }

    private void initProperties() {
        String name = getActionName();
        ResourceMap resourceMap = getResourceMap();
        if (resourceMap != null) {
            String actionName = getActionName();
            String text = resourceMap.getString(actionName + ACTION_TEXT);
            if (text != null) {
                name = text;
            }
            String shortDesc = resourceMap.getString(actionName + ACTION_SHORT_DESCRIPTION);
            if (shortDesc != null) {
                putValue(IAction.SHORT_DESCRIPTION, shortDesc);
            }
            String longDesc = resourceMap.getString(actionName + ACTION_LONG_DESCRIPTION);
            if (longDesc != null) {
                putValue(IAction.LONG_DESCRIPTION, longDesc);
            }
            ULCIcon icon = resourceMap.getULCIcon(actionName + ACTION_ICON);
            if (icon != null) {
                putValue(IAction.SMALL_ICON, icon);
            }
            KeyStroke accelerator = resourceMap.getKeyStroke(actionName + ACTION_ACCELERATOR);
            if (accelerator != null) {
                putValue(IAction.ACCELERATOR_KEY, accelerator);
            }

            Integer mnemonic = resourceMap.getKeyCode(actionName + ACTION_MNEMONIC);
            if (mnemonic != null) {
                putValue(IAction.MNEMONIC_KEY, mnemonic);
            }
            String command = resourceMap.getString(actionName + ACTION_COMMAND);
            if (command != null) {
                putValue(IAction.ACTION_COMMAND_KEY, command);
            }
        }
        MnemonicTextUtils.setMnemonicText(this, name);

    }

    /**
     * Calls the actionMethod on the actionTarget.
     * <p>
     * The parameters for the actionMethod are created according the following table. <table border=1>
     * <tr>
     * <th>Parameter Type</th>
     * <th>Parameter Value</th>
     * </tr>
     * <tr>
     * <td><code>ActionEvent</code>
     * <td>the <code>actionEvent</code> parameter of this method</td>
     * </td>
     * </tr>
     * <tr>
     * <td><code>IAction</code> or <code>AbstractAction</code></td>
     * <td>this <code>ApplicationAction</code> object</td>
     * </tr>
     * <tr>
     * <td><code>Application</code></td>
     * <td>the <code>Application</code> that is running</td>
     * </tr>
     * <tr>
     * <td><code>ResourceMap</code></td>
     * <td>the <code>ResourceMap</code> used to initialize this <code>ApplicationAction</code></td>
     * </tr>
     * <tr>
     * <td><code>ApplicationContext</code></td>
     * <td>the <code>ApplicationContext</code> of the <code>Application</code> that is running </td>
     * </tr>
     * </table>
     * 
     * @param event the action event
     */
    public void actionPerformed(ActionEvent event) {
        Method actionMethod = getActionMethod();
        Class<?>[] parameterTypes = actionMethod.getParameterTypes();
        Object[] parameters = new Object[parameterTypes.length];
        for (int i = 0; i < parameterTypes.length; i++) {
            parameters[i] = createActionParameter(parameterTypes[i], event);

        }

        try {
            actionMethod.invoke(getActionTarget(), parameters);
        } catch (Exception e) {
            throwActionError(e, "action invocation failed");
        }
    }

    /**
     * Logs the throwable with the given message and throws an {@link Error}.
     * 
     * @param throwable root cause
     * @param message message to log
     * @throws Error with the given Throwable as cause.
     */
    protected void throwActionError(Throwable throwable, String message) throws Error {
        LOG.log(Level.SEVERE, message, throwable);
        throw new Error(throwable);
    }

    /**
     * Creates the parameter of the given type to be used when invoking the actionMethod.
     * 
     * @param type the type of the parameter.
     * @param event that has triggered actionPerformed.
     * @return the parameter of the requested type.
     */
    protected Object createActionParameter(Class<?> type, ActionEvent event) {
        if (type == ActionEvent.class) {
            return event;
        }
        if (type == IAction.class || type == AbstractAction.class) {
            return this;
        }
        if (type == ResourceMap.class) {
            return getResourceMap();
        }
        if (type == ApplicationContext.class) {
            return getApplicationActionMap().getContext();
        }
        if (type == Application.class) {
            return getApplicationActionMap().getContext().getApplication();
        }

        throwActionError(new IllegalArgumentException("Unsupported parameter type " + type.toString()),
                "Could not create parameter");

        return null;
    }

    /**
     * @return the application map the action belongs to.
     */
    protected ApplicationActionMap getApplicationActionMap() {
        return fApplicationActionMap;
    }

    /**
     * @return the resource map used to initialize the action's properties.
     */
    protected ResourceMap getResourceMap() {
        return fResourceMap;
    }

    /**
     * @return the name of the action.
     */
    protected String getActionName() {
        return fActionName;
    }

    /**
     * @return the method that gets executed on actionPerformed.
     */
    protected Method getActionMethod() {
        return fActionMethod;
    }

    /**
     * @return the object on which the actionMethod is executed on actionPerformed.
     */
    protected Object getActionTarget() {
        return fActionTarget;
    }

    /**
     * @return the property the action enabled state is bound to.
     */
    protected String getEnabledProperty() {
        return fEnabledProperty;
    }

    private Method getEnabledPropertyReadMethod() {
        return fEnabledPropertyReadMethod;
    }

    private void setEnabledPropertyReadMethod(Method enablingPropertyReadMethod) {
        fEnabledPropertyReadMethod = enablingPropertyReadMethod;
    }

    private Method getEnabledPropertyWriteMethod() {
        return fEnabledPropertyWriteMethod;
    }

    private void setEnabledPropertyWriteMethod(Method enablingPropertyWriteMethod) {
        fEnabledPropertyWriteMethod = enablingPropertyWriteMethod;
    }

    /**
     * Fires a property change event if the enabledProperty has changed.
     * 
     * @param evt the property change event
     */
    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getPropertyName().equals(getEnabledProperty())) {

            firePropertyChange("enabled", evt.getOldValue(), evt.getNewValue());
        }
    }


    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        writeMethod(out, fActionMethod);
        writeMethod(out, fEnabledPropertyReadMethod);
        writeMethod(out, fEnabledPropertyWriteMethod);
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        fActionMethod = readMethod(in);
        fEnabledPropertyReadMethod = readMethod(in);
        fEnabledPropertyWriteMethod = readMethod(in);
    }

    private void writeMethod(ObjectOutputStream out, Method aMethod) throws IOException {
        if (aMethod == null) {
            out.writeObject(null);
        } else {
            Class<?> declaringClass = aMethod.getDeclaringClass();
            out.writeObject(declaringClass);
            String name2 = aMethod.getName();
            out.writeObject(name2);
            Class<?>[] parameterTypes = aMethod.getParameterTypes();
            out.writeObject(parameterTypes);
        }
    }

    private Method readMethod(ObjectInputStream in) throws IOException, ClassNotFoundException {
        Class<?> declaringClass = (Class<?>)in.readObject();
        if (declaringClass == null) {
            return null;
        }
        String methodName = (String)in.readObject();
        Class<?>[] parameterTypes = (Class<?>[])in.readObject();
        try {
            return declaringClass.getMethod(methodName, parameterTypes);
        } catch (NoSuchMethodException e) {
            throw new IOException(e.getMessage());
        }
    }
}
