/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import com.ulcjava.base.application.IAction;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * The <code>ApplicationActionMap</code> creates <code>ApplicationAction</code>s from the &#064;Action annotated
 * methods of an object. The actions can be retrieved with the <code>get(String actionName)</code> method.
 * <p>
 * A parent <code>ApplicationActionMap</code> can be set which can be used to retrieve an action if it is not found in
 * the own map.
 */
public class ApplicationActionMap implements Serializable {

    private final ApplicationContext fContext;
    private final Class<?> fActionsClass;
    private final Object fActionsObject;
    private final ResourceMap fResourceMap;
    private final Map<String, ApplicationAction> fMap;
    private ApplicationActionMap fParentMap;

    /**
     * Creates the {@code ApplicationActionMap} for the given object with the annotated methods from the given class.
     * The <code>actionsObject</code> must be an instance of the <code>actionsClass</code>.
     * 
     * @param context the {@link ApplicationContext} of the running {@link Application}. Must not be <code>null</code>.
     * @param actionsClass the class this map retrieves the action methods from. Must not be <code>null</code>.
     * @param actionsObject the object which methods are called by this map's actions. Must not be <code>null</code>.
     * @param resourceMap that is used to initialize the action properties.
     */
    public ApplicationActionMap(ApplicationContext context, Class<?> actionsClass, Object actionsObject, ResourceMap resourceMap) {
        if (context == null) {
            throw new IllegalArgumentException("context must not be null");
        }
        if (actionsClass == null) {
            throw new IllegalArgumentException("actionsClass must not be null");
        }
        if (actionsObject == null) {
            throw new IllegalArgumentException("actionsObject must not be null");
        }
        if (!actionsClass.isInstance(actionsObject)) {
            throw new IllegalArgumentException("actionsObject must be of type " + actionsClass.toString());
        }
        fContext = context;
        fActionsClass = actionsClass;
        fActionsObject = actionsObject;
        fResourceMap = resourceMap;
        fMap = new HashMap<String, ApplicationAction>();
        createActionsFromAnnotation(actionsClass);
    }

    private void createActionsFromAnnotation(Class<?> actionsClass) {
        Method[] declaredMethods = actionsClass.getDeclaredMethods();
        for (Method method : declaredMethods) {
            Action action = method.getAnnotation(Action.class);
            if (action != null) {

                String actionName = isDefined(action.name()) ? action.name() : method.getName();
                String enabledProperty = isDefined(action.enabledProperty()) ? action.enabledProperty() : null;

                ApplicationAction applicationAction = new ApplicationAction(this, getResourceMap(), actionName, method,
                        enabledProperty);
                fMap.put(actionName, applicationAction);
            }
        }

    }

    private boolean isDefined(String name) {
        return name != null && name.length() > 0;
    }

    /**
     * @return the {@link ApplicationContext} of the running {@link Application}.
     */
    protected ApplicationContext getContext() {
        return fContext;
    }

    /**
     * @return the class this map retrieves the action methods from.
     */
    protected Class<?> getActionsClass() {
        return fActionsClass;
    }

    /**
     * @return the object which methods are called by this map's actions.
     */
    protected Object getActionsObject() {
        return fActionsObject;
    }

    /**
     * @return the {@link ResourceMap} used to initialize the action properties.
     */
    protected ResourceMap getResourceMap() {
        return fResourceMap;
    }

    /**
     * retrieves an Action created for the method with the given name.
     * 
     * @param key the ActionName of the <code>&#064;Action</code> to be retrieved, by default the method name.
     * @return the {@link ApplicationAction} that performs the <code>@Action</code> with the given name. <code>null</code> if there is no action for this name found.
     * @see Action
     */
    public IAction get(String key) {
        if (key == null) {
            throw new IllegalArgumentException("key must not be null");
        }
        IAction action = fMap.get(key);
        return action == null && getParentMap() != null ? getParentMap().get(key) : action;
    }

    /**
     * @return all action names of this and its parent map.
     */
    public Set<String> allKeys() {
        Set<String> allKeys = new HashSet<String>(fMap.keySet());
        if (getParentMap() != null) {
            allKeys.addAll(getParentMap().allKeys());
        }
        return allKeys;
    }

    /**
     * @return the parent map, that is used to retrieve an action if it is not found in the own map, or
     *         <code>null</code> if this <code>ApplicationActionMap</code> has no parent.
     */
    protected ApplicationActionMap getParentMap() {
        return fParentMap;
    }

    /**
     * Sets the parent map, that is used to retrieve an action, if it is not found in the own map.
     * 
     * @param parentMap is the parent of this <code>ApplicationActionMap</code>. Can be <code>null</code>.
     */
    protected void setParentMap(ApplicationActionMap parentMap) {
        fParentMap = parentMap;
    }
}
