/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import java.io.Serializable;

/**
 * Helper class used by {@link ApplicationAction}s and {@link ResourceMap}s to parse a String if it contains an &#064;
 * signifying that the following char is to be used as Mnemonic.
 */
public class MnemonicText implements Serializable {

    private String fText;
    private Character fMnemonicChar;
    private int fMnemonicIndex;

    /**
     * Parses the given text looking for the first & that is followed by another character. If it is found then the & is
     * removed from the String, the mnemonicIndex is set to the position where the & character was found, and the
     * mnemonicChar is set to the character at the mnemonicIndex converted to upper case. Two consecutive & are replaced
     * by a single &. This allows the text to contain a &.
     * <p>
     * "&&&Save" results in the text "&Save" with the mnemonicIndex = 1 and the mnemonicChar = 'S'
     * 
     * @param mnemonicText the text to parse.
     */
    public MnemonicText(String mnemonicText) {
        if (mnemonicText == null) {
            mnemonicText = "";
        }

        fText = mnemonicText;
        fMnemonicIndex = -1;
        int escapes = 0;
        for (int i = 0; i < mnemonicText.length() - 1; i++) {
            char currentChar = mnemonicText.charAt(i);
            if (currentChar == '&') {
                if (mnemonicText.charAt(i + 1) == '&') {
                    i++;
                    escapes++;
                } else {
                    fText = mnemonicText.substring(0, i) + mnemonicText.substring(i + 1);
                    fMnemonicChar = mnemonicText.substring(i + 1, i + 2).toUpperCase().charAt(0);
                    fMnemonicIndex = i - escapes;
                    break;
                }
            }
        }
        fText = fText.replace("&&", "&");
    }

    /**
     * @return the text with the first single & removed.
     */
    public String getText() {
        return fText;
    }

    /**
     * @return the mnemonic character.
     */
    public Character getMnemonicChar() {
        return fMnemonicChar;
    }

    /**
     * @return the integer value of the mnemonic character.
     */
    public int getMnemonicCharCode() {
        return fMnemonicChar;
    }

    /**
     * @return the index of the mnemonic character.
     */
    public int getMnemonicIndex() {
        return fMnemonicIndex;
    }

    /**
     * @return true if the text has a mnemonic character, false if the string did not contain a & marking a mnemonic
     *         character.
     */
    public boolean hasMnemonic() {
        return fMnemonicChar != null;
    }
}