/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import com.ulcjava.base.application.IAction;
import com.ulcjava.base.application.ULCAbstractButton;
import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCLabel;

/**
 * Utility class that provides methods to set a text, that might have a & marked mnemonic character, on different
 * {@link ULCComponent}s that support mnemonic characters.
 * 
 * @see MnemonicText
 */
public class MnemonicTextUtils {

    private MnemonicTextUtils() {
    }

    /**
     * The character after the first unescaped '&' is considered the Mnemonic char. The '&' is removed from the string
     * that is then set as text on the button. An '&' at the end of the mnemonicText is ignored. Use "&&" to produce an
     * escaped '&' ( <code>&&Save</code> results in <code>&Save</code> without mnemonic)
     * 
     * @param target the {@code ULCAbstractButton} on which to set the mnemonic text
     * @param mnemonicText {@code String} text with mnemonic char in it
     */
    public static void setMnemonicText(ULCAbstractButton target, String mnemonicText) {
        MnemonicText mText = new MnemonicText(mnemonicText);
        target.setText(mText.getText());
        if (mText.hasMnemonic()) {
            target.setMnemonic(mText.getMnemonicChar());
            target.setDisplayedMnemonicIndex(mText.getMnemonicIndex());
        }
    }

    /**
     * The character after the first unescaped '&' is considered the Mnemonic char. The & is removed from the string
     * that is then set as text on the button. An '&' at the end of the mnemonicText is ignored. Use "&&" to produce an
     * escaped '&' ( <code>&&Save</code> results in <code>&Save</code> without mnemonic)
     * 
     * @param target the {@code ULCLabel} on which to set the mnemonic text
     * @param mnemonicText {@code String} text with mnemonic char in it
     */
    public static void setMnemonicText(ULCLabel target, String mnemonicText) {
        MnemonicText mText = new MnemonicText(mnemonicText);
        target.setText(mText.getText());
        if (mText.hasMnemonic()) {
            target.setDisplayedMnemonic(mText.getMnemonicChar());
            target.setDisplayedMnemonicIndex(mText.getMnemonicIndex());
        }
    }

    /**
     * The character after the first unescaped '&' is considered the Mnemonic char. The & is removed from the string
     * that is then set as text on the button. An '&' at the end of the mnemonicText is ignored. Use && to produce an
     * escaped & ( <code>&&Save</code> results in <code>&Save</code> without mnemonic)
     * 
     * @param action the {@code IAction} on which to set the mnemonic property
     * @param mnemonicText {@code String} text with mnemonic char in it
     */
    public static void setMnemonicText(IAction action, String mnemonicText) {
        MnemonicText mText = new MnemonicText(mnemonicText);
        action.putValue(IAction.NAME, mText.getText());
        if (mText.hasMnemonic()) {
            action.putValue(IAction.MNEMONIC_KEY, mText.getMnemonicCharCode());
        }
    }
}
