/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import org.apache.commons.beanutils.PropertyUtils;

import com.ulcjava.applicationframework.application.AbstractResourceConverter.ResourceConverterException;
import com.ulcjava.base.application.ULCAbstractButton;
import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCContainer;
import com.ulcjava.base.application.ULCLabel;
import com.ulcjava.base.application.ULCMenu;
import com.ulcjava.base.application.util.Color;
import com.ulcjava.base.application.util.Font;
import com.ulcjava.base.application.util.KeyStroke;
import com.ulcjava.base.application.util.ULCIcon;
import com.ulcjava.base.shared.internal.ClassUtilities;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * The ResourceMap gives access to the resources from several resource bundles. First it can contain a list of resource bundles that all
 * have come from the same directory. Second it may have a parent ResourceMap. If the resource for a given key is requested, it is searched
 * first in the own bundles and if not found there in the parent ResourceMap.
 * <p>
 * There is a extended syntax for the resources, allowing to compose resources more flexible:
 * <p>
 * An expression of the form ${<i>key</i>} anywhere in the resource is replaced by the resource with the key that is between the brackets.
 * 
 * <pre>
 * label1=Hello
 * label2=${label1} World
 * </pre>
 * getString("label2") will return Hello World
 * <p>
 * A resource that starts with <b>#{</b> and ends with <b>}</b> is replaced by the java constant that is between the brackets. Classes must
 * be fully qualified except classes from the com.ulcjava.base.application package.
 * 
 * <pre>
 * myLabel.verticalAlignment= #{ULCLabel.BOTTOM}
 * </pre>
 * getInteger("myLabel.verticalAlignment") will return 3, because this is the value of the com.ulcjava.base.application.ULCLabel member
 * BOTTOM
 * <p>
 * The ResourceMap provides a set of methods to retrieve a resource and to convert it into several types (e.g. getInteger(key)).
 * <p>
 * A ResourceMap can be used to inject the resources into a ULCComponent tree. The properties of all ULCComponents in the tree are set to
 * values from the ResourceMap by the following rules:
 * <ul>
 * <li>Properties of components that have their name set are set to the first of the resources that is found using the following keys :
 * <ol>
 * <li><i>componentName.propertyName</i> if the component has a name set.</li>
 * <li><i>styleName.propertyName</i> if a style is set.</li>
 * <li><i>ClassOrSuperClassOfTheComponent.propertyName</i>.</li>
 * </ol>
 * </li>
 * </ul>
 * Where
 * <ul>
 * <li><i>componentName</i> the String that is set on the component with <code>setName()</code></li>
 * <li><i>propertyName</i> is the name of the property</li>
 * <li><i>styleName</i> is the resource as String with the key <i>componentName</i>.style or is set as clientProperty with the key
 * ResourceMap.STYLE_CLIENT_PROPERTY, where the former overrides the latter.</li>
 * <li><i>ClassOrSuperClassOfTheComponent</i> is the first simple class name in the class hierarchy from the component's class up to
 * ULCComponent, for which a resource with the key <i>simpleClassName.propertyName</i> is defined .</li>
 * </ul>
 * <br>
 * Here is an example:
 * <p>
 * The property file MyApplication.properties contains
 * 
 * <pre>
 * ULCComponent.font=Arial-PLAIN-12
 * ULCComponent.background=red
 * ULCComponent.foreground=yellow
 * ULCComponent.alignmentX=0.25
 * 
 * ULCLabel.font=Arial-BOLD-12
 * ULCLabel.background=green
 * ULCLabel.foreground=blue
 * 
 * Heading.font=Arial-BOLD-20
 * Heading.foreground=black
 * 
 * myLabel.style=Heading
 * myLabel.foreground=cyan
 * </pre>
 * <p>
 * in the application :
 * 
 * <pre>
 * public class MyApplication extends Application {
 * 
 *     protected void startup() {
 *          ....
 *          ULCLabel label = new ULCLabel();
 *          label.setName(&quot;myLabel&quot;)
 *          Application.getInstance().getContext().getResourceMap().injectComponent(label);
 *          ....
 *     }
 * }
 * </pre>
 * this results in the <code>label</code> gets the following properties set:
 * 
 * <pre>
 * foreground           cyan                from myLabel.foreground
 * background           green               from ULCLabel.background
 * font                 Arial bold 20       from Heading.font
 * alignmentX           0.25                from ULCComponent#alignmentX
 * </pre>
 */
public class ResourceMap implements Serializable {
    
    private static class ResourceMapKey implements Serializable {
        private final Locale fKeyLocale;
        private final String[] fBundleNames;
        
        public ResourceMapKey(List<String> bundleNames, Locale locale) {
            if (locale == null) {
                throw new IllegalArgumentException("locale must not be null");
            }
            if (bundleNames == null) {
                throw new IllegalArgumentException("bundleNames must not be null");
            }
            fKeyLocale = locale;
            fBundleNames = bundleNames.toArray(new String[bundleNames.size()]);
        }
        
        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + Arrays.hashCode(fBundleNames);
            result = prime * result + fKeyLocale.hashCode();
            return result;
        }
        
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final ResourceMapKey other = (ResourceMapKey)obj;
            if (!Arrays.equals(fBundleNames, other.fBundleNames)) {
                return false;
            }
            if (!fKeyLocale.equals(other.fKeyLocale)) {
                return false;
            }
            return true;
        }
    }
    
    private static class ResourceMapCacheEntry implements Serializable {
        private final Map<String, String> fMap;
        private final Set<String> fResourceKeySet;
        
        public ResourceMapCacheEntry(Map<String, String> map, Set<String> resourceKeySet) {
            fMap = map;
            fResourceKeySet = resourceKeySet;
        }
        
        public Map<String, String> getMap() {
            return fMap;
        }
        
        public Set<String> getResourceKeySet() {
            return fResourceKeySet;
        }
        
    }
    
    private static Map<ResourceMapKey, ResourceMapCacheEntry> sResourceMapCache = new HashMap<ResourceMapKey, ResourceMapCacheEntry>(25);
    
    /**
     * Clears the resource map cache.
     */
    public static void resetCache() {
        sResourceMapCache.clear();
    }
    
    
    public static final String EXPRESSION_NULL = "null";
    public static final String CONSTANT_BEGIN = "#{";
    public static final String EXPRESSION_BEGIN = "${";
    public static final String EXPRESSION_END = "}";
    public static final String STYLE_CLIENT_PROPERTY = "ulc.style";
    
    private static final Logger LOG = Logger.getLogger(ResourceMap.class.getName());
    
    private final ResourceMap fParent;
    private final List<String> fBundleNames;
    private Locale fLocale;
    private String fResourcesDir;
    private ResourceMapCacheEntry fCacheEntry;
    private final ILocaleProvider fLocaleProvider;
    
    /**
     * Creates a ResourceMap with the given parent. The resources are loaded from the given bundles that all must be from the same package.
     * The list defines the order in which resources are found. This means that resources in the first bundle are always taken, the ones in
     * the last bundle only if none of the other bundles contains a resource with the same key.
     * <p>
     * If the resource for a key is not found in the own bundles and a parent is set, the resource is looked up in the parent.
     * 
     * @param parent map of this resource map. May be <code>null</code>.
     * @param bundleNames list of resource bundle names from the same package. The first bundle has the highest, the last the lowest
     *            priority.
     */
    public ResourceMap(ResourceMap parent, List<String> bundleNames) {
        this(parent, bundleNames, ClientContextLocaleProvider.getInstance());
    }
    
    public ResourceMap(ResourceMap parent, List<String> bundleNames, ILocaleProvider localeProvider) {
        if (bundleNames == null) {
            throw new IllegalArgumentException("bundleNames must not be null");
        }
        if (localeProvider == null) {
            throw new IllegalArgumentException("localeProvider must not be null");
        }
        for (String bundleName : bundleNames) {
            if (bundleName == null || bundleName.length() == 0) {
                throw new IllegalArgumentException("bundleName must not be null nor empty");
            }
            String baseDir = getBaseDir(bundleName);
            if (fResourcesDir == null) {
                fResourcesDir = baseDir;
            } else if (!baseDir.equals(fResourcesDir)) {
                throw new IllegalArgumentException("all bundle files  must be from the the same directory (" + bundleName + " not in "
                        + fResourcesDir + ")");
            }
        }
        fLocaleProvider = localeProvider;
        fResourcesDir = fResourcesDir.replace('.', '/') + "/";
        fParent = parent;
        fBundleNames = Collections.unmodifiableList(new ArrayList<String>(bundleNames));
        fLocale = localeProvider.getLocale();
    }
    
    /**
     * Creates a ResourceMap with the given parent and the given bundle names. Converts the array into a list preserving the order.
     * 
     * @see #ResourceMap(ResourceMap, List)
     */
    public ResourceMap(ResourceMap parent, String... bundleNames) {
        this(parent, Arrays.asList(bundleNames));
    }
    
    public ResourceMap(ResourceMap parent, ILocaleProvider localeProvider, String... bundleNames) {
        this(parent, Arrays.asList(bundleNames), localeProvider);
    }
    
    private String getBaseDir(String bundleName) {
        return bundleName.substring(0, bundleName.lastIndexOf('.'));
    }
    
    /**
     * @return the parent map of this resource map. May be <code>null</code>.
     */
    public ResourceMap getParent() {
        return fParent;
    }
    
    /**
     * @return the unmodifiable list of bundle names that are used to build the map.
     */
    public List<String> getBundleNames() {
        return fBundleNames;
    }
    
    /**
     * @return the {@link ClassLoader} used to load the resources.
     */
    public ClassLoader getClassLoader() {
        return ClassUtilities.getDefaultClassLoader();
    }
    
    /**
     * @return the directory where this map's resources are loaded from.
     */
    public String getResourcesDir() {
        return fResourcesDir;
    }
    
    /**
     * @param key to lookup.
     * @return true if this map or any of this map's parent map contains a resource for this key.
     */
    public boolean containsKey(String key) {
        return getMap().containsKey(key) || getParent().containsKey(key);
    }
    
    /**
     * @return this ResourceMap map that stores the resources by key.
     */
    protected Map<String, String> getMap() {
        return getCacheEntry().getMap();
    }
    
    // The resource maps are cached and shared among all sessions
    private ResourceMapCacheEntry getCacheEntry() {
        if (fCacheEntry == null || !fLocale.equals(fLocaleProvider.getLocale())) {
            fCacheEntry = loadResources();
        }
        return fCacheEntry;
    }
    
    private synchronized ResourceMapCacheEntry loadResources() {
        fLocale = fLocaleProvider.getLocale();
        ResourceMapKey mapKey = new ResourceMapKey(getBundleNames(), fLocale);
        ResourceMapCacheEntry cacheEntry = sResourceMapCache.get(mapKey);
        if (cacheEntry == null) {
            HashMap<String, String> resourceMap = new HashMap<String, String>();
            List<String> bundleNames = getBundleNames();
            
            for (int i = bundleNames.size() - 1; i >= 0; i--) {
                String bundleName = bundleNames.get(i);
                ResourceBundle bundle = null;
                try {
                    bundle = ResourceBundle.getBundle(bundleName, fLocale, getClassLoader());
                } catch (MissingResourceException e) {
                    // ignored
                }
                if (bundle != null) {
                    Enumeration<String> keys = bundle.getKeys();
                    while (keys.hasMoreElements()) {
                        String key = keys.nextElement();
                        resourceMap.put(key, bundle.getString(key));
                    }
                }
                
            }
            HashSet<String> resourceKeySet = new HashSet<String>(resourceMap.keySet());
            if (getParent() != null) {
                resourceKeySet.addAll(getParent().getResourceKeySet());
            }
            cacheEntry = new ResourceMapCacheEntry(resourceMap, resourceKeySet);
            sResourceMapCache.put(mapKey, cacheEntry);
        }
        return cacheEntry;
    }
    
    /**
     * @return set of all keys of the resources in this ResourceMap or in any parent ResourceMap.
     */
    protected Set<String> getResourceKeySet() {
        return getCacheEntry().getResourceKeySet();
    }
    
    /**
     * Retrieves the resource for the given key from the own map or from the parent if it is not in the own map.
     * 
     * @param key of the resource to lookup.
     * @return the resource found or <code>null</code> if there is no resource for the key.
     */
    protected String getResource(String key) {
        String value = getMap().get(key);
        return value == null && getParent() != null ? getParent().getResource(key) : value;
    }
    
    /**
     * Lookup the resource for the given key. If found, any expressions that it contains are resolved, ${key} expressions before
     * #{constantName} expression. If the requested type is not equal to String, a {@link IResourceConverter} is requested from {@link
     * AbstractResourceConverter#forType(<i>type</i>)} to convert the String into the requested type.
     * 
     * @param key of the resource to lookup.
     * @param type requested type of the returned resource.
     * @return the resource converted to the requested type.
     * @throws IllegalArgumentException if the key or type is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     * @see ResourceMap
     */
    public Object getObject(String key, Class<?> type) {
        if (type == null) {
            throw new IllegalArgumentException("type must not be null");
        }
        if (key == null) {
            throw new IllegalArgumentException("key must not be null");
        }
        ;
        String resourceString = getResource(key);
        resourceString = evaluateExpression(resourceString, type, key);
        
        Object resource = resourceString;
        if (resourceString != null) {
            if (resourceString.startsWith(CONSTANT_BEGIN)) {
                resource = lookupConstant(key, type, resourceString);
                if (!type.isInstance(resource)) {
                    if (String.class.isAssignableFrom(type)) {
                        resource = resource.toString();
                    }
                }
            } else if (!type.isInstance(resourceString)) {
                IResourceConverter converter = AbstractResourceConverter.forType(type);
                try {
                    if (converter != null) {
                        resource = converter.parseString(resourceString, this);
                    } else {
                        throw new LookupException("no converter defined", key, type);
                    }
                } catch (ResourceConverterException e) {
                    throw new LookupException(e.getMessage(), key, type, e);
                }
                
            }
        }
        return resource;
    }
    
    
    private Object lookupConstant(String key, Class<?> type, String resourceString) {
        Object resource;
        if (resourceString.endsWith(EXPRESSION_END)) {
            String constantName = resourceString.substring(2, resourceString.length() - 1);
            
            int dotAfterClassname = constantName.lastIndexOf('.');
            if (dotAfterClassname <= 0) {
                throw new LookupException("No Classname defined in constant expression" + resourceString + "<", key, type);
            }
            String fullClassName = constantName.substring(0, dotAfterClassname);
            int dotBeforeClassSimpleName = fullClassName.lastIndexOf('.');
            String classSimpleName = fullClassName.substring(dotBeforeClassSimpleName < 0 ? 0 : dotBeforeClassSimpleName + 1);
            if (dotBeforeClassSimpleName < 0) {
                fullClassName = "com.ulcjava.base.application." + classSimpleName;
            }
            String fieldName = constantName.substring(dotAfterClassname + 1);
            Class<?> clazz;
            try {
                clazz = Class.forName(fullClassName, true, getClassLoader());
            } catch (ClassNotFoundException e) {
                throw new LookupException("Class not found in constant expression >" + resourceString + "<", key, type, e);
            } catch (Exception e) {
                throw new LookupException("Error on loading class in constant expression >" + resourceString + "<", key, type, e);
            }
            Field constant;
            try {
                constant = clazz.getField(fieldName);
                resource = constant.get(null);
            } catch (NoSuchFieldException e) {
                throw new LookupException("Field not defined in constant expression >" + resourceString + "<", key, type, e);
            } catch (Exception e) {
                throw new LookupException("Field could not be read in constant expression >" + resourceString + "<", key, type, e);
            }
        } else {
            throw new LookupException("Missing closing } in constant expression >" + resourceString + "<", key, type);
        }
        return resource;
    }
    
    private String evaluateExpression(String resourceString, Class type, String key) {
        if (resourceString == null) {
            return resourceString;
        }
        String result = resourceString;
        int expStart = result.indexOf(EXPRESSION_BEGIN);
        while (expStart > -1) {
            if (isNotEscaped(result, expStart)) {
                int expEnd = result.indexOf(EXPRESSION_END, expStart + 2);
                if (expEnd < 0) {
                    throw new LookupException("Missing closing } in expression >" + resourceString + "<", key, type);
                }
                String expression = result.substring(expStart + 2, expEnd);
                if (expression.length() == 0) {
                    throw new LookupException("Empty expression in >" + resourceString + "<", key, type);
                }
                if (expression.equals(EXPRESSION_NULL)) {
                    return null;
                }
                String expressionResult = getString(expression);
                if (expressionResult == null) {
                    throw new LookupException("Expression results in NULL >" + resourceString + "<", key, type);
                }
                result = result.substring(0, escapeCharOnPosition(result, expStart - 1) ? expStart - 1 : expStart) + expressionResult
                        + result.substring(expEnd + 1);
                expStart = result.indexOf(EXPRESSION_BEGIN, expStart);
            } else {
                result = result.substring(0, expStart - 1) + result.substring(expStart);
                expStart = result.indexOf(EXPRESSION_BEGIN, expStart + 1);
            }
        }
        
        return result;
    }
    
    
    private boolean isNotEscaped(String result, int expStart) {
        return !escapeCharOnPosition(result, expStart - 1) || escapeCharOnPosition(result, expStart - 2);
    }
    
    
    private boolean escapeCharOnPosition(String result, int backslashPos) {
        return backslashPos >= 0 && result.charAt(backslashPos) == '\\';
    }
    
    /**
     * Looks up the String resource for the given key. If there are arguments given, the result is formatted with MessageFormat.format using
     * the arguments.
     * 
     * @param key of the resource.
     * @param args used to be inserted into the pattern string.
     * @return the found string resource, if arguments are specified they are inserted into the string with MessageFormat.format. If no
     *         resource is for the key is found, <code>null</code> is returned.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup.
     */
    public String getString(String key, Object... args) {
        String result = (String)getObject(key, String.class);
        if (args != null && args.length > 0 && result != null) {
            MessageFormat messageFormat = new MessageFormat(result, fLocale);
            result = messageFormat.format(args);
        }
        return result;
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Boolean.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Boolean getBoolean(String key) {
        return (Boolean)getObject(key, Boolean.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Integer.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Integer getInteger(String key) {
        return (Integer)getObject(key, Integer.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Long.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Long getLong(String key) {
        return (Long)getObject(key, Long.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Short.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Short getShort(String key) {
        return (Short)getObject(key, Short.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Byte.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Byte getByte(String key) {
        return (Byte)getObject(key, Byte.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Float.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Float getFloat(String key) {
        return (Float)getObject(key, Float.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Double.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Double getDouble(String key) {
        return (Double)getObject(key, Double.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a ULCIcon.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final ULCIcon getULCIcon(String key) {
        return (ULCIcon)getObject(key, ULCIcon.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Font.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Font getFont(String key) {
        return (Font)getObject(key, Font.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Color.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final Color getColor(String key) {
        return (Color)getObject(key, Color.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a KeyStroke.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public final KeyStroke getKeyStroke(String key) {
        return (KeyStroke)getObject(key, KeyStroke.class);
    }
    
    /**
     * Looks up the String resource for the given key and converts it to a Integer representing a key code.
     * 
     * @param key of the resource.
     * @return the resource for the key.
     * @throws IllegalArgumentException if the key is <code>null</code>.
     * @throws LookupException if there is an error during the lookup or the type conversion.
     */
    public Integer getKeyCode(String key) {
        KeyStroke keyStroke = getKeyStroke(key);
        return keyStroke == null ? null : keyStroke.getKeyCode();
    }
    
    /**
     * Sets the properties of <code>target</code> to the values of the resources in the map that follow the naming scheme:
     * <i>target.getName().propertyName</i> If there is a resource define but it cannot be set on the property a
     * {@link PropertyInjectionException} is thrown.
     * 
     * @param target Component that properties are to be set. Must not be <code>null</code>.
     * @throws IllegalArgumentException if <code>target</code> is <code>null</code>
     * @throws LookupException if a resource is found but could not be converted to the required type.
     * @throws PropertyInjectionException if a resource is found but could not be set on the target.
     */
    public void injectComponent(ULCComponent target) {
        if (target == null) {
            throw new IllegalArgumentException("target must not be null");
        }
        Map<String, String> propertyKeys = new HashMap<String, String>();
        
        fillResourceKeys(propertyKeys, target.getClass(), ULCComponent.class);
        String name = target.getName();
        String style = (name != null && name.trim().length() > 0) ? getString(name + "." + "style") : null;
        if (style == null) {
            final Object styleProperty = target.getClientProperty(STYLE_CLIENT_PROPERTY);
            style = (styleProperty != null ? styleProperty.toString() : null);
        }
        if (style != null) {
            fillResourceKeys(propertyKeys, style);
        }
        if (name != null && name.trim().length() > 0) {
            fillResourceKeys(propertyKeys, name);
        }
        Set<Entry<String, String>> entrySet = propertyKeys.entrySet();
        for (Entry<String, String> propertyKeyEntry : entrySet) {
            String key = propertyKeyEntry.getValue();
            String propertyName = propertyKeyEntry.getKey();
            Class<?> propertyType;
            try {
                propertyType = PropertyUtils.getPropertyType(target, propertyName);
            } catch (Exception e) {
                propertyType = null;
            }
            if (propertyType != null) {
                Object value = getObject(key, propertyType);
                if (propertyName.equals("text")) {
                    if (target instanceof ULCAbstractButton) {
                        MnemonicTextUtils.setMnemonicText((ULCAbstractButton)target, (String)value);
                    }
                    if (target instanceof ULCLabel) {
                        MnemonicTextUtils.setMnemonicText((ULCLabel)target, (String)value);
                    }
                } else {
                    try {
                        PropertyUtils.setProperty(target, propertyName, value);
                    } catch (Exception e) {
                        throw new PropertyInjectionException("Could not set property", target, key, value);
                    }
                }
            }
        }
    }
    
    private <T> void fillResourceKeys(Map<String, String> propertyKeys, Class<? extends T> startClass, Class<T> endClass) {
        List<String> classes = new ArrayList<String>();
        Class<?> clazz = startClass;
        while (clazz != endClass.getSuperclass()) {
            classes.add(clazz.getSimpleName());
            clazz = clazz.getSuperclass();
        }
        for (int i = classes.size() - 1; i >= 0; i--) {
            fillResourceKeys(propertyKeys, classes.get(i));
        }
    }
    
    private void fillResourceKeys(Map<String, String> propertyKeys, String prefix) {
        prefix = prefix + ".";
        List<String> resourceKeys = resourceKeysFor(prefix);
        for (String key : resourceKeys) {
            if (key.length() == prefix.length()) {
                LOG.log(Level.WARNING, "Missing property name for key " + key);
            }
            String propertyName = key.substring(prefix.length());
            propertyKeys.put(propertyName, key);
        }
    }
    
    /**
     * @param prefix must not be <code>null</code> nor empty.
     * @return all keys in the resource hierarchy that are starting with the <code>prefix</code>
     */
    public List<String> getResourceKeys(String prefix) {
        if (prefix == null || prefix.trim().length() == 0) {
            throw new IllegalArgumentException("prefix must not be null nor empty");
        }
        return resourceKeysFor(prefix);
    }
    
    private List<String> resourceKeysFor(String prefix) {
        ArrayList<String> result = new ArrayList<String>();
        Set<String> keys = getResourceKeySet();
        for (String key : keys) {
            if (key.startsWith(prefix)) {
                result.add(key);
            }
        }
        return result;
    }
    
    /**
     * recursively injects properties on the component tree
     * 
     * @param root the root component.
     * @throws IllegalArgumentException if <code>root</code> is <code>null</code>
     * @throws LookupException if a resource is found but could not be converted to the required type.
     * @throws PropertyInjectionException if a resource is found but could not be set on the target.
     */
    public void injectComponents(ULCComponent root) {
        
        injectComponent(root);
        ULCComponent[] components = new ULCComponent[0];
        if (root instanceof ULCMenu) {
            components = ((ULCMenu)root).getMenuComponents();
        }
        if (root instanceof ULCContainer) {
            components = ((ULCContainer)root).getComponents();
        }
        for (ULCComponent component : components) {
            injectComponents(component);
        }
        
    }
    
    /**
     * Exception thrown by the ResourceMap if there is an error during the resource lookup or the type conversion.
     */
    public static class LookupException extends RuntimeException {
        

        public LookupException(String msg, String key, Class<?> type) {
            this(msg, key, type, null);
        }
        
        public LookupException(String msg, String key, Class<?> type, Exception e) {
            super(msg + ": key " + key + " type " + type, e);
        }
        
    }
    /**
     * Exception thrown by the ResourceMap if there is an error during the property injection.
     */
    public static class PropertyInjectionException extends RuntimeException {
        

        public PropertyInjectionException(String msg, ULCComponent component, String key, Object value) {
            super(msg + " key:" + key + " value:" + value + " component:" + component);
        }
        

    }
    
}