/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;


import com.ulcjava.base.application.ULCBorderLayoutPane;
import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCContainer;
import com.ulcjava.base.application.ULCFrame;
import com.ulcjava.base.application.ULCMenuBar;
import com.ulcjava.base.application.ULCRootPane;
import com.ulcjava.environment.applet.application.ULCAppletPane;

import java.io.Serializable;

/**
 * Base class for classes that provide the root pane with a border layout. Subclasses create either a {@link ULCFrame} or a
 * {@link ULCAppletPane} as root pane. <code>setToolBar(ULCComponent)</code> sets the component in the NORTH area,
 * <code>setStatusBar(ULCComponent)</code> sets the component in the SOUTH area. The main component in the CENTER area is set with
 * <code>setComponent(ULCComponent)</code>.
 */
public abstract class View implements Serializable {
    
    private final Application fApplication;
    private ULCComponent fMainContent;
    private ULCComponent fStatusBar;
    private ULCMenuBar fMenuBar;
    private ULCComponent fToolBar;
    
    /**
     * Creates a View for the given application.
     * 
     * @param application that is running.
     */
    public View(Application application) {
        if (application == null) {
            throw new IllegalArgumentException("application must not be null");
        }
        fApplication = application;
    }
    
    /**
     * @return the running {@link Application}.
     */
    public final Application getApplication() {
        return fApplication;
    }
    
    /**
     * @return the {@link ApplicationContext} of the running {@link Application}.
     */
    public final ApplicationContext getContext() {
        return getApplication().getContext();
    }
    
    /**
     * @return the view's root pane for the application.
     */
    public abstract ULCRootPane getRootPane();
    
    /**
     * @return the component that is in the center area of the view.
     * @deprecated use getMainContent() instead
     */
    public ULCComponent getComponent() {
        return getMainContent();
    }

    /**
     * @return the component that is in the center area of the view.
     */
    public ULCComponent getMainContent() {
        return fMainContent;
    }
    
    /**
     * Sets the given component into the center area.
     * 
     * @param component that is the main content of the view.
     * @deprecated use setMainContent() instead
     */
    public void setComponent(ULCComponent component) {
        setMainContent(component);
    }

    /**
     * Sets the given component into the center area.
     * 
     * @param mainContent that is the main content of the view.
     */
    public void setMainContent(ULCComponent mainContent) {
        putIntoContentPane(mainContent, fMainContent, ULCBorderLayoutPane.CENTER);
        fMainContent = mainContent;
    }
    
    private void putIntoContentPane(ULCComponent component, ULCComponent oldComponent, String borderLayoutPaneConstraints) {
        ULCContainer contentPane = getRootPane().getContentPane();
        if (oldComponent != null) {
            contentPane.remove(oldComponent);
        }
        if (component != null) {
            contentPane.add(component, borderLayoutPaneConstraints);
        }
    }
    
    /**
     * @return the component that is in the bottom area of the view.
     */
    public ULCComponent getStatusBar() {
        return fStatusBar;
    }
    
    /**
     * Sets the given component into the SOUTH area.
     * 
     * @param statusBar is set into the bottom area of the view.
     */
    public void setStatusBar(ULCComponent statusBar) {
        putIntoContentPane(statusBar, fStatusBar, ULCBorderLayoutPane.SOUTH);
        fStatusBar = statusBar;
    }
    
    /**
     * @return the view's menu bar.
     */
    public ULCMenuBar getMenuBar() {
        if (fMenuBar == null) {
            setMenuBar(new ULCMenuBar());
        }
        return fMenuBar;
    }
    
    /**
     * Sets the view's menu bar.
     * 
     * @param menuBar of the view.
     */
    public void setMenuBar(ULCMenuBar menuBar) {
        getRootPane().setMenuBar(menuBar);
        fMenuBar = menuBar;
    }
    
    /**
     * @return the component that is in the top area of the view.
     */
    public ULCComponent getToolBar() {
        return fToolBar;
    }
    
    /**
     * Sets the given component into the NORTH area.
     * 
     * @param toolBar is set into the top area of the view.
     */
    public void setToolBar(ULCComponent toolBar) {
        putIntoContentPane(toolBar, fToolBar, ULCBorderLayoutPane.NORTH);
        fToolBar = toolBar;
    }
    
    /**
     * Uses this views {@link ResourceMap} to inject the properties for the root pane's component tree. Sets the root pane visible property
     * to <code>true</code>.
     */
    public void show() {
        getResourceMap().injectComponents(getRootPane());
        prepareShow();
        getRootPane().setVisible(true);
    }
    
    /**
     * @return the ResourceMap for this view. It is built with the view's class hierarchy and the application's default ResourceMap.
     */
    public ResourceMap getResourceMap() {
        return getContext().getResourceMap(getClass(), View.class);
    }
    
    /**
     * Subclasses can overwrite this to prepare the view to be displayed after the properties are injected by the view's {@code ResourceMap}.
     */
    protected void prepareShow() {
    }
    
    /**
     * Sets the view's visible property to false.
     */
    public void hide() {
        getRootPane().setVisible(false);
        
    }
    
}
