/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding;

import org.apache.commons.beanutils.MappedPropertyDescriptor;
import org.apache.commons.beanutils.PropertyUtils;

import java.beans.IndexedPropertyDescriptor;
import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.ArrayList;

/**
 * @ulc.internal
 */
public class BindingUtils {
    private BindingUtils() {
    }
    
    public static String getBeanName(Class<?> beanType) {
        String fullQualifiedName = beanType.getName();
        return fullQualifiedName.substring(fullQualifiedName.lastIndexOf(".") + 1);
    }
    
    public static String getBeanPackage(Class<?> beanType) {
        String fullQualifiedName = beanType.getName();
        return fullQualifiedName.substring(0, fullQualifiedName.lastIndexOf("."));
    }
    
    public static PropertyDescriptor[] getSimpleReadablePropertyDescriptors(Class<?> beanType) {
        PropertyDescriptor[] propertyDescriptors = PropertyUtils.getPropertyDescriptors(beanType);
        
        ArrayList<PropertyDescriptor> result = new ArrayList<PropertyDescriptor>();
        
        for (int i = 0; i < propertyDescriptors.length; i++) {
            PropertyDescriptor propertyDescriptor = propertyDescriptors[i];
            
            if (propertyDescriptor instanceof IndexedPropertyDescriptor) {
                continue;
            }
            
            Method readMethod = propertyDescriptor.getReadMethod();
            if (readMethod != null && !readMethod.getDeclaringClass().equals(Object.class)) {
                result.add(propertyDescriptor);
            }
        }
        
        return result.toArray(new PropertyDescriptor[result.size()]);
    }
    
    /**
     * Tests in a static way if a given property string can be resolved to a valid PropertyDescriptor on a provided Java bean class. The
     * method is able to resolve nested properties (e.g. Address.City.PostCode), mapped (e.g. PhoneNumer(home)) or indexed properties (e.g.
     * ChildName[2]).
     * 
     * @param propertyString the string that describes the property to resolve
     * @param beanClass the bean on that the property string is resolved
     * @return a property descriptor for the given property string
     */
    public static PropertyDescriptor staticResolvePropertyDescriptor(String propertyString, Class<?> beanClass) {
        int indexOfDot = propertyString.indexOf(".");
        if (indexOfDot == -1) {
            return getPropertyDescriptorForProperty(propertyString, beanClass);
        }
        
        String firstProperty = propertyString.substring(0, indexOfDot);
        String remainingProperties = propertyString.substring(indexOfDot + 1);
        PropertyDescriptor firstPropertyDescriptor = getPropertyDescriptorForProperty(firstProperty, beanClass);
        if (firstPropertyDescriptor == null) {
            return null;
        }
        
        Class<?> propertyType = resolvePropertyType(firstPropertyDescriptor);
        return staticResolvePropertyDescriptor(remainingProperties, propertyType);
    }
    
    /**
     * Returns a type for the given property descriptor even if the provided descriptor is a instance of MappedPropertyDescriptor
     * 
     * @param propertyDescriptor the property descriptor
     * @return the type of the given property descriptor
     */
    public static Class<?> resolvePropertyType(PropertyDescriptor propertyDescriptor) {
        if (propertyDescriptor instanceof MappedPropertyDescriptor) {
            return ((MappedPropertyDescriptor)propertyDescriptor).getMappedPropertyType();
        } else if (propertyDescriptor instanceof IndexedPropertyDescriptor) {
            return ((IndexedPropertyDescriptor)propertyDescriptor).getIndexedPropertyType();
        }
        return propertyDescriptor.getPropertyType();
    }
    
    private static PropertyDescriptor getPropertyDescriptorForProperty(String propertyName, Class<?> beanClass) {
        if (propertyName.endsWith(")")) {
            propertyName = propertyName.substring(0, propertyName.indexOf("("));
        } else if (propertyName.endsWith("]")) {
            propertyName = propertyName.substring(0, propertyName.indexOf("["));
        }
        
        PropertyDescriptor[] propertyDescriptors = PropertyUtils.getPropertyDescriptors(beanClass);
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if (propertyName.equals(propertyDescriptor.getName())) {
                return propertyDescriptor;
            }
        }
        
        try {
            return new MappedPropertyDescriptor(propertyName, beanClass);
        } catch (IntrospectionException e) {
            return null;
        }
    }
}
