/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding;

import com.ulcjava.applicationframework.application.form.model.FormModel;
import com.ulcjava.applicationframework.application.form.model.PropertyAccessException;
import com.ulcjava.applicationframework.application.form.model.SetPropertyCommand;
import com.ulcjava.base.application.ULCRadioButton;
import com.ulcjava.base.application.ULCTextArea;
import com.ulcjava.base.application.event.ActionEvent;
import com.ulcjava.base.application.event.IActionListener;

/**
 * Binds a {@link ULCRadioButton} to a {@link FormModel} property.
 */
public class RadioButtonBinder extends AbstractWidgetBinder<ULCRadioButton> {

    private final Object fValue;

    /**
     * Creates a binding between the given {@link ULCTextArea} and the given property.
     * 
     * @param model that holds the property. Must not be <code>null</code>.
     * @param propertyName name of the property to be bound. Must not be <code>null</code>.
     * @param radioButton to be bound. Must not be <code>null</code>.
     * @param value that the radio button represents. Must not be <code>null</code>.
     */
    public RadioButtonBinder(FormModel<?> model, String propertyName, ULCRadioButton radioButton, Object value) {
        super(model, propertyName, radioButton);
        if (value == null) {
            throw new IllegalArgumentException("value must not be null");
        }
        fValue = value;
        getRadioButton().addActionListener(new IActionListener() {
            public void actionPerformed(ActionEvent event) {
                try {
                    getModel().setProperty(new SetPropertyCommand(getPropertyName(), getValue()));
                } catch (PropertyAccessException e) {
                    throw new RuntimeException(e);
                }
            }
        });
    }

    /**
     * @return the {@link ULCRadioButton} that is bound.
     */
    protected ULCRadioButton getRadioButton() {
        return getWidget();
    }

    /**
     * Sets the selected state of the radio button to true if the property's value is equal to the value that the radio
     * button represents.
     */
    public void updateValueFromModel(FormModel<?> model) {
        try {
            getRadioButton().setSelected(getValue().equals(model.getProperty(getPropertyName())));
        } catch (PropertyAccessException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * @return value that the radio button represents.
     */
    protected Object getValue() {
        return fValue;
    }

}
