/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding;

import com.ulcjava.applicationframework.application.form.model.FormModel;

import java.io.Serializable;

/**
 * Binds a property of the form model to a {@link IViewUpdater}. Most conveniently used to create an anonymous inner
 * class.
 * 
 * <pre>
 * aWidgetBinderManager.bindViewUpdater(&quot;aProperty&quot;, new IViewUpdater() {
 *     public void updateView(Object newValue) {
 *         // the newValue is the property's current value.
 *     }
 * });
 * </pre>
 */
public class ViewUpdaterBinder implements IFormModelBinding, Serializable {

    private final IViewUpdater fViewUpdater;
    private final String fPropertyName;

    /**
     * Creates a ViewUpdaterBinder that updates the given {@link IViewUpdater} with the value of the given property.
     * 
     * @param propertyName name of the property to be bound. Must not be <code>null</code>.
     * @param viewUpdater uses the property's value to update the view. Must not be <code>null</code>.
     */
    public ViewUpdaterBinder(String propertyName, IViewUpdater viewUpdater) {
        if (viewUpdater == null) {
            throw new IllegalArgumentException("viewUpdater must not be null");
        }
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        fPropertyName = propertyName;
        fViewUpdater = viewUpdater;
    }

    /**
     * @return the {@link IViewUpdater} that is bound to the property.
     */
    public IViewUpdater getViewUpdater() {
        return fViewUpdater;
    }

    /**
     * @return the name of the property that is bound.
     */
    public String getPropertyName() {
        return fPropertyName;
    }

    public void attachToModel(FormModel<?> model) {
        if (model == null) {
            throw new IllegalArgumentException("Model must not be null");
        }
        updateValueFromModel(model);
    }


    public void detachFromModel(FormModel<?> model) {

    }

    public void updateValueFromModel(FormModel<?> model) {
        getViewUpdater().updateView(model.getProperty(getPropertyName()));
    }

}
