/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding;

import com.ulcjava.applicationframework.application.form.model.FormModel;
import com.ulcjava.base.application.ULCCheckBox;
import com.ulcjava.base.application.ULCComboBox;
import com.ulcjava.base.application.ULCList;
import com.ulcjava.base.application.ULCRadioButton;
import com.ulcjava.base.application.ULCSlider;
import com.ulcjava.base.application.ULCSpinner;
import com.ulcjava.base.application.ULCTextArea;
import com.ulcjava.base.application.ULCTextField;
import com.ulcjava.base.application.datatype.ULCAbstractErrorManager;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * Manages a set of bindings between the form and the bean in a {@link FormModel}. If the {@code FormModel} is replaced
 * the bindings to old model are released and the widgets are then bound to the new model.
 * 
 * @param <T> The generic type representing the type of bean wrapped by the {@code FormModel}
 */
public class WidgetBinderManager<T extends FormModel<?>> implements Serializable {

    private T fModel;
    private List<IFormModelBinding> fBindings;

    /**
     * Creates a {@code WidgetBinderManager} without a {@code FormModel}.
     */
    public WidgetBinderManager() {
        fBindings = new ArrayList<IFormModelBinding>();
        fModel = null;
    }

    /**
     * Creates a {@code WidgetBinderManager} with the given {@code  FormModel}.
     * 
     * @param model the FormModel to be bound.
     */
    public WidgetBinderManager(T model) {
        this();
        setModel(model);
    }


    /**
     * @return the bound {@code FormModel}.
     */
    public T getModel() {
        return fModel;
    }

    /**
     * Replaces the bound {@code FormModel} with the new one.
     * 
     * @param model the new formModel to be bound.
     */
    public void setModel(T model) {
        T oldModel = fModel;
        fModel = model;
        moveBindings(oldModel);
    }


    private void moveBindings(T oldModel) {
        for (IFormModelBinding binding : fBindings) {
            if (oldModel != null) {
                binding.detachFromModel(oldModel);
            }
            if (getModel() != null) {
                binding.attachToModel(getModel());
            }
        }
    }

    /**
     * Updates the form by propagating the values of the model to the form. Calls
     * {@code updateValueFromModel(getModel())} on all bindings.
     */
    public void updateFromModel() {
        T model = getModel();
        if (model != null) {
            for (IFormModelBinding binding : fBindings) {
                binding.updateValueFromModel(model);
            }
        }
    }

    /**
     * Binds a {@code ULCSlider}.
     * 
     * @param propertyName the property name on which the slider is bound.
     * @param slider the {@code ULCSlider} to be bound.
     */
    public void bindSlider(final String propertyName, final ULCSlider slider) {
        attach(new SliderBinder(getModel(), propertyName, slider));
    }

    /**
     * Binds a {@code ULCSpinner}.
     * 
     * @param propertyName the property name on which the spinner is bound.
     * @param spinner the {@code ULCSpinner} to be bound.
     */
    public void bindSpinner(final String propertyName, final ULCSpinner spinner) {
        attach(new SpinnerBinder(getModel(), propertyName, spinner));
    }


    /**
     * Binds a {@code ULCTextField}.
     * 
     * @param propertyName the property name on which the text field is bound.
     * @param textField the {@code ULCTextField} to be bound.
     * @param errorManager the {@code ULCErrorManager} that keeps errors that are detected by the text field's datatype
     *            on the client side.
     */
    public void bindTextField(final String propertyName, final ULCTextField textField,
            ULCAbstractErrorManager errorManager) {
        attach(new TextFieldBinder(getModel(), propertyName, textField, errorManager));
    }

    /**
     * Binds a {@code ULCComboBox}.
     * 
     * @param propertyName the property name on which the combobox is bound.
     * @param comboBox the {@code ULCComboBox} to be bound.
     */
    public void bindComboBox(final String propertyName, final ULCComboBox comboBox) {
        attach(new ComboBoxBinder(getModel(), propertyName, comboBox));
    }


    /**
     * Binds a {@code ULCCheckBox}.
     * 
     * @param propertyName the property name on which the check box is bound.
     * @param checkBox the {@code ULCCheckBox} to be bound.
     * @param forTrue if true, true is set on the property if the checkBox is checked, if false, true is set on the
     *            property if the checkBox is unchecked.
     */
    public void bindCheckBox(final String propertyName, final ULCCheckBox checkBox, final boolean forTrue) {
        attach(new CheckBoxBinder(getModel(), propertyName, checkBox, forTrue));

    }

    /**
     * Binds a {@code ULCRadioButton}.
     * 
     * @param propertyName the property name on which the radio button is bound.
     * @param radioButton the {@code ULCRadioButton} to be bound.
     * @param value to be set the property to if the radio button is selected.
     */
    public void bindRadioButton(final String propertyName, final ULCRadioButton radioButton, final Object value) {
        attach(new RadioButtonBinder(getModel(), propertyName, radioButton, value));
    }

    /**
     * Binds a {@code ULCTextArea}.
     * 
     * @param propertyName the property name on which the test area is bound.
     * @param textArea the {@code ULCTextArea} to be bound.
     */
    public void bindTextArea(final String propertyName, final ULCTextArea textArea) {
        attach(new TextAreaBinder(getModel(), propertyName, textArea));
    }

    /**
     * Binds a {@code ULCList}.
     * 
     * @param propertyName the property name on which the list is bound.
     * @param widget the {@code ULCList} to be bound.
     */
    public void bindList(String propertyName, ULCList widget) {
        attach(new ListBinder(getModel(), propertyName, widget));
    }

    /**
     * Binds a {@link IViewUpdater} to the mandatory state of the given property.
     * 
     * @param propertyName the property name on which the viewUpdater is bound.
     * @param viewUpdater that updates the form with the value of the mandatory state of the property.
     */
    public void bindMandatoryStateListener(String propertyName, IViewUpdater viewUpdater) {
        attach(new ViewUpdaterBinder(FormModel.getMandatoryStatePropertyName(propertyName), viewUpdater));
    }

    /**
     * Binds a {@link IViewUpdater} to the enabled state of the given property.
     * 
     * @param propertyName the property name on which the viewUpdater is bound.
     * @param viewUpdater that updates the form with the value of the enabled state of the property.
     */
    public void bindEnabledStateListener(String propertyName, IViewUpdater viewUpdater) {
        attach(new ViewUpdaterBinder(FormModel.getEnabledStatePropertyName(propertyName), viewUpdater));
    }

    /**
     * Binds a {@link IViewUpdater} to the readonly state of the given property.
     * 
     * @param propertyName the property name on which the viewUpdater is bound.
     * @param viewUpdater that updates the form with the value of the readonly state of the property.
     */
    public void bindReadonlyStateListener(String propertyName, IViewUpdater viewUpdater) {
        attach(new ViewUpdaterBinder(FormModel.getReadonlyStatePropertyName(propertyName), viewUpdater));
    }

    /**
     * Binds a {@link IViewUpdater} to the error state of the given property.
     * 
     * @param propertyName the property name on which the viewUpdater is bound.
     * @param viewUpdater that updates the form with the value of the error state of the property.
     */
    public void bindErrorStateListener(String propertyName, IViewUpdater viewUpdater) {
        propertyName = FormModel.getErrorStatePropertyName(propertyName);
        bindViewUpdater(propertyName, viewUpdater);
    }

    /**
     * Binds a {@link IViewUpdater} to the given property.
     * 
     * @param propertyName the property name on which the viewUpdater is bound.
     * @param viewUpdater that updates the form with the value of the property.
     */
    public void bindViewUpdater(String propertyName, IViewUpdater viewUpdater) {
        attach(new ViewUpdaterBinder(propertyName, viewUpdater));
    }

    /**
     * Attach a {@link IFormModelBinding} to the model and add it to the list of binders.
     * 
     * @param binder the {@code IFormModelBinding} to be added.
     */
    public void attach(IFormModelBinding binder) {
        T model = getModel();
        if (model != null) {
            binder.attachToModel(model);
        }
        fBindings.add(binder);
    }
}
