/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding.table;

import java.io.Serializable;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;


/**
 * The <code>ObservableCollections</code> class provides one factory method to create an observable list. An
 * <code>ObservableListListener</code> can be registered with <code>ObservableList</code> to notify about add, remove and update operations
 * on the list.
 */
public final class ObservableCollections {
    
    /**
     * Creates an <code>ObservableList</code> instance that is a wrapper for the given List.
     * 
     * @param <T> The formal type for the list elements
     * @param list The <code>List</code> instance that gets wrapped by the <code>ObservableList</code>
     * @return An <code>ObservableList</code> instance
     */
    public static <T> ObservableList<T> createObservableList(List<T> list) {
        if (list == null) {
            throw new IllegalArgumentException();
        }
        return new ObservableListImpl<T>(list);
    }
    
    @SuppressWarnings("serial")
    private static final class ObservableListImpl<T> extends AbstractList<T> implements ObservableList<T>, Serializable {
        private List<T> fList;
        private List<ObservableListListener<T>> fListeners;
        
        ObservableListImpl(List<T> list) {
            this.fList = list;
            fListeners = new CopyOnWriteArrayList<ObservableListListener<T>>();
        }
        
        public void addObservableListListener(ObservableListListener<T> listener) {
            fListeners.add(listener);
        }
        
        public void removeObservableListListener(ObservableListListener<T> listener) {
            fListeners.remove(listener);
        }
        
        @Override
        public T get(int index) {
            return fList.get(index);
        }
        
        @Override
        public int size() {
            return fList.size();
        }
        
        @Override
        public void add(int index, T element) {
            fList.add(index, element);
            modCount++;
            for (ObservableListListener<T> listener : fListeners) {
                listener.listElementsAdded(this, index, 1);
            }
        }
        
        @Override
        public boolean addAll(Collection<? extends T> collection) {
            return addAll(size(), collection);
        }
        
        @Override
        public boolean addAll(int index, Collection<? extends T> collection) {
            boolean result = fList.addAll(index, collection);
            if (result) {
                modCount++;
                for (ObservableListListener<T> listener : fListeners) {
                    listener.listElementsAdded(this, index, collection.size());
                }
            }
            
            return result;
        }
        
        @Override
        public void clear() {
            if (fList.size() == 0) {
                return;
            }
            
            List<T> oldElements = new ArrayList<T>(fList);
            fList.clear();
            modCount++;
            for (ObservableListListener<T> listener : fListeners) {
                listener.listElementsRemoved(this, 0, oldElements);
            }
        }
        
        @Override
        public int indexOf(Object o) {
            return fList.indexOf(o);
        }
        
        @Override
        public int lastIndexOf(Object o) {
            return fList.lastIndexOf(o);
        }
        
        @Override
        public T remove(int index) {
            T result = fList.remove(index);
            modCount++;
            for (ObservableListListener<T> listener : fListeners) {
                listener.listElementsRemoved(this, index, Collections.singletonList(result));
            }
            
            return result;
        }
        
        @Override
        public boolean remove(Object o) {
            int indexOfElement = fList.indexOf(o);
            boolean result = fList.remove(o);
            if (result) {
                modCount++;
                for (ObservableListListener<T> listener : fListeners) {
                    @SuppressWarnings("unchecked")
                    T element = (T)o;
                    
                    listener.listElementsRemoved(this, indexOfElement, Collections.singletonList(element));
                }
            }
            
            return result;
        }
        
        @Override
        public T set(int index, T element) {
            T oldValue = fList.set(index, element);
            for (ObservableListListener<T> listener : fListeners) {
                listener.listElementReplaced(this, index, oldValue);
            }
            
            return oldValue;
        }
        
        @Override
        public boolean contains(Object o) {
            return fList.contains(o);
        }
        
        @Override
        public boolean containsAll(Collection<?> c) {
            return fList.containsAll(c);
        }
        
        @Override
        public Object[] toArray() {
            return fList.toArray();
        }
        
        @Override
        public <E> E[] toArray(E[] a) {
            return fList.toArray(a);
        }
    }
}
