/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding.table;

import org.apache.commons.beanutils.PropertyUtils;

import com.ulcjava.applicationframework.application.binding.BindingUtils;
import com.ulcjava.applicationframework.application.form.model.PropertyAccessException;
import com.ulcjava.base.application.ULCTable;
import com.ulcjava.base.application.table.AbstractTableModel;
import com.ulcjava.base.application.table.ITableCellEditor;
import com.ulcjava.base.application.table.ITableCellRenderer;
import com.ulcjava.base.application.table.ITableModel;
import com.ulcjava.base.application.table.ULCTableColumn;
import com.ulcjava.base.application.table.ULCTableColumnModel;
import com.ulcjava.base.server.streamcoder.DefaultServerCoderRegistryProvider;
import com.ulcjava.base.shared.CoderRegistry;
import com.ulcjava.base.shared.IStreamCoder;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The {@code TableBinding} class provides a way to bind list of java beans of the same type to a {@code ULCTable}. Each bean is mapped to a
 * single row in a table, whereas the properties of the bean represents the cells of the table row. The mapping of bean properties to table
 * columns is defined by {@code ColumnBinding} objects which are created if the {@code setAutoCreateColumnFromBean} is set or that are added
 * manually to the {@code TableBinding}. Internally the given list of Java beans are mapped to a {@code ITableModel} model instance that is
 * set to the specified {@code ULCTable}. If no sorter or filter mechanism is set to the {@code ULCTabel} the order of the table rows is
 * defined by the order of the beans in the given list.
 * <p>
 * </p>
 * The following sample code illustrates the usage of {@code TableBinding}. While creating this {@code TableBinding} instance the {@code
 * setAutoCreateColumnFromBean} is set to true. Therefore all bean properties are mapped to default table columns with the property names as
 * column headers.
 * 
 * <pre><code>
 * List&lt;Person&gt; persons = createPersonList();
 * ULCTable table = new ULCTable();
 * TableBinding tableBinding = TableBinding.createTableBindingFromBeanList(persons, table, Person.class, true);
 * </code></pre>
 * 
 * The next code sample shows a manually configured {@code TableBinding}. Each {@code ColumnBinding}, which maps a bean property to a table
 * column, is added manually and is configured as needed. The configuration of a {@code ColumnBinding} object can be done by chaining the
 * different method calls for the modification of the {@code ColumnBinding}.
 * 
 * <pre><code>
 * //create the person List
 * List&lt;Person&gt; persons = createPersonList();
 * 
 * ULCTable table = new ULCTable();
 * 
 * TableBinding tableBinding = TableBinding.createTableBindingFromBeanList(persons, table, Person.class, false);
 * tableBinding.addColumnBinding(&quot;firstName&quot;).headerValue(&quot;First Name&quot;).preferredWidth(150);
 * tableBinding.addColumnBinding(&quot;lastName&quot;).headerValue(&quot;Last Name&quot;).preferredWidth(150);
 * tableBinding.addColumnBinding(&quot;vip&quot;).headerValue(&quot;VIP&quot;).preferredWidth(50).columnType(Boolean.class);
 * </code></pre>
 * 
 * If a {@code ObservableList} is provided while creating the {@code TableBinding} instance (with the call of the {@code
 * createTableBindingFromBeanList} factory method), the underlying table model will react on changes in the bean list. If a bean is removed,
 * added or changed, the {@code ULCTable} will be updated. If a bean property is updated, the appropriate table cell will be updated
 * automatically. If the underlying Java bean uses bounded properties, the table model will also react on changes to these properties.
 * <p>
 * A cell in the {@code ULCTable} is editable if all of the following criteria are true:
 * <ul>
 * <li>The property of the underlying bean is writable
 * <li>The {@code isEditable} property of the {@code ColumnBinding} is true.
 * </ul>
 * 
 * @param <T> the generic parameter that defines the type of the bean whose properties will be mapped to table columns
 * @see ULCTable
 */
@SuppressWarnings("serial")
public class TableBinding<T> implements Serializable {
    
    private final ULCTable fTable;
    private final ObservableList<T> fBeanList;
    private final Class<T> fBeanClass;
    private boolean fSetAutoCreatColumnFromBean;
    private final BeanTableModel fBeanTableModel;
    private final List<ColumnBinding> fColumnBindings;
    
    /**
     * The factory method to create an instance of {@code TableBinding} and to bind the given beans to the {@code ULCTable}. The list of
     * beans will be converted to a {@code ITableModel} instance which is set on the given {@code ULCTable}.
     * 
     * @param <T>
     * @param beans the list of beans holding the row information. If this provided list is an instance of {@code ObservableList} the
     *            {@code ULCTable} will be notified about the add, remove or update operations on the list and will then update the
     *            appropriate cells.
     * @param table the {@code ULCTable} that will be bound to the list of beans.
     * @param typeOfBean the object type of the beans that will be bound to the {@code ULCTable}
     * @param setAutoCreatColumnFromBean specifies if default columns will be created out of the information provided by the given beans.
     *            Each bean property will be mapped to a column. The property name will be used as column name. The order of the generated
     *            columns is specified by the order of properties.
     * @return returns the created {@code TableBinding} instance
     */
    public static <T> TableBinding<T> createTableBindingFromBeanList(List<T> beans, ULCTable table, Class<T> typeOfBean,
            boolean setAutoCreatColumnFromBean) {
        return new TableBinding<T>(beans, table, typeOfBean, setAutoCreatColumnFromBean);
    }
    
    /**
     * The factory method to create an instance of {@code TableBinding} and to bind the given beans to the {@code ULCTable}. The list of
     * beans will be converted to a {@code ITableModel} instance which is set on the given {@code ULCTable}.
     * 
     * @param <T>
     * @param beans the list of beans holding the row information. If this provided list is an instance of {@code ObservableList} the
     *            {@code ULCTable} will be notified about the add, remove or update operations on the list and will then update the
     *            appropriate cells.
     * @param table the {@code ULCTable} that will be bound to the list of beans.
     * @param typeOfBean the object type of the beans that will be bound to the {@code ULCTable}
     * @return returns the created {@code TableBinding} instance
     */
    public static <T> TableBinding<T> createTableBindingFromBeanList(List<T> beans, ULCTable table, Class<T> typeOfBean) {
        return new TableBinding<T>(beans, table, typeOfBean, false);
        
    }
    
    private TableBinding(List<T> beanList, ULCTable table, Class<T> beanClass, boolean setAutoCreatColumnFromBean) {
        if (beanList == null) {
            throw new IllegalArgumentException("The list of elements provided in the constructor can't be null");
        }
        
        if (table == null) {
            throw new IllegalArgumentException("The table provided in the constructor can't be null");
        }
        
        fTable = table;
        fBeanClass = beanClass;
        fSetAutoCreatColumnFromBean = setAutoCreatColumnFromBean;
        
        if (beanList instanceof ObservableList) {
            fBeanList = (ObservableList<T>)beanList;
        } else {
            fBeanList = ObservableCollections.createObservableList(beanList);
        }
        fBeanList.addObservableListListener(new BeanListHandler());
        
        fBeanTableModel = new BeanTableModel();
        fColumnBindings = new ArrayList<ColumnBinding>();
        prepareTableModel();
    }
    
    private void prepareTableModel() {
        fTable.setAutoCreateColumnsFromModel(false);
        deleteAllColumnsOnColumnModel();
        fTable.setModel(fBeanTableModel);
        
        if (fSetAutoCreatColumnFromBean) {
            createDefaultColumnsFromBean();
        }
    }
    
    private void deleteAllColumnsOnColumnModel() {
        ULCTableColumnModel columnModel = fTable.getColumnModel();
        while (columnModel.getColumnCount() > 0) {
            columnModel.removeColumn(columnModel.getColumn(0));
        }
    }
    
    /**
     * Notifies that the whole table has changed
     */
    public void bind() {
        fBeanTableModel.fireTableStructureChanged();
    }
    
    private void createDefaultColumnsFromBean() {
        deleteAllColumnsOnColumnModel();
        fColumnBindings.clear();
        
        PropertyDescriptor[] propertyDescriptors = BindingUtils.getSimpleReadablePropertyDescriptors(fBeanClass);
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            String columnName = propertyDescriptor.getName();
            addColumnBinding(columnName, columnName);
        }
        bind();
    }
    
    /**
     * Adds a {@code ColumnBinding} for a given bean property. The column header name will be defined by the property name and the column is
     * added to the end of the table.
     * 
     * @param propertyName The name of the bean property that is mapped to the {@code ULCTabel} column.
     * @return The modified {@code ColumnBinding} object.
     */
    public ColumnBinding addColumnBinding(String propertyName) {
        return addColumnBinding(propertyName, propertyName);
    }
    
    /**
     * Adds a {@code ColumnBinding} for a given bean property. The column is added to the end of the table.
     * 
     * @param propertyName The name of the bean property that is mapped to the {@code ULCTabel} column.
     * @param columnName The value of the column header.
     * @return The modified {@code ColumnBinding} object.
     */
    public ColumnBinding addColumnBinding(String propertyName, String columnName) {
        int index = fColumnBindings.size();
        ULCTableColumn tableColumn = new ULCTableColumn(index);
        ColumnBinding columnBinding = new ColumnBinding(propertyName, columnName, tableColumn);
        fColumnBindings.add(columnBinding);
        fTable.addColumn(tableColumn);
        
        addPropertyChangeListenerToAllBeans(propertyName);
        
        return columnBinding;
    }
    
    private void addPropertyChangeListenerToAllProperties(T bean) {
        PropertyDescriptor[] beanProperties = BindingUtils.getSimpleReadablePropertyDescriptors(fBeanClass);
        for (PropertyDescriptor propertyDescriptor : beanProperties) {
            addPropertyChangeListenerToBean(propertyDescriptor.getName(), bean);
        }
    }
    
    /*
     * Adds a property change listener for the given property to all beans
     */
    private void addPropertyChangeListenerToAllBeans(String propertyName) {
        for (final T bean : fBeanList) {
            addPropertyChangeListenerToBean(propertyName, bean);
        }
    }
    
    
    private void addPropertyChangeListenerToBean(String propertyName, final T bean) {
        ColumnBinding columnBinding = findColumnBindingForPropertyName(propertyName);
        if (columnBinding != null) {
            try {
                BeanPropertyHandler beanPropertyChangeListener = new BeanPropertyHandler(bean, columnBinding.getTableColumn()
                        .getModelIndex());
                
                Method method = bean.getClass().getMethod("addPropertyChangeListener",
                        new Class[] {String.class, PropertyChangeListener.class});
                method.invoke(bean, new Object[] {propertyName, beanPropertyChangeListener});
                
                columnBinding.addPropertyChangeListener(bean, beanPropertyChangeListener);
            } catch (Exception ignored) {
            }
        }
    }
    
    private void removePropertyChangeListenerFromAllProperties(T bean) {
        PropertyDescriptor[] beanProperties = BindingUtils.getSimpleReadablePropertyDescriptors(fBeanClass);
        for (PropertyDescriptor propertyDescriptor : beanProperties) {
            removePropertyChangeListenerFromBean(propertyDescriptor.getName(), bean);
        }
    }
    
    private void removePropertyChangeListenerFromAllBeans(String propertyName) {
        for (final T bean : fBeanList) {
            removePropertyChangeListenerFromBean(propertyName, bean);
        }
    }
    
    private void removePropertyChangeListenerFromBean(String propertyName, final T bean) {
        ColumnBinding columnBinding = findColumnBindingForPropertyName(propertyName);
        if (columnBinding != null) {
            PropertyChangeListener propertyChangeListener = columnBinding.getPropertyChangeListener(bean);
            if (propertyChangeListener != null) {
                try {
                    Method method = bean.getClass().getMethod("removePropertyChangeListener",
                            new Class[] {String.class, PropertyChangeListener.class});
                    method.invoke(bean, new Object[] {propertyName, propertyChangeListener});
                    
                    columnBinding.removePropertyChangeListener(bean);
                } catch (Exception ignored) {
                }
            }
        }
    }
    
    private ColumnBinding findColumnBindingForPropertyName(String propertyName) {
        for (ColumnBinding columnBinding : fColumnBindings) {
            if (columnBinding.getPropertyName().equals(propertyName)) {
                return columnBinding;
            }
        }
        return null;
    }
    
    /**
     * Removes the {@code ColumnBinding} instance from the list of column bindings and the mapped {@code ULCTableColumn} from the {@code
     * ULCTable} for the given property name.
     * 
     * @param propertyName the property that specifies the columnBinding, which should be removed.
     */
    public void removeColumnBinding(String propertyName) {
        ColumnBinding columnBinding = findColumnBindingForPropertyName(propertyName);
        if (columnBinding != null) {
            removeColumnBinding(columnBinding);
        }
    }
    
    /**
     * Removes the given {@code ColumnBinding} instance from the list of column bindings and the mapped {@code ULCTableColumn} from the
     * {@code ULCTable}.
     * 
     * @param columnBinding The columnBinding that will be removed. Also the mapped {@code ULCTableColumn} will be removed from the {@code
     *            ULCTable}
     */
    public void removeColumnBinding(ColumnBinding columnBinding) {
        removePropertyChangeListenerFromAllBeans(columnBinding.getPropertyName());
        fColumnBindings.remove(columnBinding);
        fTable.removeColumn(columnBinding.getTableColumn());
    }
    
    /**
     * Removes the {@code ULCTableColumn} from the {@code ULCTable} and also removes the corresponding column binding instance from the list
     * of column bindings.
     * 
     * @param tableColumn The {@code ULCTableColumn} that should be removed.
     */
    public void removeColumnBinding(ULCTableColumn tableColumn) {
        ColumnBinding columnBinding = getColumnBinding(tableColumn);
        if (columnBinding != null) {
            removeColumnBinding(columnBinding);
        }
    }
    
    /**
     * Returns the {@code ColumnBinding} instance for the given {@code ULCTableColumn}. If no {@code ColumnBinding} is registered for the
     * given {@code ULCTableColumn}, the method returns {@code null}.
     * 
     * @param tableColumn The {@code ULCTableColumn}
     * @return The {@code ColumnBinding}, which is mapped to the {@code ULCTableColumn} with the given index, {@code null} if no {@code
     *         ColumnBinding} was found
     */
    public ColumnBinding getColumnBinding(ULCTableColumn tableColumn) {
        for (ColumnBinding columnBinding : fColumnBindings) {
            if (tableColumn.equals(columnBinding.getTableColumn())) {
                return columnBinding;
            }
        }
        
        return null;
    }
    
    /**
     * Returns the list of all {@code ColumnBinding} instances
     * 
     * @return The list of all {@code ColumnBinding} instances
     */
    public List<ColumnBinding> getColumnBindings() {
        return fColumnBindings;
    }
    
    /**
     * Returns the table model that is used to map the list of Java beans to the corresponding table rows.
     * 
     * @return the table model that is used for the bean to table row mapping
     */
    public ITableModel getTableModel() {
        return fBeanTableModel;
    }
    
    /**
     * If this property is set to true, a default column is created for each property contained in the specified bean. The value of the
     * column header is set to the property name. If this property is set to false then columns have to be created manually.
     * 
     * @param autoCreateColumnsFromBean If set to true, default columns are created for all properties in the given bean.
     */
    public void setAutoCreateColumnsFromBean(boolean autoCreateColumnsFromBean) {
        if (fSetAutoCreatColumnFromBean != autoCreateColumnsFromBean) {
            fSetAutoCreatColumnFromBean = autoCreateColumnsFromBean;
            if (fSetAutoCreatColumnFromBean) {
                createDefaultColumnsFromBean();
            }
        }
    }
    
    /**
     * Returns the list of beans used for this table binding.
     * 
     * @return The observable list that contains the beans, which represents the table rows.
     */
    public ObservableList<T> getBeanList() {
        return fBeanList;
    }
    
    /**
     * Returns the Java bean for the specified table row index
     * 
     * @param row The index of the table row
     * @return the Java bean, which is mapped to this row
     */
    public T getBeanForRow(int row) {
        return fBeanList.get(row);
    }
    
    /**
     * This handler is responsible to react to changes in the bean list observed by the table binding.
     */
    private class BeanListHandler implements ObservableListListener<T> {
        public void listElementReplaced(ObservableList<T> list, int index, T oldElement) {
            T newElement = list.get(index);
            
            removePropertyChangeListenerFromAllProperties(oldElement);
            addPropertyChangeListenerToAllProperties(newElement);
            
            fBeanTableModel.fireTableRowsUpdated(index, index);
        }
        
        public void listElementsAdded(ObservableList<T> list, int index, int length) {
            for (int i = index; i < (index + length); i++) {
                addPropertyChangeListenerToAllProperties(list.get(i));
            }
            
            fBeanTableModel.fireTableRowsInserted(index, index + length - 1);
        }
        
        public void listElementsRemoved(ObservableList<T> list, int index, List<T> oldElements) {
            for (int i = index, n = 0; i < (index + oldElements.size()); i++, n++) {
                removePropertyChangeListenerFromAllProperties(oldElements.get(n));
            }
            
            fBeanTableModel.fireTableRowsDeleted(index, index + oldElements.size() - 1);
        }
    }
    
    /**
     * This handler is responsible to react to changes in a property of a bean observed by the table binding.
     */
    private class BeanPropertyHandler implements PropertyChangeListener, Serializable {
        private final T fBean;
        private final int fColumn;
        
        public BeanPropertyHandler(T bean, int column) {
            fBean = bean;
            fColumn = column;
        }
        
        public void propertyChange(PropertyChangeEvent evt) {
            fBeanTableModel.fireTableCellUpdated(fBeanList.indexOf(fBean), fColumn);
        }
    }
    
    /**
     * This class contains the necessary information about the mapping of a single bean property to a table column. It also provides
     * convenience methods to configure this table columns. The configuration can be done by method chaining. The following sample shows the
     * creation and configuration of a single table column:
     * 
     * <pre>
     * 
     * <code> tableBinding.addColumnBinding(&quot;firstName&quot;).headerValue(&quot;First Name&quot;).preferredWidth(150); </code>
     * 
     * </pre>
     */
    public class ColumnBinding implements Serializable {
        private String fPropertyName;
        private String fColumnName;
        private boolean fEditable;
        private Class<?> fColumnType;
        private final ULCTableColumn fTableColumn;
        private Map<T, BeanPropertyHandler> fPropertyChangeListenerByBean;
        
        /**
         * Creates a column binding for a given bean property.
         * 
         * @param propertyName The name of the property
         * @param columnName The name of the column that appears on the column header
         * @param tableColumn {@code ULCTableColumn} the table column to which the property will be bound
         */
        public ColumnBinding(String propertyName, String columnName, ULCTableColumn tableColumn) {
            fPropertyName = propertyName;
            fColumnName = columnName;
            fTableColumn = tableColumn;
            
            fColumnType = initColumnType();
            fPropertyChangeListenerByBean = new HashMap<T, BeanPropertyHandler>();
            
        }
        
        private Class<?> initColumnType() {
            PropertyDescriptor propertyDescriptor = BindingUtils.staticResolvePropertyDescriptor(fPropertyName, fBeanClass);
            if (propertyDescriptor != null) {
                Class<?> serverType = BindingUtils.resolvePropertyType(propertyDescriptor);
                
                return convertToClientType(serverType);
            } else {
                // its not always possible to resolve a property descriptor in a static way -> return the object type as default value
                return Object.class;
            }
        }
        
        private Class<?> convertToClientType(Class<?> propertyType) {
            if (!propertyType.isPrimitive()) {
                if (propertyType != String.class && propertyType != Class.class && !hasCoder(propertyType)) {
                    propertyType = Object.class;
                }
                
                return propertyType;
            }
            if (propertyType == boolean.class) {
                return Boolean.class;
            }
            if (propertyType == char.class) {
                return Character.class;
            }
            if (propertyType == byte.class) {
                return Byte.class;
            }
            if (propertyType == short.class) {
                return Short.class;
            }
            if (propertyType == int.class) {
                return Integer.class;
            }
            if (propertyType == long.class) {
                return Long.class;
            }
            if (propertyType == float.class) {
                return Float.class;
            }
            if (propertyType == double.class) {
                return Double.class;
            }
            
            throw new IllegalArgumentException(propertyType + " is not permitted");
        }
        
        private boolean hasCoder(Class<?> type) {
            // Currently there is no way to get the coder registry used by ULC.
            // => Create a new one
            // => but then of course we don't know of any user registered coders :-(
            DefaultServerCoderRegistryProvider coderRegistryProvider = new DefaultServerCoderRegistryProvider();
            CoderRegistry coderRegistry = coderRegistryProvider.getCoderRegistry();
            IStreamCoder coder = coderRegistry.getCoder(type);
            
            return coder != null;
        }
        
        protected BeanPropertyHandler getPropertyChangeListener(T bean) {
            return fPropertyChangeListenerByBean.get(bean);
            
        }
        
        protected void addPropertyChangeListener(T bean, BeanPropertyHandler beanPropertyChangeListener) {
            fPropertyChangeListenerByBean.put(bean, beanPropertyChangeListener);
        }
        
        protected void removePropertyChangeListener(T bean) {
            fPropertyChangeListenerByBean.remove(bean);
        }
        
        /**
         * Returns the mapped {@code ULCTableColumn} for this {@code ColumnBinding}
         * 
         * @return the mapped {@code ULCTableColumn}
         */
        public ULCTableColumn getTableColumn() {
            return fTableColumn;
        }
        
        /**
         * Returns the view index of the mapped {@code ULCTableColumn} for this {@code ColumnBinding}
         * 
         * @return The view index of the mapped {@code ULCTableColumn}
         * @deprecated use table.getColumnModel().getColumnIndex(columnBinding.getTableColumn) instead
         */
        public int getTableColumnIndex() {
            return fTable.getColumnModel().getColumnIndex(fTableColumn);
        }
        
        /**
         * Returns the value of the Java bean property in the given row. In case there is no {@code IStreamCoder} registered for the type of
         * the property, then a string value of the property value is returned.
         * 
         * @param row The index of the row to which the Java bean is mapped
         * @return the value of the Java bean property
         * @see com.ulcjava.base.shared.IStreamCoder
         */
        public Object getProperty(int row) {
            try {
                T bean = fBeanList.get(row);
                Object property = PropertyUtils.getProperty(bean, fPropertyName);
                
                if (fColumnType == Object.class) {
                    property = String.valueOf(property);
                }
                
                return property;
            } catch (Exception e) {
                throw new PropertyAccessException("Can't get property and propertyType for property " + fPropertyName, e);
            }
        }
        
        /**
         * Sets the value to the Java bean property that is mapped to this {@code ColumnBinding} instance.
         * 
         * @param value The new value for the Java bean property
         * @param row the table row to which the Java bean containing the property is mapped
         */
        public void setProperty(Object value, int row) {
            T bean = fBeanList.get(row);
            try {
                PropertyUtils.setProperty(bean, fPropertyName, value);
            } catch (Exception e) {
                throw new PropertyAccessException("Can't set property: " + fPropertyName + " -> value: " + value, e);
            }
        }
        
        /**
         * Returns the type for this {@code ColumnBinding} and therefore for the mapped {@code ULCTableColumn}. The type of the column is by
         * default the type of the mapped Java bean property.
         * 
         * @return the type for the mapped {@code ULCTableColumn}
         * @see #columnType
         */
        public Class<?> getColumnType() {
            return fColumnType;
        }
        
        /**
         * Returns the name which is set for this {@code ColumnBinding}. This value is set to the corresponding {@code ULCTableColumn}
         * header.
         * 
         * @return The column name.
         */
        public String getColumnName() {
            return fColumnName;
        }
        
        /**
         * Sets the column name for this {@code ColumnBinding}. This value is set to the appropriate {@code ULCTableColumn} header.
         * 
         * @param columnName The column name.
         * @return The modified {@code ColumnBinding} instance
         */
        public ColumnBinding columnName(String columnName) {
            fColumnName = columnName;
            return this;
        }
        
        /**
         * Returns the name of the Java bean property that is mapped to this {@code ColumnBinding} instance
         * 
         * @return The Java bean property name
         */
        public String getPropertyName() {
            return fPropertyName;
        }
        
        /**
         * Sets a cell editor on the referenced {@code ULCTableColumn}.
         * 
         * @param cellEditor The {@code ITableCellEditor} that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.table.ULCTableColumn#setCellEditor
         */
        public ColumnBinding cellEditor(ITableCellEditor cellEditor) {
            fTableColumn.setCellEditor(cellEditor);
            return this;
        }
        
        /**
         * Sets a cell renderer on the referenced {@code ULCTableColumn}.
         * 
         * @param cellRenderer The {@code ITableCellRenderer} that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.table.ULCTableColumn#setCellEditor
         */
        public ColumnBinding cellRenderer(ITableCellRenderer cellRenderer) {
            fTableColumn.setCellRenderer(cellRenderer);
            return this;
        }
        
        /**
         * Sets a header renderer on the referenced {@code ULCTableColumn}.
         * 
         * @param headerRenderer The {@code ITableCellRenderer} for the header that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.table.ULCTableColumn#setHeaderRenderer
         */
        public ColumnBinding headerRenderer(ITableCellRenderer headerRenderer) {
            fTableColumn.setHeaderRenderer(headerRenderer);
            return this;
        }
        
        /**
         * Sets a header value on the referenced {@code ULCTableColumn}.
         * 
         * @param headerValue The header value object that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#setHeaderValue
         */
        public ColumnBinding headerValue(Object headerValue) {
            fTableColumn.setHeaderValue(headerValue);
            return this;
        }
        
        /**
         * Sets an identifier for the referenced {@code ULCTableColumn}.
         * 
         * @param identifier The identifier object that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#setIdentifier
         */
        public ColumnBinding identifier(Object identifier) {
            fTableColumn.setIdentifier(identifier);
            return this;
        }
        
        /**
         * Sets the maximum width for the referenced {@code ULCTableColumn}.
         * 
         * @param maxWidth The max width that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#setMaxWidth
         */
        public ColumnBinding maxWidth(int maxWidth) {
            fTableColumn.setMaxWidth(maxWidth);
            return this;
        }
        
        /**
         * Sets the minimum width for the referenced {@code ULCTableColumn}
         * 
         * @param minWidth The min width that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#setMinWidth
         */
        public ColumnBinding minWidth(int minWidth) {
            fTableColumn.setMinWidth(minWidth);
            return this;
        }
        
        /**
         * Sets the model index on the referenced {@code ULCTableColumn}.
         * 
         * @param modelIndex The model index that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#setModelIndex
         * @deprecated don't use this method
         */
        public ColumnBinding modelIndex(int modelIndex) {
            fTableColumn.setModelIndex(modelIndex);
            return this;
        }
        
        /**
         * Sets the preferred width on the referenced {@code ULCTableColumn}.
         * 
         * @param preferredWidth The preferred width that is set on the referenced table column.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#setPreferredWidth
         */
        public ColumnBinding preferredWidth(int preferredWidth) {
            fTableColumn.setPreferredWidth(preferredWidth);
            return this;
        }
        
        /**
         * Sets the resizable property on the referenced {@code ULCTableColumn}.
         * 
         * @param isResizable The resizable property that is set on the referenced {@code ULCTableColumn}.
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#setResizable
         */
        public ColumnBinding resizable(boolean isResizable) {
            fTableColumn.setResizable(isResizable);
            return this;
        }
        
        /**
         * Sets the sizeWidthToFit property on the referenced {@code ULCTableColumn}
         * 
         * @return The modified {@code ColumnBinding} instance.
         * @see com.ulcjava.base.application.AbstractColumn#sizeWidthToFit
         */
        public ColumnBinding sizeWidthToFit() {
            fTableColumn.sizeWidthToFit();
            return this;
        }
        
        /**
         * Specifies if the cells of this column should be editable. For a cell to be editable, its the mapped property must be editable.
         * 
         * @param editable The editable property that is set on the referenced {@code ULCTableColumn}.
         * @return The modified {@code ColumnBinding} instance.
         */
        public ColumnBinding setEditable(boolean editable) {
            fEditable = editable;
            return this;
        }
        
        /**
         * Returns if this column is editable.
         * 
         * @return true if the editable property is set to true (default), false otherwise
         */
        public boolean isEditable() {
            return fEditable;
        }
        
        /**
         * Sets the column Type for this {@code ColumnBinding}.
         * 
         * @param columnType The column type that is returned when the table model determines the type of the displayed column
         * @return The modified {@code ColumnBinding} instance.
         */
        public ColumnBinding columnType(Class<?> columnType) {
            fColumnType = columnType;
            return this;
        }
        
        @Override
        public String toString() {
            return super.toString() + " -> columnName: " + fColumnName + " -> columnNo: " + fTableColumn.getModelIndex();
        }
    }
    
    private class BeanTableModel extends AbstractTableModel {
        public int getColumnCount() {
            return fColumnBindings.size();
        }
        
        public int getRowCount() {
            return fBeanList.size();
        }
        
        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return fColumnBindings.get(columnIndex).getColumnType();
        }
        
        @Override
        public String getColumnName(int columnIndex) {
            return fColumnBindings.get(columnIndex).getColumnName();
        }
        
        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return fColumnBindings.get(columnIndex).isEditable();
        }
        
        public Object getValueAt(int rowIndex, int columnIndex) {
            return fColumnBindings.get(columnIndex).getProperty(rowIndex);
        }
        
        @Override
        public void setValueAt(Object value, int rowIndex, int columnIndex) {
            fColumnBindings.get(columnIndex).setProperty(value, rowIndex);
        }
    }
}
