/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.applicationframework.application.Application;
import com.ulcjava.applicationframework.application.ApplicationContext;
import com.ulcjava.applicationframework.application.ResourceMap;
import com.ulcjava.applicationframework.application.binding.IFormModelBinding;
import com.ulcjava.applicationframework.application.binding.IViewUpdater;
import com.ulcjava.applicationframework.application.binding.WidgetBinderManager;
import com.ulcjava.applicationframework.application.form.model.FormModel;
import com.ulcjava.applicationframework.application.form.model.PropertyAccessException;
import com.ulcjava.base.application.BorderFactory;
import com.ulcjava.base.application.ClientContext;
import com.ulcjava.base.application.DefaultComboBoxModel;
import com.ulcjava.base.application.DefaultListModel;
import com.ulcjava.base.application.IComboBoxModel;
import com.ulcjava.base.application.IListModel;
import com.ulcjava.base.application.ISpinnerModel;
import com.ulcjava.base.application.ULCBoxPane;
import com.ulcjava.base.application.ULCCheckBox;
import com.ulcjava.base.application.ULCComboBox;
import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCFiller;
import com.ulcjava.base.application.ULCLabel;
import com.ulcjava.base.application.ULCList;
import com.ulcjava.base.application.ULCPasswordField;
import com.ulcjava.base.application.ULCProxy;
import com.ulcjava.base.application.ULCRadioButton;
import com.ulcjava.base.application.ULCScrollPane;
import com.ulcjava.base.application.ULCSeparator;
import com.ulcjava.base.application.ULCSlider;
import com.ulcjava.base.application.ULCSpinner;
import com.ulcjava.base.application.ULCSpinnerDateModel;
import com.ulcjava.base.application.ULCSpinnerNumberModel;
import com.ulcjava.base.application.ULCTabbedPane;
import com.ulcjava.base.application.ULCTextArea;
import com.ulcjava.base.application.ULCTextComponent;
import com.ulcjava.base.application.ULCTextField;
import com.ulcjava.base.application.border.ULCAbstractBorder;
import com.ulcjava.base.application.datatype.IDataType;
import com.ulcjava.base.application.datatype.ULCAbstractErrorManager;
import com.ulcjava.base.application.datatype.ULCDateDataType;
import com.ulcjava.base.application.datatype.ULCDefaultErrorManager;
import com.ulcjava.base.application.datatype.ULCNumberDataType;
import com.ulcjava.base.application.datatype.ULCPercentDataType;
import com.ulcjava.base.application.enabler.IEnabler;
import com.ulcjava.base.application.enabler.IHasChangedSource;
import com.ulcjava.base.application.enabler.ULCHasChangedEnabler;
import com.ulcjava.base.application.enabler.ULCMandatoryAndEnabler;
import com.ulcjava.base.application.event.IRoundTripListener;
import com.ulcjava.base.application.event.RoundTripEvent;
import com.ulcjava.base.application.util.Color;
import com.ulcjava.base.server.ULCSession;
import com.ulcjava.base.shared.ErrorCodes;
import com.ulcjava.base.shared.ErrorObject;
import com.ulcjava.base.shared.UlcEventCategories;
import com.ulcjava.base.shared.UlcEventConstants;

import java.io.Serializable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

/**
 * Builds a panel that contains several widgets to build a form. The state and the data of the form is kept in a {@link FormModel}. The main
 * goal was to provide an easy way to build a form dialog for a java bean, where you write just one line of code for each widget. To do this
 * create a subclass and overwrite initForm() to add widgets to the form.
 * 
 * <pre><code>
 * protected void initForm() {
 *     addTextField(&quot;title&quot;);
 *     addTextField(&quot;name&quot;);
 *     addTextArea(&quot;description&quot;).growVertical(true);
 *     addCheckBox(&quot;orderable&quot;, &quot;Availability&quot;).text(&quot;Orderable&quot;).notGrowing(true);
 * }
 * </code></pre>
 * 
 * The builder provides an <i>add</i>-method for all the widgets. The methods return a {@link LayoutParameter} object. The LayoutParameter
 * objects keep all the parameters that are needed to create the widget and to integrate it into the layout. Because there are many optional
 * parameters the LayoutParameter have setter methods that return the LayoutParameter again. This allows to set all parameters needed on one
 * line.
 * <p>
 * {@link ULCBoxPane} is used for laying out widgets. Each <i>add</i>-method puts a {@code ULCLabel} in cell 0 and the widget into cell 1 on
 * a new row. The widget will grow horizontally. Use the LayoutParameter to change this standard behavior.
 * <ul>
 * <li>noLabel(true): does not add a label, the widget is placed in cell 0.</li>
 * <li>notGrowing(true): the widget does not grow horizontally.</li>
 * <li>startColumn(int): the Label is placed in cell <i>startColumn</i>, the widget in cell <i>startColumn + 1</i>
 * </ul>
 * The form may be split up in several tabs by calling {@code startTab()} with the title for the tab.
 * <p>
 * Besides the {@link LayoutParameter} there are two other classes that are involved in the process:
 * <ul>
 * <li> {@link WidgetFactory}: Creates instances of the widgets to be added to the form. Subclasses can create a own WidgetFactory to have
 * the form built with subclasses of the standard ULC widgets.</li>
 * <li> {@link WidgetBinderManager}: Holds a list of {@link IFormModelBinding}, attaches and detaches them to the current formModel.</li>
 * </ul>
 * <p>
 * The form builder works with a {@link ResourceMap} made of bundles for the builder class and the {@code FormModel} class. If nothing else
 * is specified the label for a property bound widget is created with the name set to <i>propertyName</i>.Label. The widgets have their name
 * set to <i>propertyName</i>.<i>WidgetType</i>, where WidgetType is one of TextField, TextArea, PasswordField, List, Slider, Spinner,
 * ComboBox, or CheckBox. <br>
 * <p>
 * The form builder creates a {@link ULCDefaultErrorManager} that manages feedback for errors.
 * <p>
 * To integrate the form into an application, use getFormPane() to get the form component. The builder also maintains two {@link IEnabler}s
 * (a hasChanged enabler and a mandatory enabler)to be used by the form's surrounding components.
 * <p>
 * 
 * @param <T> the generic parameter that defines the type of the bean whose properties will be mapped to the form
 * @see LayoutParameter
 * @see WidgetFactory
 * @see WidgetBinderManager
 */
public abstract class AbstractFormBuilder<T extends FormModel<?>> implements Serializable {
    
    /**
     * This class enables building of each part of a form.
     * <p>
     * When one of the <i>add</i>-methods is called a LayoutParameter is returned, which may be further configured. Therefore, the actual
     * building of the form takes place after initForm(). The methods that are to be used in initForm() to build the form add a {@code
     * FormBuilderPart} to the list that the {@code AbstractFormBuilder} holds. After {@code initForm()} is called, the {@code build()}
     * method of all the added parts is called in the sequence the parts are added.
     */
    protected abstract class FormBuilderPart implements Serializable {
        /**
         * This method is called after initForm() has finished.
         */
        protected abstract void build();
    }
    
    
    protected static final int DEFAULT_SPACE_BEFORE_TITLE = 10;
    protected static final int DEFAULT_SPACE_AFTER_TITLE = 5;
    private static final Integer DEFAULT_COLUMN_WEIGHT = new Integer(1);
    private static final Integer DEFAULT_LABEL_COLUMN_WEIGHT = new Integer(0);
    private static final int DEFAULT_SPACER_WIDTH = 50;
    private static final int HORIZONTAL_GAP = 3;
    private static final int VERTICAL_GAP = 3;
    private T fModel;
    private ULCTabbedPane fTabbedPane;
    private ULCBoxPane fFormPane;
    private ULCHasChangedEnabler fDirtyStateEnabler;
    private ULCMandatoryAndEnabler fMandatoryEnabler;
    private int fYPos;
    private WidgetStore fLastWidget;
    private float[] fColumnWeights;
    private List<Number> fDefaultWeights;
    private final WidgetBinderManager<T> fWidgetBinderManager;
    private final WidgetFactory fWidgetFactory;
    private int[] fColumnWidths;
    private int fDefaultEventDeliveryMode;
    private int fLabelHorizontalAlignment;
    private List<FormBuilderPart> fPart;
    private IRoundTripListener fRoundTripListener;
    private final ULCAbstractErrorManager fErrorManager;
    private ResourceMap fResourceMap;
    
    
    /**
     * Constructs a {@code FormBuilder} with the given {@code FormModel} and synchronous mode as default event delivery mode.
     * 
     * @param model {@code FormModel} that holds the data for the form
     */
    public AbstractFormBuilder(T model) {
        this(model, UlcEventConstants.SYNCHRONOUS_MODE);
    }
    
    /**
     * Constructs a {@code FormBuilder} with the given {@code FormModel} and the given default event delivery mode.
     * 
     * @param model {@code FormModel} that holds the data for the form
     * @param defaultEventDeliveryMode the event delivery mode to use.
     */
    public AbstractFormBuilder(T model, int defaultEventDeliveryMode) {
        fWidgetFactory = createWidgetFactory();
        fWidgetBinderManager = createWidgetBinder();
        doSetModel(model);
        fDirtyStateEnabler = new ULCHasChangedEnabler();
        fMandatoryEnabler = new ULCMandatoryAndEnabler();
        getMandatoryEnabler().setHighLightColor(Color.yellow);
        fDefaultEventDeliveryMode = defaultEventDeliveryMode;
        fTabbedPane = null;
        fDefaultWeights = new ArrayList<Number>();
        fColumnWeights = null;
        fColumnWidths = null;
        fFormPane = null;
        fLabelHorizontalAlignment = ULCLabel.LEADING;
        fPart = new ArrayList<FormBuilderPart>(25);
        fErrorManager = createErrorManager();
        fResourceMap = null;
    }
    
    /**
     * Subclasses must overwrite this to get the form filled with the widgets that are bound to the beans properties
     */
    protected abstract void initForm();
    
    /**
     * Creates a {@link ULCDefaultErrorManager} with the results of {@code createErrorMessages(), getErrorColor() and getErrorBorder()}.
     * Subclasses should overwrite this method to return their custom error manager.
     * 
     * @return the {@code ULCErrorManager} the form works with.
     */
    protected ULCAbstractErrorManager createErrorManager() {
        return new ULCDefaultErrorManager(createErrorMessages(), getErrorColor(), getErrorBorder());
    }
    
    /**
     * @return the {@code ULCErrorManager} the form works with.
     */
    public ULCAbstractErrorManager getErrorManager() {
        return fErrorManager;
    }
    
    /**
     * Creates a one line red border. Subclasses can overwrite this to create another border.
     * 
     * @return the border that is used to visualize the error state on a widget.
     */
    protected ULCAbstractBorder getErrorBorder() {
        return BorderFactory.createLineBorder(Color.red, 1);
    }
    
    /**
     * Returns the Color pink. Subclasses can overwrite this to return another color.
     * 
     * @return the color that is used as background color to visualize the error state on a widget.
     */
    protected Color getErrorColor() {
        return Color.pink;
    }
    
    /**
     * Creates and returns the {@code Map} that is used to produce the error messages shown to the user from an {@link ErrorObject}. The
     * keys must be the expected error codes and the values are the message format patterns that are fed to {@code MessageFormat.format}
     * together with the parameters from the {@code ErrorObject}. This implementation retrieves all resources from the {@code ResourceMap}
     * that start with {@code ErrorCodes.ERROR_CODE_PREFIX}.
     * 
     * @return a {@code Map} that contains all resources that start with {@code ErrorCodes.ERROR_CODE_PREFIX}.
     * @see ErrorObject
     * @see ErrorCodes
     * @see MessageFormat
     */
    protected HashMap<String, String> createErrorMessages() {
        
        HashMap<String, String> result = new HashMap<String, String>();
        ResourceMap resourceMap = getResourceMap();
        List<String> resourceKeys = resourceMap.getResourceKeys(ErrorCodes.ERROR_CODE_PREFIX);
        for (String errorCode : resourceKeys) {
            result.put(errorCode, resourceMap.getString(errorCode));
        }
        return result;
    }
    
    /**
     * Creates and returns the {@code WidgetBinderManager} that the Form works with. Should be overwritten when working with a own subclass
     * of {@code WidgetBinderManager}.
     * 
     * @return the WidgetBinderManager that the form is using.
     */
    protected WidgetBinderManager<T> createWidgetBinder() {
        return new WidgetBinderManager<T>();
    }
    
    /**
     * Creates and returns the {@code WidgetFactory} that the Form uses to create widgets. Should be overwritten when working with a own
     * subclass of {@code WidgetFactory}.
     * 
     * @return the WidgetFactory that the form is using.
     */
    protected WidgetFactory createWidgetFactory() {
        return new WidgetFactory();
    }
    
    /**
     * @return the {@code WidgetFactory} that the form is using.
     */
    protected WidgetFactory getWidgetFactory() {
        return fWidgetFactory;
    }
    
    /**
     * @return the default event delivery mode that is used.
     */
    public int getDefaultEventDeliveryMode() {
        return fDefaultEventDeliveryMode;
    }
    
    /**
     * This method determines the event delivery mode using the given {@code LayoutParameter} and the default event delivery mode.
     * 
     * @param layoutParameter that may define a event delivery mode for its widget.
     * @return the event delivery mode to be used for the layoutParameter's widget.
     */
    protected int getEventDeliveryMode(LayoutParameter<?, ?, ?> layoutParameter) {
        return layoutParameter.getEventDeliveryMode(getDefaultEventDeliveryMode());
    }
    
    /**
     * Changes the default event delivery mode for the widgets that are added after the call to this method.
     * 
     * @param eventDeliveryMode the new event delivery mode to use after this call.
     */
    public void setDefaultEventDeliveryMode(final int eventDeliveryMode) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                fDefaultEventDeliveryMode = eventDeliveryMode;
            }
        });
    }
    
    private ULCAbstractBorder createBorder(int top, int left, int buttom, int right) {
        return BorderFactory.createEmptyBorder(top, left, buttom, right);
    }
    
    /**
     * @return the panel that contains the form.
     */
    public ULCComponent getFormPane() {
        if (fFormPane == null) {
            doInitForm();
            ULCComponent pane = fTabbedPane == null ? fFormPane : fTabbedPane;
            getResourceMap().injectComponents(pane);
            return pane;
        } else {
            return fTabbedPane == null ? fFormPane : fTabbedPane;
        }
    }
    
    /**
     * @return the {@link WidgetBinderManager} the form works with.
     */
    protected WidgetBinderManager<T> getWidgetBinderManager() {
        return fWidgetBinderManager;
    }
    
    /**
     * reverts all changes on the model and resets the dirty state enabler.
     */
    public void reset() {
        try {
            getModel().undoAll();
            getDirtyStateEnabler().reset();
        } catch (PropertyAccessException e) {
            throw new RuntimeException(e);
        }
    }
    
    /**
     * Adds a {@code FormBuilderPart} to the list of {@code FormBuilderPart}s.
     * 
     * @param part the building block to be executed after {@code initForm()} to produce the form.
     */
    protected void addPart(FormBuilderPart part) {
        fPart.add(part);
    }
    
    /**
     * @return the {@code FormModel} that holds the state and the data for the form.
     */
    public T getModel() {
        return fModel;
    }
    
    /**
     * Creates a {@link ULCTextField}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property
     * @return the {@code TextFieldParameter} to configure the text field and its integration.
     */
    protected TextFieldParameter addTextField(String propertyName) {
        return addTextField(new TextFieldParameter(propertyName, getWidgetFactory()));
    }
    
    
    /**
     * Creates a {@link ULCTextField}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property
     * @param labelText text to be shown before the widget.
     * @return the {@code TextFieldParameter} to configure the text field and its integration.
     */
    protected TextFieldParameter addTextField(String propertyName, String labelText) {
        return addTextField(new TextFieldParameter(propertyName, getWidgetFactory()).labelText(labelText));
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCTextField} to the form using the given {@link TextFieldParameter}.
     * 
     * @param parameter {@code TextFieldParameter} that configures the text field and its integration.
     * @return the passed in {@code TextFieldParameter}.
     */
    protected TextFieldParameter addTextField(final TextFieldParameter parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                ULCTextField widget = createBoundTextField(parameter);
                ClientContext
                        .setEventDeliveryMode(widget, UlcEventCategories.VALUE_CHANGED_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                addComponentToView(widget, widget, parameter);
            }
            
        });
        return parameter;
    }
    
    /**
     * Creates a {@link ULCPasswordField}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property
     * @return the {@code PasswordFieldParameter} to configure the password field and its integration.
     */
    protected PasswordFieldParameter addPasswordField(String propertyName) {
        return addPasswordField(new PasswordFieldParameter(propertyName, getWidgetFactory()));
    }
    
    /**
     * Creates a {@link ULCPasswordField}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property
     * @param labelText text to be shown before the widget.
     * @return the {@code PasswordFieldParameter} to configure the password field and its integration.
     */
    protected PasswordFieldParameter addPasswordField(String propertyName, String labelText) {
        return addPasswordField(propertyName).labelText(labelText);
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCPasswordField} to the form using the given {@link PasswordFieldParameter}.
     * 
     * @param parameter {@code PasswordFieldParameter} that configures the password field and its integration.
     * @return the passed in {@code PasswordFieldParameter}.
     */
    protected PasswordFieldParameter addPasswordField(final PasswordFieldParameter parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                ULCPasswordField widget = parameter.getWidget();
                getWidgetBinderManager().bindTextField(parameter.getPropertyName(), widget, getErrorManager());
                ClientContext
                        .setEventDeliveryMode(widget, UlcEventCategories.VALUE_CHANGED_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                addComponentToView(widget, widget, parameter);
            }
            
        });
        return parameter;
    }
    
    /**
     * Creates a {@link ULCTextArea}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of type String.
     * @return the {@code TextAreaParameter} to configure the text area and its integration.
     */
    protected TextAreaParameter addTextArea(String propertyName) {
        return addTextArea(new TextAreaParameter(propertyName, getWidgetFactory()));
    }
    
    
    /**
     * Creates a {@link ULCTextArea}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of type String.
     * @param labelText text to be shown before the widget.
     * @return the {@code TextAreaParameter} to configure the text area and its integration.
     */
    protected TextAreaParameter addTextArea(String propertyName, String labelText) {
        return addTextArea(propertyName).labelText(labelText);
    }
    
    
    /**
     * Creates a {@link ULCTextArea}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of type String.
     * @param labelText text to be shown before the widget.
     * @param rows the number of rows of the text area.
     * @param columns the number of columns of the text area.
     * @return the {@code TextAreaParameter} to configure the text area and its integration.
     */
    protected TextAreaParameter addTextArea(String propertyName, String labelText, int rows, int columns) {
        return addTextArea(propertyName).labelText(labelText).rows(rows).columns(columns);
    }
    
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCTextArea} to the form using the given {@link TextAreaParameter}.
     * 
     * @param parameter {@code TextAreaParameter} that configures the text area and its integration.
     * @return the passed in {@code TextAreaParameter}.
     */
    protected TextAreaParameter addTextArea(final TextAreaParameter parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                getWidgetBinderManager().bindTextArea(parameter.getPropertyName(), parameter.getWidget());
                ULCTextArea widget = parameter.getWidget();
                ClientContext
                        .setEventDeliveryMode(widget, UlcEventCategories.VALUE_CHANGED_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                addComponentToView(new ULCScrollPane(widget), widget, parameter);
            }
        });
        return parameter;
    }
    
    /**
     * Creates a {@link ULCList} with the given model, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property.
     * @param model the {@link IListModel} that defines the list.
     * @return the {@code ListParameter} to configure the list and its integration.
     */
    protected ListParameter addList(String propertyName, IListModel model) {
        return addList(new ListParameter(propertyName, model, getWidgetFactory()));
    }
    
    /**
     * Creates a {@link ULCList} with a {@link DefaultListModel} using the given values, binds it to the given property and adds it to the
     * form.
     * 
     * @param propertyName name of the property.
     * @param values that define the list.
     * @return the {@code ListParameter} to configure the list and its integration.
     */
    protected ListParameter addList(String propertyName, Object[] values) {
        return addList(propertyName, new DefaultListModel(values));
    }
    
    /**
     * Creates a {@link ULCList} with a {@link DefaultListModel} using the given values, binds it to the given property and adds it to the
     * form.
     * 
     * @param propertyName name of the property.
     * @param values that define the list.
     * @return the {@code ListParameter} to configure the list and its integration.
     */
    protected ListParameter addList(String propertyName, List<?> values) {
        return addList(propertyName, new DefaultListModel(values.toArray()));
    }
    
    /**
     * Creates a {@link ULCList} with the given model, binds it to the given property and adds it to the form. The given label is placed
     * before the list.
     * 
     * @param propertyName name of the property.
     * @param labelText the text on the label before the widget
     * @param model the {@code IListModel} for the list
     * @return the {@code ListParameter} to configure the list and its integration.
     */
    protected ListParameter addList(String propertyName, String labelText, IListModel model) {
        return addList(new ListParameter(propertyName, model, getWidgetFactory())).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCList} with a {@link DefaultListModel} using the given values, binds it to the given property and adds it to the
     * form. The given label is placed before the field.
     * 
     * @param propertyName name of the property.
     * @param labelText the text on the label before the widget.
     * @param values that define the list.
     * @return the {@code ListParameter} to configure the list and its integration.
     */
    protected ListParameter addList(String propertyName, String labelText, Object[] values) {
        return addList(propertyName, values).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCList} with a {@link DefaultListModel} using the given values, binds it to the given property. The given label is
     * placed before the field. and adds it to the form.
     * 
     * @param propertyName name of the property.
     * @param labelText the text on the label for the widget
     * @param values that define the list.
     * @return the {@code ListParameter} to configure the list and its integration.
     */
    protected ListParameter addList(String propertyName, String labelText, List<?> values) {
        return addList(propertyName, values).labelText(labelText);
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCList} to the form using the given {@link ListParameter}.
     * 
     * @param parameter {@code ListParameter} that configures the list and its integration.
     * @return the passed in {@code ListParameter}.
     */
    protected ListParameter addList(final ListParameter parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                ULCList widget = parameter.getWidget();
                getWidgetBinderManager().bindList(parameter.getPropertyName(), widget);
                ClientContext.setEventDeliveryMode(widget, UlcEventCategories.ACTION_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                addComponentToView(new ULCScrollPane(widget), widget.getSelectionModel(), parameter);
            }
            
        });
        return parameter;
    }
    
    /**
     * Creates a {@link ULCSlider}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of a number type.
     * @return the {@code SliderParameter} to configure the slider and its integration.
     */
    protected SliderParameter addSlider(String propertyName) {
        return addSlider(new SliderParameter(propertyName, getWidgetFactory()));
    }
    
    
    /**
     * Creates a {@link ULCSlider}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of a number type.
     * @param labelText text to be shown before the widget.
     * @return the {@code SliderParameter} to configure the slider and its integration.
     */
    protected SliderParameter addSlider(String propertyName, String labelText) {
        return addSlider(propertyName).labelText(labelText);
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCSlider} to the form using the given {@link SliderParameter}.
     * 
     * @param parameter {@code SliderParameter} that configures the slider and its integration.
     * @return the passed in {@code SliderParameter}.
     */
    protected SliderParameter addSlider(final SliderParameter parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                ULCSlider widget = parameter.getWidget();
                getWidgetBinderManager().bindSlider(parameter.getPropertyName(), widget);
                ClientContext
                        .setEventDeliveryMode(widget, UlcEventCategories.VALUE_CHANGED_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                addComponentToView(widget, widget, parameter);
            }
        });
        return parameter;
    }
    
    /**
     * Creates a {@link ULCSpinner}, binds it to the given property and adds it to the form. For {@link Date} type properties a
     * {@link DateSpinnerParameter} is returned, for numeric properties a {@link NumericSpinnerParameter}.
     * 
     * @param propertyName name of the property. The property must be of a number type or {@link Date} type.
     * @return the {@code SpinnerParameter} to configure the spinner and its integration.
     * @throws IllegalArgumentException if the property's type is neither {@link Date} nor any numeric type.
     */
    protected LayoutParameter<?, ULCSpinner, ?> addSpinner(String propertyName) {
        Class<?> propertyType = getModel().getPropertyType(propertyName);
        if (Date.class.isAssignableFrom(propertyType)) {
            return addSpinner(new DateSpinnerParameter(propertyName, getWidgetFactory()));
        } else if (isIntegerNumberType(propertyType) || isFloatNumberType(propertyType)) {
            return addSpinner(new NumericSpinnerParameter(propertyName, getWidgetFactory()));
        }
        throw new IllegalArgumentException("Cannot instantiate a Spinner for the given property of type " + propertyType);
    }
    
    /**
     * Creates a {@link ULCSpinner}, binds it to the given property and adds it to the form. For {@link Date} type properties a
     * {@link DateSpinnerParameter} is returned, for numeric properties a {@link NumericSpinnerParameter}.
     * 
     * @param propertyName name of the property. The property must be of a number type or {@link Date} type.
     * @param labelText text to be shown before the widget.
     * @return the {@code SpinnerParameter} to configure the spinner and its integration.
     * @throws IllegalArgumentException if the property's type is neither {@link Date} nor any numeric type.
     */
    protected LayoutParameter<?, ULCSpinner, ?> addSpinner(String propertyName, String labelText) {
        return addSpinner(propertyName).labelText(labelText);
    }
    
    
    /**
     * Creates a {@link ULCSpinner} with a list model, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property.
     * @param items the list that provides the values for the spinner.
     * @return the {@code ListSpinnerParameter} to configure the spinner and its integration.
     */
    protected ListSpinnerParameter addListSpinner(String propertyName, List<?> items) {
        return addSpinner(new ListSpinnerParameter(propertyName, getWidgetFactory(), items));
    }
    
    /**
     * Creates a {@link ULCSpinner} with a list model, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property.
     * @param labelText text to be shown before the widget.
     * @param items the list that provides the values for the spinner.
     * @return the {@code ListSpinnerParameter} to configure the spinner and its integration.
     */
    protected ListSpinnerParameter addListSpinner(String propertyName, String labelText, List<?> items) {
        return addListSpinner(propertyName, items).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCSpinner} with a list model, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property.
     * @param items the list that provides the values for the spinner.
     * @return the {@code ListSpinnerParameter} to configure the spinner and its integration.
     */
    protected ListSpinnerParameter addListSpinner(String propertyName, Object[] items) {
        return addSpinner(new ListSpinnerParameter(propertyName, getWidgetFactory(), items));
    }
    
    /**
     * Creates a {@link ULCSpinner} with a list model, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property.
     * @param labelText text to be shown before the widget.
     * @param items the list that provides the values for the spinner.
     * @return the {@code ListSpinnerParameter} to configure the spinner and its integration.
     */
    protected ListSpinnerParameter addListSpinner(String propertyName, String labelText, Object[] items) {
        return addListSpinner(propertyName, items).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCSpinner} with a {@link ULCSpinnerNumberModel}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of a number type.
     * @return the {@code NumericSpinnerParameter} to configure the spinner and its integration.
     */
    protected NumericSpinnerParameter addNumericSpinner(String propertyName) {
        return addSpinner(new NumericSpinnerParameter(propertyName, getWidgetFactory()));
    }
    
    /**
     * Creates a {@link ULCSpinner} with a {@link ULCSpinnerNumberModel}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of a number type.
     * @param labelText text to be shown before the widget.
     * @return the {@code NumericSpinnerParameter} to configure the spinner and its integration.
     */
    protected NumericSpinnerParameter addNumericSpinner(String propertyName, String labelText) {
        return addNumericSpinner(propertyName).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCSpinner} with a {@link ULCSpinnerDateModel}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of {@link Date} type.
     * @param labelText text to be shown before the widget.
     * @return the {@code DateSpinnerParameter} to configure the spinner and its integration.
     */
    protected DateSpinnerParameter addDateSpinner(String propertyName, String labelText) {
        return addDateSpinner(propertyName).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCSpinner} with a {@link ULCSpinnerDateModel}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of {@link Date} type.
     * @return the {@code DateSpinnerParameter} to configure the spinner and its integration.
     */
    protected DateSpinnerParameter addDateSpinner(String propertyName) {
        return addSpinner(new DateSpinnerParameter(propertyName, getWidgetFactory()));
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCSpinner} to the form using the given SpinnerParameter.
     * <p>
     * 
     * @param <LP> The generic type that defines the type of the LayoutParameter
     * @param parameter {@link LayoutParameter} for a ULCSpinner that configures the spinner and its integration.
     * @return the passed in {@link LayoutParameter}.
     */
    protected <LP extends LayoutParameter<?, ULCSpinner, ?>> LP addSpinner(final LP parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                ULCSpinner widget = parameter.getWidget();
                getWidgetBinderManager().bindSpinner(parameter.getPropertyName(), widget);
                addSpinnerToView(parameter, widget);
            }
        });
        return parameter;
    }
    
    private void addSpinnerToView(LayoutParameter<?, ULCSpinner, ?> layoutInfo, ULCSpinner widget) {
        ISpinnerModel model = widget.getModel();
        if (model instanceof ULCProxy) {
            ULCProxy modelProxy = (ULCProxy)model;
            ClientContext.setEventDeliveryMode(modelProxy, UlcEventCategories.VALUE_CHANGED_EVENT_CATEGORY,
                    getEventDeliveryMode(layoutInfo));
        }
        addComponentToView(widget, model, layoutInfo);
    }
    
    
    /**
     * Creates a {@link ULCComboBox}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of the same type as the items elements.
     * @param model the {@link IComboBoxModel} that defines the combobox's entries.
     * @return the {@code ComboboxParameter} to configure the combobox and its integration.
     */
    protected ComboboxParameter addCombobox(String propertyName, IComboBoxModel model) {
        return addCombobox(new ComboboxParameter(propertyName, model, getWidgetFactory()));
    }
    
    /**
     * Creates a {@link ULCComboBox} with a {@link DefaultComboBoxModel} using the given items, binds it to the given property and adds it
     * to the form.
     * 
     * @param propertyName name of the property. The property must be of the same type as the items elements.
     * @param items defining the combobox's entries.
     * @return the {@code ComboboxParameter} to configure the combobox and its integration.
     */
    protected ComboboxParameter addCombobox(String propertyName, List<?> items) {
        return addCombobox(propertyName, new DefaultComboBoxModel(items));
    }
    
    /**
     * Creates a {@link ULCComboBox} with a {@link DefaultComboBoxModel} using the given items, binds it to the given property and adds it
     * to the form.
     * 
     * @param propertyName name of the property. The property must be of the same type as the items elements.
     * @param items defining the combobox's entries.
     * @return the {@code ComboboxParameter} to configure the combobox and its integration.
     */
    protected ComboboxParameter addCombobox(String propertyName, Object[] items) {
        return addCombobox(propertyName, new DefaultComboBoxModel(items));
    }
    
    /**
     * Creates a {@link ULCComboBox}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property.
     * @param labelText text to be shown before the widget.
     * @param model the {@link IComboBoxModel} that defines the combobox's entries.
     * @return the created {@code ComboboxParameter}
     */
    protected ComboboxParameter addCombobox(String propertyName, String labelText, IComboBoxModel model) {
        return addCombobox(propertyName, model).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCComboBox} with a {@link DefaultComboBoxModel} using the given items, binds it to the given property and adds it
     * to the form.
     * 
     * @param propertyName name of the property. The property must be of the same type as the items elements.
     * @param labelText text to be shown before the widget.
     * @param items defining the combobox's entries.
     * @return the {@code ComboboxParameter} to configure the combobox and its integration.
     */
    protected ComboboxParameter addCombobox(String propertyName, String labelText, List<?> items) {
        return addCombobox(propertyName, items).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCComboBox} with a {@link DefaultComboBoxModel} using the given items, binds it to the given property and adds it
     * to the form.
     * 
     * @param propertyName name of the property. The property must be of the same type as the items elements.
     * @param labelText text to be shown before the widget.
     * @param items defining the combobox's entries.
     * @return the {@code ComboboxParameter} to configure the combobox and its integration.
     */
    protected ComboboxParameter addCombobox(String propertyName, String labelText, Object[] items) {
        return addCombobox(propertyName, items).labelText(labelText);
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCComboBox} to the form using the given {@link ComboboxParameter}.
     * 
     * @param parameter {@link ComboboxParameter} that configures the combobox and its integration.
     * @return the passed in {@code ComboboxParameter}.
     */
    protected ComboboxParameter addCombobox(final ComboboxParameter parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                ULCComboBox widget = parameter.getWidget();
                getWidgetBinderManager().bindComboBox(parameter.getPropertyName(), widget);
                ClientContext.setEventDeliveryMode(widget, UlcEventCategories.ACTION_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                addComponentToView(widget, widget.getModel(), parameter);
            }
        });
        return parameter;
    }
    
    /**
     * Creates a {@link ULCCheckBox}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of a boolean type.
     * @return the {@code CheckboxParameter} to configure the checkbox and its integration.
     */
    protected CheckboxParameter addCheckBox(String propertyName) {
        return addCheckBox(new CheckboxParameter(propertyName, getWidgetFactory()));
    }
    
    /**
     * Creates a {@link ULCCheckBox}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of a boolean type.
     * @param labelText text to be shown before the widget.
     * @return the {@code CheckboxParameter} to configure the checkbox and its integration.
     */
    protected CheckboxParameter addCheckBox(String propertyName, String labelText) {
        return addCheckBox(propertyName).labelText(labelText);
    }
    
    /**
     * Creates a {@link ULCCheckBox}, binds it to the given property and adds it to the form.
     * 
     * @param propertyName name of the property. The property must be of a boolean type.
     * @param labelText text to be shown before the widget.
     * @param textAfter text to be shown directly after the checkbox
     * @return the {@code CheckboxParameter} to configure the checkbox and its integration.
     */
    protected CheckboxParameter addCheckBox(String propertyName, String labelText, String textAfter) {
        return addCheckBox(propertyName).labelText(labelText).text(textAfter);
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a {@link ULCCheckBox} to the form using the given {@link CheckboxParameter}.
     * 
     * @param parameter {@code CheckboxParameter} that configures the checkbox and its integration.
     * @return the passed in {@code CheckboxParameter}.
     */
    protected CheckboxParameter addCheckBox(final CheckboxParameter parameter) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                ULCCheckBox checkBox = parameter.getWidget();
                getWidgetBinderManager().bindCheckBox(parameter.getPropertyName(), checkBox, true);
                checkBox.setAlignmentX(ULCCheckBox.LEFT_ALIGNMENT);
                ClientContext.setEventDeliveryMode(checkBox, UlcEventCategories.ACTION_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                addComponentToView(checkBox, checkBox, parameter);
            }
        });
        return parameter;
        
    }
    
    /**
     * Creates a a panel with {@link ULCRadioButton}<code>s</code>, one for each entry in values, binds it to the given property and adds it
     * to the form. Each radio button repesents a value that can be set on the property.
     * 
     * @param propertyName name of the property. The property must be of the same type as the values.
     * @param labelText text to be shown before the widget.
     * @param labels the labels that are written after the radio buttons
     * @param values the values that are set on the property
     * @return the {@link RadioButtonParameter} that holds the parameters to create and integrate the radio buttons.
     */
    protected RadioButtonParameter addRadioButtons(String propertyName, String labelText, String[] labels, Object[] values) {
        return addRadioButtons(new RadioButtonParameter(propertyName, labels, values, getWidgetFactory()).labelText(labelText));
        
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds the radio buttons defined by the given {@link RadioButtonParameter} to the form.
     * 
     * @param parameter the {@code RadioButtonParameter} that are defining the radio buttons to be added.
     * @return the {@code RadioButtonParameter} passed in.
     */
    protected RadioButtonParameter addRadioButtons(final RadioButtonParameter parameter) {
        addPart(new FormBuilderPart() {
            @Override
            protected void build() {
                ULCRadioButton[] radios = parameter.getRadioButtons();
                Object[] values = parameter.getValues();
                ULCRadioButton radio = null;
                for (int i = 0; i < radios.length; i++) {
                    radio = radios[i];
                    getWidgetBinderManager().bindRadioButton(parameter.getPropertyName(), radio, values[i]);
                    ClientContext.setEventDeliveryMode(radio, UlcEventCategories.ACTION_EVENT_CATEGORY, getEventDeliveryMode(parameter));
                }
                addComponentToView(parameter.getWidget(), radio, parameter);
            }
        });
        return parameter;
    }
    
    /**
     * Add's a {@code FormBuilderPart} that adds a widget to the form. The widget is not bound to the formModel.
     * 
     * @param widget to be added
     * @return the {@link GeneralParameter} that holds the parameters to integrate the widget.
     */
    protected GeneralParameter addComponentToView(final ULCComponent widget) {
        final GeneralParameter generalParameter = new GeneralParameter(widget, getWidgetFactory());
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                addComponentToView(widget, null, generalParameter);
            }
        });
        return generalParameter;
        
    }
    
    /**
     * Adds a widget to the form. A label with the given text is placed before the widget. The widget is not bound to the formModel.
     * 
     * @param labelText text to be shown before the widget.
     * @param widget to be added.
     * @return the {@link GeneralParameter} that holds the parameters to integrate the widget.
     */
    protected GeneralParameter addComponentToView(String labelText, final ULCComponent widget) {
        return addComponentToView(widget).labelText(labelText);
    }
    
    
    /**
     * Adds a text to the form.
     * 
     * @param text to be put on the form.
     * @return the {@link GeneralParameter} that holds the parameters to integrate the text.
     */
    protected LabelParameter addLabel(String text) {
        final LabelParameter labelParameter = new LabelParameter(text, getWidgetFactory());
        labelParameter.widgetName(text + ".Label");
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                if (!labelParameter.isHorizontalAlignmentSet()) {
                    labelParameter.horizontalAlignment(getLabelHorizontalAlignment());
                }
                addComponentToView(labelParameter.getWidget(), null, labelParameter);
            }
        });
        return labelParameter;
    }
    
    /**
     * Adds whitespace at the given column.
     * 
     * @param startColumn the layout column where the whitespace starts.
     * @return the {@link GeneralParameter} that holds the parameters to integrate the whitspace.
     */
    protected GeneralParameter addSpacer(int startColumn) {
        return addSpacer(startColumn, DEFAULT_SPACER_WIDTH);
    }
    
    
    /**
     * Adds whitespace of given width at the given column.
     * 
     * @param startColumn the layout column where the whitespace starts.
     * @param width of the whitespace in pixel.
     * @return the {@link GeneralParameter} that holds the parameters to integrate the spacer.
     */
    protected GeneralParameter addSpacer(int startColumn, int width) {
        return addComponentToView(ULCFiller.createHorizontalStrut(width)).startColumn(startColumn);
    }
    
    /**
     * Adds a title text to the form with the values {@code FormModel.DEFAULT_SPACE_BEFORE_TITLE, FormModel.DEFAULT_SPACE_AFTER_TITLE}.
     * 
     * @param titleText the text of the title.
     * @return the parameter object of the label.
     */
    protected LabelParameter addTitle(String titleText) {
        return addTitle(titleText, DEFAULT_SPACE_BEFORE_TITLE, DEFAULT_SPACE_AFTER_TITLE);
    }
    
    /**
     * Adds a title text with inset to the form.
     * 
     * @param titleText the text of the title.
     * @param spaceBefore number of pixels inserted before the title.
     * @param spaceAfter number of pixels inserted after the title.
     * @return the parameter object of the label.
     */
    protected LabelParameter addTitle(String titleText, int spaceBefore, int spaceAfter) {
        addComponentToView(ULCFiller.createVerticalStrut(spaceBefore));
        LabelParameter labelParameter = addLabel(titleText).horizontalAlignment(ULCLabel.LEADING);
        labelParameter.notGrowing(true);
        addComponentToView(new ULCSeparator(ULCSeparator.HORIZONTAL)).append(true);
        addComponentToView(ULCFiller.createVerticalStrut(spaceAfter));
        return labelParameter;
    }
    
    /**
     * Adds a widget to the form. This method is to be called from within the build() method of a {@code FormBuilderPart}.
     * 
     * @param widget {@code ULCComponent} to be added
     * @param changedSource if not null, the source is added to the {@link ULCHasChangedEnabler} of the save and reset button.
     * @param layoutInfo {@code LayoutParameter} that holds the parameters that defines the integration into the layout.
     * @return the {@code ULCComponent} containing the label and the widget that is added to the form
     */
    protected ULCComponent addComponentToView(final ULCComponent widget, IHasChangedSource changedSource,
            LayoutParameter<?, ?, ?> layoutInfo) {
        String propertyName = layoutInfo.getPropertyName();
        if (widget == null) {
            throw new IllegalArgumentException("Widget must not be null");
        }
        
        if (layoutInfo.shouldStartRow()) {
            finishCurrentRow();
        }
        int column = layoutInfo.getStartColumn();
        final ULCLabel label = layoutInfo.getLabel();
        
        if (label != null) {
            label.setHorizontalAlignment(getLabelHorizontalAlignment());
            setDefaultColumnWeight(column, DEFAULT_LABEL_COLUMN_WEIGHT);
            addWidget(label, column, -1, layoutInfo.isGrowingHorizontally(), false, layoutInfo.isAppend());
            column++;
        }
        
        widget.setAlignmentX(1);
        addWidget(widget, column, layoutInfo.getEndColumn(), layoutInfo.isGrowingHorizontally(), layoutInfo.isGrowingVertically(),
                layoutInfo.isAppend());
        if (changedSource != null) {
            getDirtyStateEnabler().add(changedSource);
        }
        
        if (isDefined(propertyName)) {
            fWidgetBinderManager.bindErrorStateListener(propertyName, new IViewUpdater() {
                
                public void updateView(Object newValue) {
                    if (newValue instanceof ErrorObject) {
                        ErrorObject error = (ErrorObject)newValue;
                        getErrorManager().showError(widget, error);
                    } else if (newValue == null) {
                        getErrorManager().showError(widget, null);
                    } else {
                        throw new IllegalArgumentException("value is not null and not an instance of BeanError");
                    }
                }
                
            });
            if (widget.getName() == null) {
                widget.setName(propertyName + "." + toWidgetName(widget));
            }
            
            if (layoutInfo.isReadonly()) {
                setWidgetReadonly(widget, false);
            } else {
                fWidgetBinderManager.bindReadonlyStateListener(propertyName, new IViewUpdater() {
                    
                    public void updateView(Object value) {
                        boolean isReadonly = Boolean.TRUE.equals(value);
                        boolean editable = !isReadonly;
                        setWidgetReadonly(widget, editable);
                    }
                    
                });
                fWidgetBinderManager.bindMandatoryStateListener(propertyName, new IViewUpdater() {
                    
                    public void updateView(Object value) {
                        boolean isMandatory = Boolean.TRUE.equals(value);
                        if (isMandatory) {
                            if (widget instanceof IEnabler) {
                                getMandatoryEnabler().remove((IEnabler)widget);
                                getMandatoryEnabler().add((IEnabler)widget);
                            } else {
                                // widget.setBackground(Color.yellow);
                            }
                        } else {
                            if (widget instanceof IEnabler) {
                                getMandatoryEnabler().remove((IEnabler)widget);
                            } else {
                                // widget.setBackground(Color.yellow);
                            }
                        }
                    }
                    
                });
                fWidgetBinderManager.bindEnabledStateListener(propertyName, new IViewUpdater() {
                    
                    public void updateView(Object value) {
                        boolean isEnabled = Boolean.TRUE.equals(value);
                        widget.setEnabled(isEnabled);
                        if (label != null) {
                            label.setEnabled(isEnabled);
                        }
                        
                    }
                    
                });
            }
        }
        
        return widget;
    }
    
    private String toWidgetName(final ULCComponent widget) {
        String simpleClassName = widget.getClass().getSimpleName();
        return simpleClassName.startsWith("ULC") ? simpleClassName.substring(3) : simpleClassName;
    }
    
    /**
     * @return the horizontal alignment for the labels.
     */
    protected int getLabelHorizontalAlignment() {
        return fLabelHorizontalAlignment;
    }
    
    /**
     * Add's a {@code FormBuilderPart} that sets the alignment of the label's contents along the X axis.
     * <p>
     * The alignment can be one of the following:
     * <ul>
     * <li><code>ULCLabel.LEFT</code>,
     * <li><code>ULCLabel.CENTER</code>,
     * <li><code>ULCLabel.RIGHT</code>,
     * <li><code>ULCLabel.LEADING</code> or
     * <li><code>ULCLabel.TRAILING</code>.
     * </ul>
     * 
     * @param labelAlignement the horizontal alignment
     * @see ULCLabel#setHorizontalAlignment
     */
    protected void setLabelHorizontalAlignment(final int labelAlignement) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                fLabelHorizontalAlignment = labelAlignement;
            }
        });
    }
    
    /**
     * Add's a {@code FormBuilderPart} that creates a tab with the given title. Use the {@code add...} methods to add widgets to the tab.
     * 
     * @param title of the tab.
     */
    protected void startTab(final String title) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                if (fTabbedPane == null) {
                    fTabbedPane = new ULCTabbedPane();
                    fTabbedPane.setBorder(createBorder(5, 10, 0, 10));
                    addFormPaneToTab(title);
                } else {
                    finishCurrentAndCreateNewFormPane();
                    addFormPaneToTab(title);
                }
            }
        });
    }
    
    private void finishCurrentRow() {
        addWidget(null, 0, -1, false, false, false);
        fYPos++;
    }
    
    private void finishCurrentAndCreateNewFormPane() {
        finishCurrentPane();
        
        fFormPane = createFormPane();
    }
    
    private void finishCurrentPane() {
        finishCurrentRow();
        if (fColumnWeights != null) {
            for (int i = 0; i < fColumnWeights.length; i++) {
                
                setColumnWeight(i, fColumnWeights[i], getColumnWidth(i));
            }
        } else {
            for (int i = 0; i < fDefaultWeights.size(); i++) {
                setColumnWeight(i, (fDefaultWeights.get(i)).floatValue(), getColumnWidth(i));
            }
        }
        fFormPane.set(0, fYPos, 0, 1, 0, 0.0001, ULCBoxPane.BOX_CENTER_EXPAND, ULCFiller.createHorizontalGlue());
    }
    
    private int getColumnWidth(int i) {
        int width = 0;
        if (fColumnWidths != null && i < fColumnWidths.length) {
            width = fColumnWidths[i];
        }
        return width;
    }
    
    private void setColumnWeight(int i, float weight, int width) {
        fFormPane.set(i, 0, weight, 0.0, ULCBoxPane.BOX_CENTER_CENTER, ULCFiller.createHorizontalStrut(width));
    }
    
    private ULCBoxPane createFormPane() {
        fYPos = 0;
        fDefaultWeights.clear();
        ULCBoxPane boxPane = new ULCBoxPane(0, 0, HORIZONTAL_GAP, VERTICAL_GAP);
        return boxPane;
    }
    
    private static class WidgetStore implements Serializable {
        
        private ULCComponent fWidget;
        private final int fColumn;
        private final int fRow;
        private boolean fIsGrowingHorizontally;
        private int fNextStartColumn;
        private boolean fIsGrowingVertically;
        private boolean fAppended;
        
        private WidgetStore(ULCComponent widget, int column, int nextStartColumn, int row, boolean isGrowing, boolean isGrowingVertically) {
            fWidget = widget;
            fColumn = column;
            fNextStartColumn = nextStartColumn;
            fRow = row;
            fIsGrowingHorizontally = isGrowing;
            fIsGrowingVertically = isGrowingVertically;
            fAppended = false;
        }
        
        private ULCComponent getWidget() {
            return fWidget;
        }
        
        private int getColumn() {
            return fColumn;
        }
        
        private int getRow() {
            return fRow;
        }
        
        private boolean isGrowingVertically() {
            return fIsGrowingVertically;
        }
        
        private boolean isGrowingHorizontally() {
            return fIsGrowingHorizontally;
        }
        
        private void setLastColumn(int nextStartColumn) {
            fNextStartColumn = nextStartColumn;
        }
        
        private int getColumnSpan() {
            int span = fNextStartColumn - fColumn;
            return span < 0 ? 0 : span;
        }
        
        private String getAlingment() {
            return alignment(isGrowingHorizontally(), isGrowingVertically());
        }
        
        private String alignment(boolean growingHorizontally, boolean growingVertically) {
            return (growingHorizontally ? "e" : "l") + (growingVertically ? "e" : "c");
        }
        
        private void append(ULCComponent anotherWidget, boolean isGrowingHorizontally, boolean isGrowingVertically) {
            if (!fAppended) {
                ULCBoxPane widgetBox = new ULCBoxPane(false);
                widgetBox.add(alignment(isGrowingHorizontally(), isGrowingVertically()), fWidget);
                fIsGrowingHorizontally = true;
                fIsGrowingVertically = false;
                fWidget = widgetBox;
                fAppended = true;
            }
            ((ULCBoxPane)fWidget).add(alignment(isGrowingHorizontally, isGrowingVertically), anotherWidget);
        }
        

    }
    
    private void addWidget(final ULCComponent widget, int startColumn, int endColumn, boolean isGrowingHorizontally,
            boolean isGrowingVertically, boolean append) {
        if (fDefaultWeights.size() < startColumn + 1) {
            setDefaultColumnWeight(startColumn, DEFAULT_COLUMN_WEIGHT);
        }
        if (fLastWidget != null) {
            if (append && widget != null) {
                fLastWidget.append(widget, isGrowingHorizontally, isGrowingVertically);
                return;
            } else {
                if (fLastWidget.getColumnSpan() == 0) {
                    fLastWidget.setLastColumn(startColumn);
                }
                doAddWidget(fLastWidget);
            }
        }
        if (widget != null) {
            fLastWidget = new WidgetStore(widget, startColumn, endColumn, fYPos, isGrowingHorizontally, isGrowingVertically);
            
        } else {
            fLastWidget = null;
        }
        
    }
    
    private void doAddWidget(final WidgetStore widgetStore) {
        fFormPane.set(widgetStore.getColumn(), widgetStore.getRow(), widgetStore.getColumnSpan(), 1, 0,
                widgetStore.isGrowingVertically() ? 1 : 0, widgetStore.getAlingment(), widgetStore.getWidget());
    }
    
    private boolean isDefined(String aString) {
        return aString != null && aString.length() > 0;
    }
    
    private ULCTextField createBoundTextField(TextFieldParameter parameter) {
        String propertyName = parameter.getPropertyName();
        ULCTextField widget = parameter.getWidget();
        
        final IDataType dataType = parameter.getDataType();
        if (dataType == null) {
            try {
                Class<?> propertyType = getModel().getPropertyType(propertyName);
                if (isIntegerNumberType(propertyType)) {
                    ULCNumberDataType numberDataType = new ULCNumberDataType(getErrorManager());
                    numberDataType.setInteger(true);
                    widget.setDataType(numberDataType);
                    if (!parameter.isHorizontalAlignmentSet()) {
                        widget.setHorizontalAlignment(ULCTextField.TRAILING);
                    }
                } else if (isFloatNumberType(propertyType)) {
                    widget.setDataType(new ULCNumberDataType(getErrorManager()));
                    if (!parameter.isHorizontalAlignmentSet()) {
                        widget.setHorizontalAlignment(ULCTextField.TRAILING);
                    }
                } else if (propertyType.equals(Date.class)) {
                    widget.setDataType(new ULCDateDataType(getErrorManager()));
                }
                
            } catch (PropertyAccessException e) {
                throw new RuntimeException(e);
            }
        } else {
            if (!parameter.isHorizontalAlignmentSet() && dataType instanceof ULCNumberDataType || dataType instanceof ULCPercentDataType) {
                widget.setHorizontalAlignment(ULCTextField.TRAILING);
            }
        }
        
        getWidgetBinderManager().bindTextField(propertyName, widget, getErrorManager());
        return widget;
    }
    
    private boolean isFloatNumberType(Class<?> propertyType) {
        return propertyType.equals(Double.TYPE) || propertyType.equals(Double.class) || propertyType.equals(Float.TYPE)
                || propertyType.equals(Float.class) || propertyType.equals(BigDecimal.class);
    }
    
    private boolean isIntegerNumberType(Class<?> propertyType) {
        return propertyType.equals(Integer.TYPE) || propertyType.equals(Short.TYPE) || propertyType.equals(Byte.TYPE)
                || propertyType.equals(Long.TYPE) || propertyType.equals(Integer.class) || propertyType.equals(Short.class)
                || propertyType.equals(Byte.class) || propertyType.equals(Long.class) || propertyType.equals(BigInteger.class);
    }
    
    private void addFormPaneToTab(String title) {
        String resourceKeyPrefix = title + ".Tab.";
        final String titleResource = getResourceMap().getString(resourceKeyPrefix + "text", null);
        if (titleResource != null) {
            title = titleResource;
        }
        fTabbedPane.addTab(title, getResourceMap().getULCIcon(resourceKeyPrefix + "icon"), fFormPane, getResourceMap().getString(
                resourceKeyPrefix + "tip", null));
    }
    
    private void doSetModel(T model) {
        fWidgetBinderManager.setModel(model);
        fModel = model;
    }
    
    private void doInitForm() {
        fFormPane = createFormPane();
        fFormPane.setBorder(createBorder(5, 10, 15, 10));
        initForm();
        for (FormBuilderPart part : fPart) {
            part.build();
        }
        finishCurrentPane();
        getModel().updatePresentationState();
        fRoundTripListener = new IRoundTripListener() {
            
            public void roundTripWillEnd(RoundTripEvent event) {
                getWidgetBinderManager().updateFromModel();
            }
            
            public void roundTripDidStart(RoundTripEvent event) {
            }
            
        };
        ULCSession.currentSession().addRoundTripListener(fRoundTripListener);
        getErrorManager().upload();
    }
    
    /**
     * Removes the round-trip listener that the form uses to update the widgets at the end of each round-trip. Call this if the form is not
     * needed anymore to free all widgets.
     */
    public void dispose() {
        if (fRoundTripListener != null) {
            ULCSession.currentSession().removeRoundTripListener(fRoundTripListener);
            fRoundTripListener = null;
        }
    }
    
    /**
     * Replaces the {@link FormModel} of the form with the given FormModel
     * 
     * @param model the new {@code FormModel} the form works with. Must not be <code>null</code>.
     */
    public void replaceModel(final T model) {
        if (model == null) {
            throw new IllegalArgumentException("model must not be null");
        }
        doSetModel(model);
        getModel().updatePresentationState();
    }
    
    /**
     * Adds a {@code FormBuilderPart} that sets the column weights.
     * 
     * @param columnWeight for the first to nth column.
     */
    protected void setColumnWeights(final float... columnWeight) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                fColumnWeights = columnWeight;
            }
        });
    }
    
    /**
     * Adds a {@code FormBuilderPart} that sets the column widths.
     * 
     * @param columnWidths for the first to nth column.
     */
    protected void setColumnWidths(final int... columnWidths) {
        addPart(new FormBuilderPart() {
            
            @Override
            protected void build() {
                fColumnWidths = columnWidths;
            }
        });
    }
    
    private void setDefaultColumnWeight(int column, Integer weight) {
        while (fDefaultWeights.size() < column + 1) {
            fDefaultWeights.add(DEFAULT_COLUMN_WEIGHT);
        }
        fDefaultWeights.set(column, weight);
    }
    
    /**
     * @return the {@link ULCHasChangedEnabler} to which the form adds all the {@link IHasChangedSource}s for the widgets that are added to
     *         the form.
     */
    public ULCHasChangedEnabler getDirtyStateEnabler() {
        return fDirtyStateEnabler;
    }
    
    /**
     * @return the {@link ULCMandatoryAndEnabler} to which the form adds all the widgets, that are bound to a property that have the
     *         mandatory state set to true.
     */
    public ULCMandatoryAndEnabler getMandatoryEnabler() {
        return fMandatoryEnabler;
    }
    
    /**
     * Sets the widget's readonly or editable state. This implementation just sets the widget's editable property to <code>editable</code>.
     * 
     * @param widget to set the state.
     * @param editable if true the widget is made editable, otherwise it is made readonly.
     */
    protected void setWidgetReadonly(final ULCComponent widget, boolean editable) {
        if (widget instanceof ULCTextComponent) {
            ((ULCTextComponent)widget).setEditable(editable);
        } else {
            widget.setEnabled(editable);
        }
        widget.setFocusable(editable);
    }
    
    /**
     * @return the {@link ApplicationContext} that gives access to the resources and actions.
     */
    protected ApplicationContext getContext() {
        return Application.getInstance().getContext();
    }
    
    /**
     * The {@link ResourceMap} contains the bundles from the {@link AbstractFormBuilder} class hierarchy {@code
     * (MyConcreteFormBuilderSubclass.properties -> -> AbstractFormBuilder.properties)} followed by the bundles of the {@link FormModel}
     * class hierarchy (MyFormModelSubclass.properties -> -> FormModel.properties). At the end, as always, are the bundles of the
     * {@link Application} class hierarchy (MyApplicationSubclass.properties -> -> Application.properties).
     * 
     * @return the forms resource map.
     */
    protected ResourceMap getResourceMap() {
        if (fResourceMap == null) {
            ResourceMap modelMap = getContext().getResourceMap(getModel().getClass(), FormModel.class);
            fResourceMap = getContext().getResourceMap(getClass(), AbstractFormBuilder.class, modelMap);
        }
        
        return fResourceMap;
    }
}
