/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import com.ulcjava.applicationframework.application.form.model.FormModel;
import com.ulcjava.applicationframework.application.form.model.PropertyAccessException;
import com.ulcjava.base.application.BorderFactory;
import com.ulcjava.base.application.ULCAlert;
import com.ulcjava.base.application.ULCBoxPane;
import com.ulcjava.base.application.ULCButton;
import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCFiller;
import com.ulcjava.base.application.border.ULCAbstractBorder;
import com.ulcjava.base.application.enabler.ULCAndEnabler;
import com.ulcjava.base.application.enabler.ULCHasChangedEnabler;
import com.ulcjava.base.application.enabler.ULCMandatoryAndEnabler;
import com.ulcjava.base.application.event.ActionEvent;
import com.ulcjava.base.application.event.IActionListener;
import com.ulcjava.base.application.event.IRoundTripListener;
import com.ulcjava.base.application.event.IWindowListener;
import com.ulcjava.base.application.event.RoundTripEvent;
import com.ulcjava.base.application.event.WindowEvent;
import com.ulcjava.base.server.ULCSession;

/**
 * Creates a box pane containing a form panel and a row of buttons. The button row contains a Save and a Reset button on
 * the right side, additional button can be added on the left side.
 * <p>
 * 
 * @param <T> The generic type representing the type of the FormModel
 */
public class BeanFormDialog<T extends FormModel<?>> implements Serializable {

    private final class ModelReplacer implements Runnable, Serializable {
        private final T fNewModel;

        private ModelReplacer(T newModel) {
            fNewModel = newModel;
        }

        public void run() {
            replaceModel(fNewModel);
        }
    }

    private final ULCBoxPane fButtonPane;
    private final ULCBoxPane fAdditionalButtonPane;
    private final ULCBoxPane fContentPane;
    private ULCButton fSaveButton;
    private ULCButton fResetButton;
    private final List<IActionListener> fSaveActions;
    private final AbstractFormBuilder<T> fFormBuilder;
    private IRoundTripListener fRoundTripListener;


    /**
     * Creates a form dialog using the given form builder.
     * 
     * @param beanForm form builder that creates the form component
     */
    public BeanFormDialog(AbstractFormBuilder<T> beanForm) {
        fFormBuilder = beanForm;
        initRoundTripListener();
        fSaveActions = new ArrayList<IActionListener>();
        fButtonPane = createButtonPane();
        fAdditionalButtonPane = createButtonBox();
        ULCBoxPane buttonRow = new ULCBoxPane(false);
        buttonRow.add(ULCBoxPane.BOX_LEFT_TOP, fAdditionalButtonPane);
        buttonRow.add(ULCBoxPane.BOX_EXPAND_TOP, ULCFiller.createHorizontalGlue());
        buttonRow.add(ULCBoxPane.BOX_RIGHT_TOP, fButtonPane);
        fContentPane = new ULCBoxPane(true);
        fContentPane.setBorder(createBorder(10, 10, 10, 10));
        fContentPane.add(ULCBoxPane.BOX_EXPAND_EXPAND, getFormBuilder().getFormPane());
        fContentPane.add(ULCBoxPane.BOX_EXPAND_TOP, buttonRow);
        doInitForm();
    }

    /**
     * @return the FormModel that holds the data for the form.
     */
    public T getModel() {
        return getFormBuilder().getModel();
    }

    private ULCAbstractBorder createBorder(int top, int left, int buttom, int right) {
        return BorderFactory.createEmptyBorder(top, left, buttom, right);
    }

    private ULCBoxPane createButtonPane() {
        ULCBoxPane boxPane = createButtonBox();
        fSaveButton = new ULCButton("OK");
        fSaveButton.setName("Button.Save");

        fSaveButton.addActionListener(new IActionListener() {

            public void actionPerformed(ActionEvent event) {
                saveBean(event);
            }

        });

        fResetButton = new ULCButton("Reset");
        fResetButton.setName("Button.Reset");
        fResetButton.addActionListener(new IActionListener() {

            public void actionPerformed(ActionEvent event) {
                reset();
            }

        });
        boxPane.add(ULCBoxPane.BOX_RIGHT_EXPAND, fResetButton);
        boxPane.add(ULCBoxPane.BOX_RIGHT_EXPAND, fSaveButton);
        return boxPane;
    }

    private ULCBoxPane createButtonBox() {
        ULCBoxPane boxPane = new ULCBoxPane(false);
        boxPane.setBorder(createBorder(0, 10, 0, 10));
        return boxPane;
    }

    /**
     * @return a panel that contains the form dialog.
     */
    public ULCComponent getContentPane() {
        return fContentPane;
    }

    /**
     * Sets a new model into the form.
     * 
     * @param newModel the model that replaces the old one.
     */
    public void setModel(final T newModel) {
        if (getModel().getBean() != newModel.getBean()) {
            interceptIfDirty(new ModelReplacer(newModel));
        }
    }

    /**
     * Adds a action handler to the forms "Save"-Button. The listener's perform action method is called after the save
     * button is pressed and the model has been validated and checked if it is ready to be saved, i.e. all mandatory
     * fields are set and it contains no errors. It is expected that the application provides a listener that does the
     * actually saving of the bean.
     * 
     * @param saveActionListener handler to react on pressing the "Save"-Button
     */
    public void addSaveActionListener(IActionListener saveActionListener) {
        fSaveActions.add(saveActionListener);
    }

    /**
     * Adds a action handler to the forms "Reset"-Button. The listener's perform action method is called after the reset
     * button is pressed and the model has been reset.
     * 
     * @param resetActionListener handler to react on pressing the "Reset"-Button
     */
    public void addResetActionListener(IActionListener resetActionListener) {
        fResetButton.addActionListener(resetActionListener);
    }

    /**
     * Lets the form save a dirty model before running the given process. If the model has unsaved edits, an alert box
     * is shown asking the user if he likes to save the edits. If the the user chooses "Yes" the bean is saved, if he
     * chooses "No" the edits are discarded. In both case the passed in {@link Runnable} is run. If the user chooses
     * "Cancel" the Runnable is not executed.
     * 
     * @param process code to be executed after saving or discarding any dirty state.
     */
    public void interceptIfDirty(final Runnable process) {
        if (getModel().hasChanges()) {
            final ULCAlert alert = createUnsavedEditsAlert();
            alert.addWindowListener(new IWindowListener() {
                public void windowClosing(WindowEvent event) {
                    if (alert.getValue().equals(alert.getFirstButtonLabel())) {
                        if (saveBean(new ActionEvent(getModel(), "Save"))) {
                            process.run();
                        }
                    } else if (alert.getValue().equals(alert.getSecondButtonLabel())) {
                        try {
                            getModel().undoAll();
                        } catch (PropertyAccessException e) {
                            throw new RuntimeException(e);
                        }
                        process.run();
                    }
                }
            });
            alert.show();
        } else {
            process.run();
        }
    }

    /**
     * Creates the {@link ULCAlert}, that is shown if there are unsaved edits on the model to ask the user if he likes
     * to save the changes or not. Selecting the first button of the alert results in saving the model, pressing the
     * second button discards the edits and continues.
     * 
     * @return the alert to show.
     * @see #interceptIfDirty(Runnable)
     */
    protected ULCAlert createUnsavedEditsAlert() {
        return new ULCAlert("Unsaved edits", "Do you want to save changes?", "Yes", "No", "Cancel");
    }

    private void initRoundTripListener() {
        fRoundTripListener = new IRoundTripListener() {

            public void roundTripWillEnd(RoundTripEvent event) {
                fSaveButton.setEnabled(getDirtyState().isEnabling() && !getModel().hasErrors()
                        && getModel().areAllMandatoryFieldsSet());
            }

            public void roundTripDidStart(RoundTripEvent event) {
            }

        };
        ULCSession.currentSession().addRoundTripListener(fRoundTripListener);
    }

    /**
     * Removes the roundtrip listener that the dialog uses to update the buttons enabling state at the end of each
     * roundtrip. Call this if the form dialog is not needed anymore to free all widgets.
     */
    public void dispose() {
        getFormBuilder().dispose();
        ULCSession.currentSession().removeRoundTripListener(fRoundTripListener);
    }

    private void resetForm() {
        getFormBuilder().reset();
        doInitForm();
    }

    private void doInitForm() {
        ULCAndEnabler andEnabler = new ULCAndEnabler();
        andEnabler.add(getDirtyState());
        andEnabler.add(getMandatoryEnabler());
        fSaveButton.setEnabler(andEnabler);
        fResetButton.setEnabler(getDirtyState());
    }


    private void replaceModel(final T model) {
        getFormBuilder().replaceModel(model);
        resetForm();
    }

    private boolean saveBean(ActionEvent event) {
        if (getModel().canSave()) {
            for (IActionListener listener : fSaveActions) {
                listener.actionPerformed(event);
            }
            getModel().resetUndo();
            return true;
        } else {
            handleModelHasErrorsOnSave();
            return false;
        }
    }

    /**
     * Shows an alert that notifies the user that the bean can not be saved due to errors. Subclasses may overwrite to
     * provide a user feedback of their choice.
     */
    protected void handleModelHasErrorsOnSave() {
        new ULCAlert("Error on save", "Cannot save as long as there are errors in the form", "Ok").show();
    }

    /**
     * Undoes all unsaved edits on the form.
     */
    public void reset() {
        getFormBuilder().reset();
    }

    /**
     * adds the ULCButton to the button row
     * 
     * @param button the button to add
     */
    public void addToButtons(ULCButton button) {
        fAdditionalButtonPane.add(ULCBoxPane.BOX_LEFT_CENTER, button);
    }

    /**
     * @return the form builder that buiolds the form pane.
     */
    protected AbstractFormBuilder<T> getFormBuilder() {
        return fFormBuilder;
    }

    private ULCHasChangedEnabler getDirtyState() {
        return getFormBuilder().getDirtyStateEnabler();
    }

    private ULCMandatoryAndEnabler getMandatoryEnabler() {
        return getFormBuilder().getMandatoryEnabler();
    }

}
