/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ULCSpinner;
import com.ulcjava.base.application.ULCSpinnerDateModel;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a {@link ULCSpinner} with
 * a date model that is added to the form. Holds the parameters to configure the spinner such as:
 * <dl>
 * <dt>initialValue</dt>
 * <dd>The spinner's initial value. . The default is the current date.</dd>
 * <dt>start</dt>
 * <dd>The spinner's start date. This defines the spinner's lower bound. The default is <code>null</code>.</dd>
 * <dt>end</dt>
 * <dd>The spinner's end date. This defines the spinner's upper bound. The default is <code>null</code>.</dd>
 * <dd></dd>
 * <dt>calendarField</dt>
 * <dd>The calendar field that defines the step size. The default is <code>Calendar.DAY_OF_MONTH</code>.</dd>
 * <dt>formatPattern</dt>
 * <dd>The pattern that defines the how the date in the spinner is formatted. By default it is not set. The spinner
 * uses a {@link SimpleDateFormat} with the pattern or a n empty constructor, if the pattern is not set, to format the
 * date. </dd>
 * </dl>
 * 
 * @see ULCSpinner
 * @see ULCSpinnerDateModel
 */
public class DateSpinnerParameter extends LayoutParameter<DateSpinnerParameter, ULCSpinner, WidgetFactory> {
    private String fDateFormatPattern;
    private ULCSpinnerDateModel fSpinnerModel;

    /**
     * Creates a DateSpinnerParameter object for the given property that uses the given WidgetFactory to create the
     * {@link ULCSpinner}.
     * 
     * @param propertyName name of the property the spinner is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public DateSpinnerParameter(String propertyName, WidgetFactory widgetFactory) {
        super(propertyName, widgetFactory);
    }

    /**
     * @return <code>this</code>.
     */
    @Override
    protected DateSpinnerParameter getThis() {
        return this;
    }

    /**
     * @return the initial value of the spinner.
     */
    public Date getInitialValue() {
        return fSpinnerModel.getDate();
    }

    /**
     * @return the spinner's start date.
     */
    public Date getStart() {
        return fSpinnerModel.getStart();
    }

    /**
     * @return the spinner's end date.
     */
    public Date getEnd() {
        return fSpinnerModel.getEnd();
    }

    /**
     * @return the calendar field that defines the step size.
     */
    public int getCalendarField() {
        return fSpinnerModel.getCalendarField();
    }

    /**
     * @return the pattern that defines the how the date in the spinner is formatted.
     */
    public String getDateFormatPattern() {
        return fDateFormatPattern;
    }

    /**
     * Set the value on the spinner.
     * 
     * @param initialValue the value to set. If <code>null</code> then the start date is set if the start date is not
     *            <code>null</code>. If the start date is <code>null</code> too, the current date is set.
     * @return the {@code DateSpinnerParameter} object itself.
     */
    public DateSpinnerParameter initialValue(Date initialValue) {
        Date init = initialValue == null ? (getStart() == null ? new Date() : getStart()) : initialValue;
        fSpinnerModel.setValue(init);
        return this;
    }

    /**
     * Sets the spinner's start date. this defines the spinner's lower bound.
     * 
     * @param start the start date to set.
     * @return the {@code DateSpinnerParameter} object itself.
     */
    public DateSpinnerParameter start(Date start) {
        fSpinnerModel.setStart(start);
        return this;
    }

    /**
     * Sets the spinner's end date. this defines the spinner's upper bound.
     * 
     * @param end the end date to set.
     * @return the {@code DateSpinnerParameter} object itself.
     */
    public DateSpinnerParameter end(Date end) {
        fSpinnerModel.setEnd(end);
        return this;
    }

    /**
     * Sets the calendar field that defines the step size
     * 
     * @param caldendarField to set. Must be one of the valid {@link Calendar} fields.
     * @return the {@code DateSpinnerParameter} object itself
     * @see ULCSpinnerDateModel#setCalendarField(int)
     */
    public DateSpinnerParameter calendarField(int caldendarField) {
        fSpinnerModel.setCalendarField(caldendarField);
        return this;
    }

    /**
     * Sets the format pattern that the spinner uses to display the value.
     * 
     * @param dateFormatPattern that defines how to display the value.
     * @return the {@code DateSpinnerParameter} object itself
     * @see SimpleDateFormat
     */
    public DateSpinnerParameter formatPattern(String dateFormatPattern) {
        getWidget().setEditor(new ULCSpinner.ULCDateEditor(getWidget(), dateFormatPattern));
        fDateFormatPattern = dateFormatPattern;
        return this;
    }

    /**
     * Creates a {@link ULCSpinner} with the {@link WidgetFactory} and sets a {@link ULCSpinnerDateModel} on it.
     * 
     * @return the created spinner.
     */
    @Override
    protected ULCSpinner createWidget() {
        fSpinnerModel = new ULCSpinnerDateModel();
        fSpinnerModel.setCalendarField(Calendar.DAY_OF_MONTH);
        ULCSpinner spinner = getWidgetFactory().createSpinner();
        spinner.setModel(fSpinnerModel);
        return spinner;
    }


}