/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.applicationframework.application.ResourceMap;
import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCLabel;
import com.ulcjava.base.shared.UlcEventConstants;

import java.io.Serializable;


/**
 * Base class for a family of helper classes that are used by the {@link AbstractFormBuilder} to collect the parameters for a widget that is
 * added to the form. This base class holds the common parameters that control how a widget and its label gets integrated into the layout.
 * These are:
 * <dl>
 * <dt>startColumn</dt>
 * <dd>The column of the cell in the current row where the label is put. If <b>noLabel</b> is true, the widget is set in this cell instead.</dd>
 * <dt>notGrowing</dt>
 * <dd>If true, the widget is added to the box pane with a LEFT horizontal alignment. If false the widget is added with the horizontal
 * alignment set to EXPAND. The default is false.</dd>
 * <dt>growVertical</dt>
 * <dd>If true, the widget is added to the box pane with the vertical alignment set to EXPAND. If false the widget is added with the
 * vertical alignment set to CENTER. The default is false.</dd>
 * <dt>append</dt>
 * <dd>If true, the label and the widget are added into the same cell as the previously added widget. The default is false.</dd>
 * <dt>readonly</dt>
 * <dd>If true, the widget is displayed as if the property's readonly state is always set to true. The readonly state of the property is not
 * considered anymore. If false, the widget is set to readonly depending on the property's readonly state. The default is false.</dd>
 * <dt>style</dt>
 * <dd>Sets the style on the widget.</dd>
 * <dt>startNewRow</dt>
 * <dd>If true, starts a new row even if startColumn is set. This can be used to start a new row not on the first cell.</dd>
 * <dt>eventDeliveryMode</dt>
 * <dd>Can be set to one of the constants of {@link UlcEventConstants} and defines the widget's event delivery mode. If not set the widget's
 * event delivery mode is set to the form's default event delivery mode.</dd>
 * <dt>noLabel</dt>
 * <dd>If true, no label will be inserted into the cell before the widget. The widget gets inserted directly into the first cell of the new
 * row, or into the the cell designated with startColumn. If false, a label for the widget is inserted into the first cell and the widget
 * into the second. The default is false.</dd>
 * <dt>labelText</dt>
 * <dd>If set, defines the label's text. If not set the label's text is injected with the form's {@link ResourceMap} using the label's name.
 * </dd>
 * <dt>labelName</dt>
 * <dd>If set, defines the name that is set on the widget's label. If not set, but a propertyName is set, the label's name is set to
 * "<i>propertyName</i>.Label", otherwise the label's name is not set.</dd>
 * <dt>propertyName</dt>
 * <dd>If set, designates the property the widget is bound to.</dd>
 * <dt>widgetName</dt>
 * <dd>If set, defines the name that is set on the widget. If not set but a propertyName is set, the widget' name is set to
 * "<i>propertyName</i>.<i>WidgetType</i>" (where <i>WidgetType</i> is the simple classname of the widget without the ULC prefix., otherwise
 * the widget's name is not set.</dd>
 * </dl>
 * The subclasses of this class create a particular widget and add setters for configuring the widget. This allows to insert a widget to the
 * form with just one line of code. For example :
 * 
 * <pre><code>
 * new TextAreaParameter(...).rows(7).columns(80).growVertical(true)
 * </code></pre>
 * 
 * The instances of a subclass of this class are created by the {@link AbstractFormBuilder}, that uses it's {@link WidgetFactory} to control
 * the widgets that are created.
 * 
 * @param <T> The generic type representing the the type of the LayoutParameter.
 * @param <W> The generic type representing the type of the widget.
 * @param <WF> The generic type representing the type of the WidgetFactory.
 * @see AbstractFormBuilder
 */
public abstract class LayoutParameter<T extends LayoutParameter<T, W, WF>, W extends ULCComponent, WF extends WidgetFactory> implements
        Serializable {
    
    /**
     * The default value of the span of the label across cells
     */
    public static final int DEFAULT_LABEL_SPAN = 1;
    
    /**
     * The default value of start column
     */
    public static final int DEFAULT_START_COLUMN = 0;
    
    private final WF fWidgetFactory;
    
    private int fStartColumn;
    private int fEndColumn;
    private boolean fNotGrowing;
    private boolean fGrowVertical;
    private boolean fAppend;
    private boolean fReadonly;
    private boolean fStartNewRow;
    private int fEventDeliveryMode;
    private String fPropertyName;
    private W fWidget;
    private ULCLabel fLabel;
    private boolean fNoLabel;
    
    /**
     * Creates a {@link LayoutParameter} without a property name.
     * 
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public LayoutParameter(WF widgetFactory) {
        this((String)null, widgetFactory);
    }
    
    /**
     * Creates a {@link LayoutParameter} for a property with the given name.
     * 
     * @param propertyName name of the property the widget is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public LayoutParameter(String propertyName, WF widgetFactory) {
        this(DEFAULT_START_COLUMN, propertyName, widgetFactory);
    }
    
    /**
     * Creates a {@link LayoutParameter} for a property with the given name and the given start column.
     * 
     * @param startColumn the column where the label or, if isNoLabel() is true, the widget is inserted.
     * @param propertyName name of the property the widget is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public LayoutParameter(int startColumn, String propertyName, WF widgetFactory) {
        if (widgetFactory == null) {
            throw new IllegalArgumentException("widgetFactory must not be null");
        }
        fStartColumn = startColumn;
        fPropertyName = propertyName;
        fWidgetFactory = widgetFactory;
        init();
        fWidget = createWidget();
        if (getPropertyName() != null) {
            createLabel();
        }
    }
    
    /**
     * Creates a {@link LayoutParameter} with the given widget.
     * 
     * @param widgetFactory factory that is used to create widgets. Must no be <code>null</code>.
     * @param widget the widget to be inserted.
     */
    
    public LayoutParameter(W widget, WF widgetFactory) {
        if (widgetFactory == null) {
            throw new IllegalArgumentException("widgetFactory must not be null");
        }
        fWidgetFactory = widgetFactory;
        fStartColumn = DEFAULT_START_COLUMN;
        fPropertyName = null;
        init();
        fWidget = widget;
    }
    
    
    private void init() {
        fEndColumn = -1;
        fNotGrowing = false;
        fGrowVertical = false;
        fAppend = false;
        fReadonly = false;
        fStartNewRow = false;
        fNoLabel = false;
        fEventDeliveryMode = -1;
    }
    
    /**
     * Subclasses implement this method to return <code>this</code>. This is needed to have the setter methods return type be the concrete
     * type of the instance. For example {@code TextAreaParameter.append(true)} returns a {@code TextAreaParameter} and {@code
     * ListParameter.append(true)} returns a {@code ListParameter}.
     * 
     * @return the object itself.
     */
    protected abstract T getThis();
    
    /**
     * Subclasses implement this method to create the widget to add to the form using the WidgetFactory.
     * 
     * @return the widget to add to the form.
     */
    protected abstract W createWidget();
    
    /**
     * @return the factory that is used to create widgets.
     */
    protected WF getWidgetFactory() {
        return fWidgetFactory;
    }
    
    private void createLabel() {
        fLabel = getWidgetFactory().createLabel();
        fLabel.setName(getPropertyName() + ".Label");
        fLabel.setLabelFor(getWidget());
    }
    
    
    /**
     * @return true if the widget should grow horizontally.
     */
    public boolean isGrowingHorizontally() {
        return !isNotGrowing();
    }
    
    /**
     * @return true if the widget should not grow horizontally.
     */
    public boolean isNotGrowing() {
        return fNotGrowing;
    }
    
    /**
     * @return true if the widget should grow vertically.
     */
    public boolean isGrowingVertically() {
        return fGrowVertical;
    }
    
    /**
     * @return true if the widget should be added into the same cell as the previous one.
     */
    public boolean isAppend() {
        return fAppend;
    }
    
    /**
     * @return true if the widget is to rendered as readonly.
     */
    public boolean isReadonly() {
        return fReadonly;
    }
    
    /**
     * @return the value of the startNewRow flag.
     */
    public boolean isStartNewRow() {
        return fStartNewRow;
    }
    
    /**
     * @return the column where the label or, if isNoLabel() is true, the widget is inserted.
     */
    public int getStartColumn() {
        return fStartColumn;
    }
    
    /**
     * @return the column before which the widget ends. Must be greater than the start column
     */
    public int getEndColumn() {
        return fEndColumn;
    }
    
    /**
     * Determines if the widget is to be inserted in a new row either because there is no startColumn set nor append is is true or if
     * isStartNewRow() is true.
     * 
     * @return true if a new row should be started.
     */
    public boolean shouldStartRow() {
        return !(isAppend() || (getStartColumn() > 0)) || isStartNewRow();
    }
    
    /**
     * @param defaultMode the mode that is returned, if no event delivery mode is set.
     * @return the event delivery mode that is set, or if not set, the defaultMode.
     */
    public int getEventDeliveryMode(int defaultMode) {
        return fEventDeliveryMode < 0 ? defaultMode : fEventDeliveryMode;
    }
    
    
    /**
     * Sets the column of the current row where the label or, if isNoLabel() is true, the widget is inserted.
     * 
     * @param startColumn the start column for current row
     * @return the {@code LayoutParameter} object itself.
     */
    public T startColumn(int startColumn) {
        fStartColumn = startColumn;
        return getThis();
    }
    
    /**
     * Sets the column of the current row where the label or, if isNoLabel() is true, the widget is inserted.
     * 
     * @param endColumn the end column before which the widget ends.
     * @return the {@code LayoutParameter} object itself.
     */
    public T endColumn(int endColumn) {
        fEndColumn = endColumn;
        return getThis();
    }
    
    /**
     * If the noLabel flag is set to true the insertion of a label before the widget is suppressed.
     * 
     * @return the value of the no label flag.
     */
    public boolean isNoLabel() {
        return fNoLabel;
    }
    
    /**
     * Set the noLabel flag to true to suppress the insertion of a label before the widget.
     * 
     * @param isNoLabel the value the flag is set to
     * @return the {@code LayoutParameter} object itself.
     */
    public T noLabel(boolean isNoLabel) {
        if (isNoLabel) {
            fLabel = null;
        }
        fNoLabel = isNoLabel;
        return getThis();
    }
    
    /**
     * @return the text set on the label or <code>null</code> if there is no label.
     */
    public String getLabelText() {
        return fLabel != null ? fLabel.getText() : null;
    }
    
    /**
     * Sets the text on the label.
     * 
     * @param labelText the text to set.
     * @return the {@code LayoutParameter} object itself.
     * @throws IllegalStateException if {@code isNoLabel} is true.
     */
    public T labelText(String labelText) {
        if (isNoLabel()) {
            throw new IllegalStateException("labelText cannot be set when noLabel is set");
        }
        if (fLabel == null) {
            createLabel();
        }
        fLabel.setText(labelText);
        return getThis();
    }
    
    /**
     * If no specific label name is set and a property name is set, the label name is "<i>propertyName</i>.Label".
     * 
     * @return the name set on the label or <code>null</code> if there is no label.
     */
    public String getLabelName() {
        return fLabel != null ? fLabel.getName() : null;
    }
    
    /**
     * Sets the name on the label.
     * 
     * @param labelName the name to set.
     * @return the {@code LayoutParameter} object itself.
     * @throws IllegalStateException if {@code isNoLabel} is true.
     */
    public T labelName(String labelName) {
        if (isNoLabel()) {
            throw new IllegalStateException("labelName cannot be set when noLabel is set");
        }
        if (fLabel == null) {
            createLabel();
        }
        fLabel.setName(labelName);
        return getThis();
    }
    
    /**
     * @return the label for the widget.
     */
    public ULCLabel getLabel() {
        return fLabel;
    }
    
    /**
     * @return the name of the property the widget gets bound to.
     */
    public String getPropertyName() {
        return fPropertyName;
    }
    
    /**
     * Sets the name of the property the widget gets bound to.
     * 
     * @param propertyName the property name to set.
     * @return the {@code LayoutParameter} object itself.
     */
    public T propertyName(String propertyName) {
        fPropertyName = propertyName;
        return getThis();
    }
    
    
    /**
     * @return the name set on the widget.
     */
    public String getWidgetName() {
        return getWidget().getName();
    }
    
    /**
     * Sets the name on the widget.
     * 
     * @param widgetName the name to set.
     * @return the {@code LayoutParameter} object itself.
     */
    public T widgetName(String widgetName) {
        getWidget().setName(widgetName);
        return getThis();
    }
    
    /**
     * Sets the not grow horizontally flag.
     * 
     * @param isNotGrowing the value the flag is set to.
     * @return the {@code LayoutParameter} object itself.
     */
    public T notGrowing(boolean isNotGrowing) {
        fNotGrowing = true;
        return getThis();
    }
    
    /**
     * Sets the grow vertically flag.
     * 
     * @param isGrowVertical the value the flag is set to.
     * @return the {@code LayoutParameter} object itself.
     */
    public T growVertical(boolean isGrowVertical) {
        fGrowVertical = true;
        return getThis();
    }
    
    /**
     * Sets the append flag. The flag can not be set if the start new row flag is already set.
     * 
     * @param isAppend the value the flag is set to
     * @return the {@code LayoutParameter} object itself.
     * @throws IllegalStateException if the {@code startNewRow} flag is set to true.
     */
    public T append(boolean isAppend) {
        if (isStartNewRow()) {
            throw new IllegalStateException("append cannot be set when start new row is set");
        }
        fAppend = true;
        return getThis();
    }
    
    /**
     * Sets the readonly flag.
     * 
     * @param isReadOnly the value the flag is set to.
     * @return the {@code LayoutParameter} object itself.
     */
    public T readOnly(boolean isReadOnly) {
        fReadonly = true;
        return getThis();
    }
    
    
    /**
     * Sets the start new row flag.
     * 
     * @param isStartNewRow the value the flag is set to.
     * @return the {@code LayoutParameter} object itself.
     * @throws IllegalStateException if the append flag is set to true.
     */
    public T startNewRow(boolean isStartNewRow) {
        if (isAppend()) {
            throw new IllegalStateException("start new row cannot be set when append is set");
        }
        fStartNewRow = isStartNewRow;
        return getThis();
    }
    
    
    /**
     * Sets the event delivery mode for the widget.
     * 
     * @param mode one of {@code UlcEventConstants.SYNCHRONOUS_MODE, UlcEventConstants.ASYNCHRONOUS_MODE, UlcEventConstants.DEFERRED_MODE}.
     * @return the {@code LayoutParameter} object itself.
     */
    public T eventDeliveryMode(int mode) {
        if (mode != UlcEventConstants.SYNCHRONOUS_MODE && mode != UlcEventConstants.ASYNCHRONOUS_MODE
                && mode != UlcEventConstants.DEFERRED_MODE) {
            throw new IllegalArgumentException("mode must be one of the UlcEventConstants Mode constants");
        }
        fEventDeliveryMode = mode;
        return getThis();
    }
    
    /**
     * Convenience method to set the event delivery mode for the widget to synchronous.
     * 
     * @return the {@code LayoutParameter} object itself.
     */
    public T syncMode() {
        return eventDeliveryMode(UlcEventConstants.SYNCHRONOUS_MODE);
    }
    
    /**
     * Convenience method to set the event delivery mode for the widget to asynchronous.
     * 
     * @return the {@code LayoutParameter} object itself.
     */
    public T asyncMode() {
        return eventDeliveryMode(UlcEventConstants.ASYNCHRONOUS_MODE);
    }
    
    /**
     * Convenience method to set the event delivery mode for the widget to deferred.
     * 
     * @return the {@code LayoutParameter} object itself.
     */
    public T deferredMode() {
        return eventDeliveryMode(UlcEventConstants.DEFERRED_MODE);
    }
    
    
    /**
     * @return the widget to be inserted into the form.
     */
    public W getWidget() {
        return fWidget;
    }
    
    /**
     * Sets the widget to be added to the form.
     * 
     * @param widget the widget to be inserted into the form.
     */
    protected void setWidget(W widget) {
        fWidget = widget;
    }
    
    /**
     * Sets the stlye of the widget
     * 
     * @param style key of the stlye
     * @return the {@code LayoutParameter} object itself.
     * @see ResourceMap
     */
    public T style(String style) {
        getWidget().putClientProperty(ResourceMap.STYLE_CLIENT_PROPERTY, style);
        return getThis();
    }
    
    /**
     * @return the style that is set on the widget, <code>null</code> if no style is set.
     */
    public String getStyle() {
        return (String)getWidget().getClientProperty(ResourceMap.STYLE_CLIENT_PROPERTY);
    }
    
}
