/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ULCBoxPane;
import com.ulcjava.base.application.ULCButtonGroup;
import com.ulcjava.base.application.ULCRadioButton;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a group of
 * {@link ULCRadioButton}s contained in a {@link ULCBoxPane} that is added to the form.
 */
public class RadioButtonParameter extends LayoutParameter<RadioButtonParameter, ULCBoxPane, WidgetFactory> {

    private final String[] fLabels;
    private final Object[] fValues;
    private ULCRadioButton[] fRadioButtons;

    /**
     * Creates a group of {@link ULCRadioButton}s for the given property with the given values and labels. The radio
     * buttons are set into a common {@link ULCButtonGroup}. The radio buttons and the button group are created by the
     * given {@link WidgetFactory}. For each value, a radio button is created that sets, if selected, the value on the
     * model.
     * 
     * @param propertyName name of the property the radio buttons are bound to.
     * @param labels that are shown next to the corresponding radio buttons. Must not be <code>null</code> nor empty
     *            and must have the same size as values.
     * @param values that are represented by the corresponding radio buttons. Must not be <code>null</code> nor empty
     *            and must have the same size as labels.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     * @throws IllegalArgumentException if either <code>labels</code> or <code>values</code> is <code>null</code>
     *             or empty or if they don't have the same size.
     */
    public RadioButtonParameter(String propertyName, String[] labels, Object[] values, WidgetFactory widgetFactory) {
        super(propertyName, widgetFactory);
        if (labels == null || labels.length == 0) {
            throw new IllegalArgumentException("labels must not be null nor empty");
        }
        if (values == null || labels.length == 0) {
            throw new IllegalArgumentException("values must not be null nor empty");
        }
        if (labels.length != values.length) {
            throw new IllegalArgumentException("labels and values must have the same size");
        }
        fValues = values;
        fLabels = labels;
        ULCBoxPane radioButtonBox = getWidget();
        ULCButtonGroup buttonGroup = getWidgetFactory().createButtonGroup();
        fRadioButtons = new ULCRadioButton[labels.length];
        for (int i = 0; i < labels.length; i++) {
            fRadioButtons[i] = getWidgetFactory().createRadioButton();
            fRadioButtons[i].setText(labels[i]);
            fRadioButtons[i].setGroup(buttonGroup);
            radioButtonBox.add(ULCBoxPane.BOX_LEFT_EXPAND, fRadioButtons[i]);
        }
    }

    /**
     * @return <code>this</code>.
     */
    @Override
    protected RadioButtonParameter getThis() {
        return this;
    }

    /**
     * @return the labels set on the radio buttons.
     */
    public String[] getLabels() {
        return fLabels;
    }

    /**
     * @return the values set on the radio buttons.
     */
    public Object[] getValues() {
        return fValues;
    }

    /**
     * @return the {@code ULCRadioButton}s.
     */
    public ULCRadioButton[] getRadioButtons() {
        return fRadioButtons;
    }


    /**
     * Creates a {@link ULCBoxPane} with the {@link WidgetFactory}. The box pane acts as a container for the
     * {@link ULCRadioButton}s.
     * 
     * @return the created box pane.
     */
    @Override
    protected ULCBoxPane createWidget() {
        ULCBoxPane b = getWidgetFactory().createBoxPane();
        b.setColumns(0);
        b.setRows(1);
        return b;
    }
}