/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCSlider;

import java.util.Map;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a {@link ULCSlider} that
 * is added to the form. It holds the parameters to configure the slider, such as:
 * <dl>
 * <dt>inverted</dt>
 * <dd>Sets whether the slider's value-range is to be shown in reverse order. The default is <code>false</code>.</dd>
 * <dt>labelTable</dt>
 * <dd>Sets a {@link Map} that assigns labels to values on the slider.</dd>
 * <dt>majorTickSpacing</dt>
 * <dd>Sets the major tick spacing on the slider. The default is 0.</dd>
 * <dt>maximum</dt>
 * <dd>Sets the slider's upper bound. The default is 100.</dd>
 * <dt>minimum</dt>
 * <dd>Sets the slider's lower bound. The default is 0.</dd>
 * <dt>minorTickSpacing</dt>
 * <dd>Sets the minor tick spacing on the slider. The default is 0.</dd>
 * <dt>orientation</dt>
 * <dd>Sets the slider's orientation to either horizontal or vertical. The default is horizontal.</dd>
 * <dt>paintLabels</dt>
 * <dd>Sets the flag that controls if the slider's labels are painted or not. The default is <code>false</code>.</dd>
 * <dt>paintTicks</dt>
 * <dd>Sets the flag that controls if the slider's tick marks are painted or not. The default is <code>false</code>.</dd>
 * <dt>paintTrack</dt>
 * <dd>Sets the flag that controls if the slider's track is painted or not. The default is <code>true</code>.</dd>
 * <dt>snapToTicks</dt>
 * <dd>Sets the flag that controls if the slider's knob snaps to the next tick. The default is <code>false</code>.</dd>
 * </dl>
 */
public class SliderParameter extends LayoutParameter<SliderParameter, ULCSlider, WidgetFactory> {

    protected static final int DEFAULT_SLIDER_MAXIMUM = 100;
    protected static final int DEFAULT_SLIDER_MINIMUM = 0;

    /**
     * Creates a {@code SliderParameter} object for the given property that uses the given {@code WidgetFactory} to
     * create the {@link ULCSlider}.
     * 
     * @param propertyName name of the property the slider is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public SliderParameter(String propertyName, WidgetFactory widgetFactory) {
        super(propertyName, widgetFactory);
        getWidget().setMaximum(DEFAULT_SLIDER_MAXIMUM);
        getWidget().setMinimum(DEFAULT_SLIDER_MINIMUM);
    }

    /**
     * @return <code>this</code>.
     */
    @Override
    protected SliderParameter getThis() {
        return this;
    }

    /**
     * @return true if the slider's value-range is shown in reverse order.
     * @see ULCSlider#getInverted()
     */
    public boolean getInverted() {
        return getWidget().getInverted();
    }

    /**
     * @return the slider's {@link Map} assigning labels to values.
     * @see ULCSlider#getLabelTable()
     */
    public Map getLabelTable() {
        return getWidget().getLabelTable();
    }

    /**
     * @return the number of values between major ticks
     * @see ULCSlider#getMajorTickSpacing()
     */
    public int getMajorTickSpacing() {
        return getWidget().getMajorTickSpacing();
    }

    /**
     * @return the number of values between major ticks
     * @see ULCSlider#getMajorTickSpacing()
     */
    public int getMinorTickSpacing() {
        return getWidget().getMinorTickSpacing();
    }

    /**
     * @return The slider's vertical or horizontal orientation.
     * @see ULCSlider#getOrientation()
     */
    public int getOrientation() {
        return getWidget().getOrientation();
    }

    /**
     * @return true if slider's labels are painted.
     * @see ULCSlider#getPaintLabels()
     */
    public boolean getPaintLabels() {
        return getWidget().getPaintLabels();
    }

    /**
     * @return true if slider's tick marks are painted.
     * @see ULCSlider#getPaintTicks()
     */
    public boolean getPaintTicks() {
        return getWidget().getPaintTicks();
    }

    /**
     * @return true if slider's track is painted.
     * @see ULCSlider#getPaintTrack()
     */
    public boolean getPaintTrack() {
        return getWidget().getPaintTrack();
    }

    /**
     * @return true if the slider knob snaps to the nearest tick mark.
     * @see ULCSlider#getSnapToTicks()
     */
    public boolean getSnapToTicks() {
        return getWidget().getSnapToTicks();
    }

    /**
     * Returns the maximum value supported by the slider.
     * 
     * @return the slider's upper bound.
     * @see ULCSlider#getMaximum()
     */
    public int getMaximum() {
        return getWidget().getMaximum();
    }

    /**
     * Returns the minimum value supported by the slider.
     * 
     * @return the slider's lower bound.
     * @see ULCSlider#getMinimum()
     */
    public int getMinimum() {
        return getWidget().getMinimum();
    }

    /**
     * Sets whether the slider's value-range is to be shown in reverse order.
     * 
     * @param inverted true if the slider's value-range is to be shown in reverse order, false otherwise.
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setInverted(boolean)
     */
    public SliderParameter inverted(boolean inverted) {
        getWidget().setInverted(inverted);
        return this;
    }

    /**
     * Sets a {@link Map} that assigns labels to values on the slider.
     * 
     * @param labels {@link Map} assigning labels to values.
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setLabelTable(Map)
     */
    public SliderParameter labelTable(Map<Integer, ULCComponent> labels) {
        getWidget().setLabelTable(labels);
        return this;
    }

    /**
     * Sets the major tick spacing on the slider. <code>majorTickSpacing</code> represents the distance, measured in
     * values, between each major tick mark.
     * 
     * @param majorTickSpacing the distance between the major ticks.
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setMajorTickSpacing(int)
     */
    public SliderParameter majorTickSpacing(int majorTickSpacing) {
        getWidget().setMajorTickSpacing(majorTickSpacing);
        return this;
    }

    /**
     * Sets the minor tick spacing on the slider. <code>minorTickSpacing</code> represents the distance, measured in
     * values, between each minor tick mark.
     * 
     * @param minorTickSpacing the distance between the minor ticks.
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setMinorTickSpacing(int)
     */
    public SliderParameter minorTickSpacing(int minorTickSpacing) {
        getWidget().setMinorTickSpacing(minorTickSpacing);
        return this;
    }

    /**
     * Sets the slider's orientation. Allowed values are {@code ULCSlider.HORIZONTAL} and {@code ULCSlider.VERTICAL} .
     * 
     * @param orientation the orientation of the slider.
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setOrientation(int)
     */
    public SliderParameter orientation(int orientation) {
        getWidget().setOrientation(orientation);
        return this;
    }

    /**
     * Sets the flag that controls if the slider's labels are painted or not.
     * 
     * @param paintLabels true if the labels of the slider are painted.
     * @return the SliderParameter object itself.
     * @see ULCSlider#setPaintLabels(boolean)
     */
    public SliderParameter paintLabels(boolean paintLabels) {
        getWidget().setPaintLabels(paintLabels);
        return this;
    }

    /**
     * Sets the flag that controls if the slider's tick marks are painted or not.
     * 
     * @param paintTicks true if the tick marks of the slider are painted.
     * @return the SliderParameter object itself.
     * @see ULCSlider#setPaintTicks(boolean)
     */
    public SliderParameter paintTicks(boolean paintTicks) {
        getWidget().setPaintTicks(paintTicks);
        return this;
    }

    /**
     * Sets the flag that controls if the slider's track is painted or not.
     * 
     * @param paintTrack true if the track of the slider are painted.
     * @return the SliderParameter object itself.
     * @see ULCSlider#setPaintTrack(boolean)
     */
    public SliderParameter paintTrack(boolean paintTrack) {
        getWidget().setPaintTrack(paintTrack);
        return this;
    }

    /**
     * Sets the flag that controls if the slider's knob snaps to the closest tick mark.
     * 
     * @param snapToTicks true if the knob of the slider should snap to the closest tick mark.
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setSnapToTicks(boolean)
     */
    public SliderParameter snapToTicks(boolean snapToTicks) {
        getWidget().setSnapToTicks(snapToTicks);
        return this;
    }

    /**
     * Sets the slider's upper bound.
     * 
     * @param maximum the maximum value the slider can be set to.
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setMaximum(int)
     */
    public SliderParameter maximum(int maximum) {
        getWidget().setMaximum(maximum);
        return this;
    }

    /**
     * Sets the slider's lower bound.
     * 
     * @param minimum the minimum value the slider can be set to..
     * @return the {@code SliderParameter} object itself.
     * @see ULCSlider#setMinimum(int)
     */
    public SliderParameter minimum(int minimum) {
        getWidget().setMinimum(minimum);
        return this;
    }

    /**
     * Creates a {@link ULCSlider} with the {@link WidgetFactory}.
     * 
     * @return the created {@code ULCSlider}.
     */
    @Override
    protected ULCSlider createWidget() {
        return getWidgetFactory().createSlider();
    }


}
