/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ULCTextArea;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a {@link ULCTextArea} that
 * is added to the form. It holds the parameters to configure the text area, such as:
 * <dl>
 * <dt>columns</dt>
 * <dd>Sets the number of columns of the text area. The default is 0.</dd>
 * <dt>lineWrap</dt>
 * <dd>Sets the text area's line-wrap policy flag. The default is <code>false</code>, that means no wrapping.</dd>
 * <dt>rows</dt>
 * <dd>Sets the number of rows of the text area. The default is 0.</dd>
 * <dt>tabSize</dt>
 * <dd>Sets the tab size on the text area. The default is 8.</dd>
 * <dt>wrapStyleWord</dt>
 * <dd>Sets the wrap style. The default is <code>false</code>, that means the text is wrapped on the last column.</dd>
 * </dl>
 */
public class TextAreaParameter extends LayoutParameter<TextAreaParameter, ULCTextArea, WidgetFactory> {
    /**
     * Creates a {@code TextAreaParameter} object for the given property that uses the given {@code WidgetFactory} to
     * create the {@link ULCTextArea}.
     * 
     * @param propertyName name of the property the text area is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public TextAreaParameter(String propertyName, WidgetFactory widgetFactory) {
        super(propertyName, widgetFactory);
    }

    /**
     * @return <code>this</code>.
     */
    @Override
    protected TextAreaParameter getThis() {
        return this;
    }

    /**
     * @return the number of rows of the text area.
     * @see ULCTextArea#getRows()
     */
    public int getRows() {
        return getWidget().getRows();
    }

    /**
     * @return the number of columns of the text area.
     * @see ULCTextArea#getColumns()
     */
    public int getColumns() {
        return getWidget().getColumns();
    }

    /**
     * Gets the text area's line-wrap policy flag.
     * 
     * @return the line-wrapping policy flag of the text area.
     * @see ULCTextArea#getLineWrap()
     */
    public boolean getLineWrap() {
        return getWidget().getLineWrap();
    }

    /**
     * @return the tab size.
     * @see ULCTextArea#getTabSize()
     */
    public int getTabSize() {
        return getWidget().getTabSize();
    }

    /**
     * Returns the wrapping style used when this text area is wrapping lines. If true, lines are wrapped at word
     * boundaries. If false, line wraps on the last column.
     * 
     * @return true for word wrapping, false for character wrapping
     * @see ULCTextArea#getWrapStyleWord()
     */
    public boolean getWrapStyleWord() {
        return getWidget().getWrapStyleWord();
    }

    /**
     * Sets the text area's line-wrap policy flag. If the line-wrapping policy flag is true, lines will be wrapped if
     * they do not fit into the length of the text area, otherwise no wrapping is performed.
     * 
     * @param lineWrap true for wrapping, false for not wrapping.
     * @return the {@code TextAreaParameter} object itself.
     * @see ULCTextArea#setLineWrap(boolean)
     */
    public TextAreaParameter lineWrap(boolean lineWrap) {
        getWidget().setLineWrap(lineWrap);
        return this;
    }

    /**
     * Sets the text area's tab size.
     * 
     * @param tabSize number of characters in tab.
     * @return the {@code TextAreaParameter} object itself.
     * @see ULCTextArea#setTabSize(int)
     */
    public TextAreaParameter tabSize(int tabSize) {
        getWidget().setTabSize(tabSize);
        return this;
    }

    /**
     * Sets the text area's wrapping style used when this text area is wrapping lines. True means that lines will be
     * wrapped at word boundaries (e.g. whitespace), otherwise the text area wraps on the last column.
     * 
     * @param wrapStyleWord true for word wrapping, false for character wrapping.
     * @return the {@code TextAreaParameter} object itself.
     * @see ULCTextArea#setWrapStyleWord(boolean)
     */
    public TextAreaParameter wrapStyleWord(boolean wrapStyleWord) {
        getWidget().setWrapStyleWord(wrapStyleWord);
        return this;
    }

    /**
     * Sets the the number of rows on the text area.
     * 
     * @param rows the number of rows.
     * @return the {@code TextAreaParameter} object itself.
     * @see ULCTextArea#setRows(int)
     */
    public TextAreaParameter rows(int rows) {
        getWidget().setRows(rows);
        return this;
    }

    /**
     * Sets the the number of columns on the text area.
     * 
     * @param columns the number of columns.
     * @return the {@code TextAreaParameter} object itself.
     * @see ULCTextArea#setColumns(int)
     */
    public TextAreaParameter columns(int columns) {
        getWidget().setColumns(columns);
        return this;
    }

    /**
     * Creates a {@link ULCTextArea} with the {@link WidgetFactory}.
     * 
     * @return the created {@code ULCTextArea}.
     */
    @Override
    protected ULCTextArea createWidget() {
        ULCTextArea createTextArea = getWidgetFactory().createTextArea();
        if (getPropertyName() != null) {
            createTextArea.setName(getPropertyName() + ".TextArea");
        }
        return createTextArea;
    }
}
