/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form.model;

import org.apache.commons.beanutils.PropertyUtils;

import com.ulcjava.applicationframework.application.AbstractBean;
import com.ulcjava.applicationframework.application.form.AbstractFormBuilder;
import com.ulcjava.base.shared.ErrorCodes;
import com.ulcjava.base.shared.ErrorObject;
import com.ulcjava.base.shared.logging.Level;
import com.ulcjava.base.shared.logging.Logger;

import java.beans.IndexedPropertyChangeEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;


/**
 * The FormModel wraps as an adapter around a java bean to be bound to a form. Most conveniently the form is created by
 * subclassing and using {@link AbstractFormBuilder}. It offers a property based interface to the binding thus allowing
 * to set and get bean properties using the property name. The form should use setProperty(SetPropertyCommand command)
 * to change a property on the wrapped bean or on the form model itself. In addition to the properties of the bean the
 * {@code FormModel} maintains presentation state for the following properties:
 * <ul>
 * <li>error - holds an {@link ErrorObject} for a property. If the the value of the property is valid,
 * getError(propertyName) return <code>null</code>, otherwise an {@code ErrorObject} that shows the kind of error.</li>
 * <li>enabled - a boolean value for each property. If <code>true</code> the property is enabled otherwise it is
 * disabled.</li>
 * <li>mandatory - a boolean value for each property. If <code>true</code> the property is mandatory otherwise it is
 * optional. All mandatory properties have to be set before saving the bean is allowed.</li>
 * <li>readonly - a boolean value for each property. If <code>true</code> the property is readonly otherwise it is
 * editable.</li>
 * </ul>
 * Two hook methods are provided to manage the enabled/readonly/mandatory state: {@code initState()} to set the initial
 * values and {@code updatePresentationState()} to change the state after a property is set. Subclasses of
 * {@code FormModel} can add more properties that can be read and written by set/getProperty like the properties on the
 * wrapped bean. After setting a property with {@code setProperty(SetPropertyCommand)} the following calculations are
 * triggered:
 * <ol>
 * <li>{@code calculate()} is called. Subclasses should overwrite this to update calculated fields.</li>
 * <li>The {@code validate(FormModel<?> formModel)} method of each {@link IValidator} is called with <code>this</code>.
 * The validator performs a check on the {@code FormModel} and its beans data. If the check fails, the validator sets an
 * error on the concerned property or properties.</li>
 * <li>{@code updatePresentationState()} is called. Subclasses should overwrite this to dynamically update the
 * presentation state.</li>
 * </ol>
 * It is recommended to keep the calculations simple and not create a chain of calculations. If there is need for
 * optimization, the {@code hasChanged(String propertyName)} method can be used to check if a property has been changed
 * by the {@link SetPropertyCommand} and then perform the calculation on the changed property.
 * <p>
 * </p>
 * As with Swing, most of the ULC Widgets work with <i>common</i> Types (String, Integer, Double ...). If the bean has
 * properties of another type a {@link IMapper} should be add to perform type conversion on setting and getting the
 * property. A mapper can be set on a single property or on all properties of a given type.
 * 
 * @param <T> The generic type representing the type of the bean that is wrapped by the form model.
 * @see SetPropertyCommand
 * @see AbstractFormBuilder
 */
public class FormModel<T> extends AbstractBean {
    private static final Logger LOG = Logger.getLogger(FormModel.class.getName());

    public static final String ERROR = "error";
    public static final String ENABLED = "enabled";
    public static final String MANDATORY = "mandatory";
    public static final String ILLEGAL_INPUT = "illegalInput";
    public static final String READONLY = "readonly";
    private final T fBean;
    private final Map<String, ErrorObject> fErrorObjectsByProperty;
    private final Map<String, ErrorObject> fIllegalInput;
    private final List<IValidator> fValidators;
    private final Map<Object, IMapper<?, ?>> fMappers;
    private final List<SetPropertyCommand> fChanges;
    private final BooleanPropertyState fEnabled;
    private final BooleanPropertyState fMandatory;
    private final BooleanPropertyState fReadonly;
    private boolean fDoClear;
    private final Set<String> fChangedProperties;

    private static class BooleanPropertyState implements Serializable {

        private final Boolean fValue;
        private final Set<String> fPropertiesNotInDefaultState;

        public BooleanPropertyState(boolean defaultValue) {
            fValue = !defaultValue;
            fPropertiesNotInDefaultState = new HashSet<String>();
        }

        public Boolean getState(String propertyName) {
            if (propertyName == null) {
                throw new IllegalArgumentException("propertyName must not be null");
            }
            if (fPropertiesNotInDefaultState.contains(propertyName)) {
                return fValue;
            } else {
                return !fValue;
            }
        }

        public void setState(String propertyName, Boolean value) {
            if (value == null) {
                throw new IllegalArgumentException("value must not be null");
            }
            if (propertyName == null) {
                throw new IllegalArgumentException("propertyName must not be null");
            }
            if (fValue.booleanValue() == value.booleanValue()) {
                fPropertiesNotInDefaultState.add(propertyName);
            } else {
                fPropertiesNotInDefaultState.remove(propertyName);
            }
        }

        public Collection<? extends String> getProperties() {
            return new ArrayList<String>(fPropertiesNotInDefaultState);
        }

    }

    /**
     * Creates a {@code FormModel} that wraps a bean to make it work with a form.
     * 
     * @param bean the bean to be wrapped, must conform to the JavaBeans specification. Must not be <code>null</code>
     * @throws IllegalArgumentException if the bean is <code>null</code>
     */
    public FormModel(T bean) {
        if (bean == null) {
            throw new IllegalArgumentException("bean must not be null");
        }
        fBean = bean;
        fDoClear = false;
        fEnabled = new BooleanPropertyState(true);
        fMandatory = new BooleanPropertyState(false);
        fReadonly = new BooleanPropertyState(false);
        fMappers = new HashMap<Object, IMapper<?, ?>>();
        fValidators = new ArrayList<IValidator>();
        fChanges = new ArrayList<SetPropertyCommand>();
        fErrorObjectsByProperty = new HashMap<String, ErrorObject>();
        fIllegalInput = new HashMap<String, ErrorObject>();
        fChangedProperties = new HashSet<String>();
        initState();
        addMappers();
        initValidators();
        update(false);
    }

    /**
     * Subclasses should overwrite this to initialize everything that is needed by {@code addMappers, addListeners and
     * createValidators}, and to set the initial presentation state, i.e. the mandatory and readonly attributes.
     */
    protected void initState() {
    }

    /**
     * Subclasses should overwrite this to add the needed mappers for the properties whose type needs to be converted
     * from and to the type in the form. For example, {@link BigDecimalToDoubleMapper} converts BigDecimal into Double
     * allowing the property to be bound to an {@code ULCSlider}. This method is called from the constructor after
     * {@code initState()}.
     * 
     * @see #addMapper(Class, IMapper)
     * @see #addMapper(String, IMapper)
     */
    protected void addMappers() {
    }

    /**
     * Subclasses should overwrite this to create validators for the form model. The validators validate the bean after
     * every input. The method is called from the constructor after {@code initState()}.
     * 
     * @return <code>null</code>, subclasses should overwrite this method to return an array with the validators for
     *         the form model.
     * @see IValidator
     */
    protected IValidator[] createValidators() {
        return null;
    }

    private void initValidators() {
        IValidator[] validators = createValidators();
        if (validators != null) {
            for (IValidator validator : validators) {
                addValidator(validator);
            }
        }
    }


    /**
     * @return the bean wrapped by this {@code FormModel}
     */
    public T getBean() {
        return fBean;
    }

    /**
     * Sets the property of the wrapped bean to a new value. Property changed notifications are sent if the newValue is
     * not equal to the old value of the property. At the end
     * {@code calculate(), validate() and updatePresentationState()} methods are called. If setting the new value fails,
     * an error is set for the property.
     * 
     * @param command The SetPropertyCommand that contains the property name and the value to be set
     */
    public void setProperty(SetPropertyCommand command) {
        String propertyName = command.getPropertyName();
        ErrorObject illegalInputError = getIllegalInput(propertyName);
        Object oldValue = illegalInputError != null ? illegalInputError.getIllegalValue() : getProperty(propertyName);
        if (areDifferent(oldValue, command.getNewValue())) {
            clearChangedProperties();
            command.setOldValue(oldValue);
            performSetProperty(propertyName, command.getNewValue(), oldValue);
            addChange(command);
            update(true);
        }
    }

    /**
     * Sets the property of the wrapped bean to a new value. Property changed notifications are sent if the newValue is
     * not equal to the old property value. Unlike setProperty(SetPropertyCommand) this method does not call
     * {@code calculate(), validate() and updatePresentationState()} at the end. It is meant to be used to set
     * properties from within these methods, for instance {@code calculate()}.
     * 
     * @param propertyName Name of the property to be set. Must not be <code>null</code>.
     * @param newValue the new value to set.
     * @throws IllegalArgumentException if propertyName is <code>null</code>.
     */
    protected void setProperty(String propertyName, Object newValue) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        Object oldValue = getProperty(propertyName);
        if (areDifferent(oldValue, newValue)) {
            performSetProperty(propertyName, newValue, oldValue);
        }

    }

    private void performSetProperty(String propertyName, Object newValue, Object oldValue) {
        try {
            Object modelValue = mapToModel(propertyName, newValue);
            try {
                PropertyUtils.setProperty(getBean(), propertyName, modelValue);
            } catch (Exception e) {
                PropertyUtils.setProperty(this, propertyName, modelValue);
            }
        } catch (Exception e) {
            String message = "Error on setting property " + propertyName + " to ";
            if (newValue != null) {
                message += newValue.toString() + " (" + newValue.getClass().getName() + ")";
            } else {
                message += "<null>";
            }
            LOG.log(Level.INFO, message);
            setIllegalInput(propertyName, new ErrorObject(ErrorCodes.ERROR_CODE_BAD_FORMAT, newValue,
                    new Object[] { propertyName }));
            return;
        }
        setIllegalInput(propertyName, null);
        PropertyChangeEvent propertyChangeEvent = isIndexedProperty(propertyName) ? new IndexedPropertyChangeEvent(
                this, propertyName, oldValue, newValue, getIndex(propertyName)) : new PropertyChangeEvent(this,
                propertyName, oldValue, newValue);
        fChangedProperties.add(propertyName);
    }

    private boolean areDifferent(Object oldValue, Object newValue) {
        return ((oldValue == null) && (newValue != null)) || (oldValue != null && !oldValue.equals(newValue));
    }

    /**
     * Reverts (undo) the last setProperty(SetPropertyCommand) call.
     * 
     * @throws PropertyAccessException if the access to the property fails
     */
    public void undo() throws PropertyAccessException {
        SetPropertyCommand propertyChange = getChanges().remove(getChanges().size() - 1);
        propertyChange.undo(this);
    }

    /**
     * Reverts (undo) all setProperty(SetPropertyCommand) calls.
     * 
     * @throws PropertyAccessException if the access to the property fails
     */
    public void undoAll() throws PropertyAccessException {
        while (!getChanges().isEmpty()) {
            undo();
        }
    }

    /**
     * Clears all recorded changes. Further calls to {@code undoAll()} will revert the {@code FormModel} to the state
     * after this call.
     */
    public void resetUndo() {
        getChanges().clear();
    }

    /**
     * @return true if there are changes on the {@code FormModel} that can be undone.
     */
    public boolean hasChanges() {
        return !getChanges().isEmpty();
    }

    private List<SetPropertyCommand> getChanges() {
        return fChanges;
    }

    private void addChange(SetPropertyCommand command) {
        getChanges().add(command);
    }

    /**
     * Updates the dependent state after setting a property. This consists of calling on the
     * {@code calculate(), validate() and updatePresentationState()}. The call to {@code validate()} can be suppressed
     * to avoid errors to be set.
     * 
     * @param doValidate if true validate will be called, if false validate is not called and errors are not set.
     */
    void update(boolean doValidate) {
        clearErrors();
        calculate();
        if (doValidate) {
            validate();
        }
        updatePresentationState();
    }

    /**
     * Checks if all mandatory fields are filled and there are no errors on the bean. An error is set on the mandatory
     * fields that are not set.
     * 
     * @return true if the bean is ready to be saved, i.e. there are no errors and all mandatory fields are filled.
     */
    public boolean canSave() {
        checkMandatoryFields();
        return !hasErrors();
    }

    private void checkMandatoryFields() {
        List<String> mandatoryProperties = getMandatoryProperties();
        for (String propertyName : mandatoryProperties) {
            try {
                Object propertyValue = getProperty(propertyName);
                if (propertyValue == null || propertyValue.toString().length() == 0) {
                    setErrorForProperty(propertyName, new ErrorObject(ErrorCodes.ERROR_CODE_MANDATORY_FIELD_NOT_FILLED,
                            propertyValue));
                }
            } catch (PropertyAccessException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * @return true if all mandatory properties are set, otherwise false.
     */
    public boolean areAllMandatoryFieldsSet() {
        List<String> mandatoryProperties = getMandatoryProperties();
        for (String propertyName : mandatoryProperties) {
            try {
                Object propertyValue = getProperty(propertyName);
                if (propertyValue == null || propertyValue.toString().length() == 0) {
                    return false;
                }
            } catch (PropertyAccessException e) {
                throw new RuntimeException(e);
            }
        }
        return true;
    }

    private List<String> getMandatoryProperties() {
        return new ArrayList<String>(fMandatory.getProperties());
    }

    /**
     * Gets the value of a property of wrapped bean.
     * 
     * @param propertyName name of the property to get the value of
     * @return the value of the property
     * @throws PropertyAccessException if the access to the property fails
     */
    public Object getProperty(String propertyName) throws PropertyAccessException {
        Object propertyValue;
        try {
            propertyValue = PropertyUtils.getProperty(getBean(), propertyName);
        } catch (Exception e) {
            if (!(e instanceof NoSuchMethodException)) {
                e.printStackTrace();
            }
            try {
                propertyValue = PropertyUtils.getProperty(this, propertyName);
            } catch (Exception e2) {
                throw new PropertyAccessException("Error on getting property " + propertyName, e2);
            }
        }
        return mapToForm(propertyName, propertyValue);

    }

    /**
     * Gets the type of the bean property with the given name.
     * 
     * @param propertyName name of the property to get the type of
     * @return the class of the property designated by propertyName
     * @throws PropertyAccessException if the access to the property fails
     */
    public Class<?> getPropertyType(String propertyName) throws PropertyAccessException {
        try {
            Class propertyType = PropertyUtils.getPropertyType(getBean(), propertyName);
            if (propertyType == null) {
                propertyType = PropertyUtils.getPropertyType(this, propertyName);
                if (propertyType == null) {
                    throw new PropertyAccessException("Error on getting property type of " + propertyName);
                }
            }
            return propertyType;
        } catch (Exception e) {
            throw new PropertyAccessException("Error on getting property type of " + propertyName, e);
        }
    }

    /**
     * Adds a {@link IMapper} that maps data between the form and the model for the property with the given name.
     * 
     * @param propertyName Name of the property on which the mapper is applied.
     * @param mapper Maps data between the form and the model.
     */
    public void addMapper(String propertyName, IMapper<?, ?> mapper) {
        getMappers().put(propertyName, mapper);
    }

    /**
     * Adds a {@link IMapper} that maps data between the form and the model for all properties with a given type that do
     * not already have a specific mapper.
     * 
     * @param propertyType Type on which the mapper is applied.
     * @param mapper Maps data between the form and the model.
     */
    public void addMapper(Class<?> propertyType, IMapper<?, ?> mapper) {
        getMappers().put(propertyType, mapper);
    }

    private Object mapToModel(String propertyName, Object newValue) {
        IMapper mapper = getMapper(propertyName);
        if (mapper != null) {
            return mapper.mapToModel(newValue);
        }
        return newValue;
    }

    private Object mapToForm(String propertyName, Object value) {
        IMapper mapper = getMapper(propertyName);
        if (mapper != null) {
            return mapper.mapToForm(value);
        }
        return value;
    }

    /**
     * @return a map containing all {@code IMapper}s for property names and classes.
     */
    protected Map<Object, IMapper<?, ?>> getMappers() {
        return fMappers;
    }

    /**
     * Retrieves the {@code IMapper} for the given property. If no specific {@code IMapper} was added, the
     * {@code IMapper} for the property type is returned.
     * 
     * @param propertyName name of the property for which the {@code IMapper} is searched for
     * @return the {@code IMapper} that was added for the property, or if none was added, the {@code IMapper} that was
     *         added for the property's type, or <code>null</code> if no matching {@code IMapper} was added.
     * @throws PropertyAccessException if there is no property with the given name.
     */
    protected IMapper<?, ?> getMapper(String propertyName) throws PropertyAccessException {
        IMapper<?, ?> mapper = getMappers().get(propertyName);
        if (mapper == null) {
            if (isIndexedProperty(propertyName)) {
                mapper = getMappers().get(propertyName.substring(0, propertyName.lastIndexOf('[')));
            }
            if (mapper == null) {
                mapper = getMappers().get(getPropertyType(propertyName));
            }
        }
        return mapper;
    }

    private boolean isIndexedProperty(String propertyName) {
        return propertyName.charAt(propertyName.length() - 1) == ']';
    }

    private int getIndex(String propertyName) {
        int startOfIndex = propertyName.lastIndexOf('[');
        if (startOfIndex < 0) {
            return -1;
        }

        return Integer.parseInt(propertyName.substring(startOfIndex + 1, propertyName.length() - 1));
    }

    /**
     * Sets an error object for a given property. This method fires propertyChangeEvent for the error property.
     * 
     * @param propertyName name of the property for which the error is set. Must not be <code>null</code>.
     * @param error the {@link ErrorObject} to be set for the property. Can be <code>null</code> which signals the
     *            absence of an error.
     */
    public void setErrorForProperty(String propertyName, ErrorObject error) {
        setProperty(getErrorStatePropertyName(propertyName), error);
    }

    /**
     * Sets the error state for the property. If the {@code ErrorObject} for the property is changed, a
     * {@link PropertyChangeEvent} is fired with the property name {@code error(<propertyName>)}, where
     * {@code propertyName} is the name of the property for which the {@code ErrorObject} is set.
     * 
     * @param propertyName name of the property for which the error is set. Must not be <code>null</code>.
     * @param error the {@link ErrorObject} to be set for the property. Can be <code>null</code> which signals the
     *            absence of an error.
     */
    public void setError(String propertyName, ErrorObject error) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        ErrorObject oldError = getError(propertyName);
        if (((oldError == null) && (error != null)) || ((oldError != null) && !oldError.equals(error))) {
            if (error == null) {
                getErrorObjectsByPropertyMap().remove(propertyName);
            } else {
                getErrorObjectsByPropertyMap().put(propertyName, error);
            }
            firePropertyChange(getErrorStatePropertyName(propertyName), oldError, error);
        }
    }

    /**
     * Clears the error for a given property
     * 
     * @param propertyName name of the property for which the error is removed
     */
    public void clearError(String propertyName) {
        setProperty(getErrorStatePropertyName(propertyName), null);
    }


    /**
     * @return true if there are any errors on the bean
     */
    public boolean hasErrors() {
        return getErrorObjectsByPropertyMap().size() > 0;
    }

    /**
     * Retrieves the {@code ErrorObject} set for the given property.
     * 
     * @param propertyName Name of the property.
     * @return The error for the given property.
     */
    public ErrorObject getError(String propertyName) {
        return getErrorObjectsByPropertyMap().get(propertyName);
    }

    /**
     * @return the map that contains the {@code ErrorObject}s as values and property names as keys.
     */
    protected Map<String, ErrorObject> getErrorObjectsByPropertyMap() {
        return fErrorObjectsByProperty;
    }

    /**
     * Removes all {@code ErrorObject}s.
     */
    protected void clearErrors() {
        Map<String, ErrorObject> errorMap = getErrorObjectsByPropertyMap();
        Set<String> keySet = errorMap.keySet();
        ArrayList<String> keys = new ArrayList<String>(keySet.size());
        for (Iterator<String> iterator = keySet.iterator(); iterator.hasNext();) {
            String propertyName = iterator.next();
            keys.add(propertyName);
        }
        for (String propertyName : keys) {
            clearError(propertyName);
        }
    }

    private void validate() {
        Set<Entry<String, ErrorObject>> entrySet = getIllegalInput().entrySet();
        for (Entry<String, ErrorObject> entry : entrySet) {
            String propertyName = entry.getKey();
            setError(propertyName, entry.getValue());
        }

        List<IValidator> validators = getValidators();
        for (IValidator validator : validators) {
            validator.validate(this);
        }
    }

    /**
     * Subclasses overwrite this method to perform calculation on the bean and set calculated properties. The properties
     * should be set with the setProperty(String propertyName, Object newValue) to trigger notifications for the value
     * change.
     */
    protected void calculate() {
    }

    /**
     * Subclasses overwrite this method to update the presentation state of the properties.
     */
    public void updatePresentationState() {
    }

    /**
     * Make all the given properties mandatory. Fires {@link PropertyChangeEvent} for each property state set.
     * Subclasses should use this method in the method {@code initState()}.
     * 
     * @param propertyNames the names of the properties which should be made mandatory.
     */
    protected void setMandatory(String... propertyNames) {
        for (String propertyName : propertyNames) {
            setMandatory(propertyName, true);
        }
    }

    /**
     * Makes all the given properties readonly. Fires {@link PropertyChangeEvent} for each property state set.
     * Subclasses should use this method in the method {@code initState()}.
     * 
     * @param propertyNames the names of the properties which should be made readonly.
     */
    protected void setReadonly(String... propertyNames) {
        for (String propertyName : propertyNames) {
            setReadonly(propertyName, true);
        }
    }

    /**
     * Makes all the given properties disabled. Fires {@link PropertyChangeEvent} for each property state set.
     * Subclasses should use this method in the method {@code initState()}.
     * 
     * @param propertyNames the names of the properties which should be disabled.
     */
    protected void setDisabled(String... propertyNames) {
        for (String propertyName : propertyNames) {
            setEnabled(propertyName, false);
        }
    }

    /**
     * Retrieves the enabled state for a given property.
     * 
     * @param propertyName name of the property for which the enabled state is returned. Must not be <code>null</code>.
     * @return true if the property is enabled, false if it is disabled.
     */
    public Boolean getEnabled(String propertyName) {
        return fEnabled.getState(propertyName);
    }

    /**
     * Sets the enabled state for a given property. If the enabled state is changed a {@link PropertyChangeEvent} is
     * fired with the property name "enabled(<b>propertyName</b>)", where <b>propertyName</b> is the name of the
     * property for which the enabled state is set.
     * 
     * @param propertyName name of the property for which the enabled state is set. Must not be <code>null</code>.
     * @param isEnabled the value to be set. Must not be <code>null</code>.
     */
    protected void setEnabled(String propertyName, Boolean isEnabled) {
        Boolean oldState = fEnabled.getState(propertyName);
        if (!oldState.equals(isEnabled)) {
            fEnabled.setState(propertyName, isEnabled);
            firePropertyChange(getEnabledStatePropertyName(propertyName), oldState, isEnabled);
        }
    }

    /**
     * Retrieves the mandatory state for a given property.
     * 
     * @param propertyName name of the property for which the mandatory state is returned. Must not be <code>null</code>.
     * @return true if the property is mandatory, false otherwise.
     */
    public Boolean getMandatory(String propertyName) {
        return fMandatory.getState(propertyName);
    }

    /**
     * Set the mandatory state for the given property. If the mandatory state is changed a {@link PropertyChangeEvent}
     * is fired with the property name "mandatory(<b>propertyName</b>)", Where <b>propertyName</b> is the name of the
     * property for which the mandatory state is set.
     * 
     * @param propertyName name of the property for which the mandatory state is set. Must not be <code>null</code>.
     * @param isMandatory true to make the property mandatory, false otherwise. Must not be <code>null</code>.
     */
    protected void setMandatory(String propertyName, Boolean isMandatory) {
        Boolean oldState = fMandatory.getState(propertyName);
        if (!oldState.equals(isMandatory)) {
            fMandatory.setState(propertyName, isMandatory);
            firePropertyChange(getMandatoryStatePropertyName(propertyName), oldState, isMandatory);
        }
    }

    /**
     * Retrieves the readonly state for a given property. If there is no setter for the property on the bean or on the
     * model, the method returns always <code>true</code>.
     * 
     * @param propertyName name of the property for which the readonly state is returned. Must not be <code>null</code>.
     * @return true if the property is readonly, false otherwise.
     */
    public Boolean getReadonly(String propertyName) {
        if ((!PropertyUtils.isWriteable(getBean(), propertyName)) && (!PropertyUtils.isWriteable(this, propertyName))) {
            return true;
        }

        return fReadonly.getState(propertyName);
    }

    /**
     * Set the readonly state for the given property. If the readonly state is changed a {@link PropertyChangeEvent} is
     * fired with the property name "readonly(<b>propertyName</b>)", where <b>propertyName</b> is the name of the
     * property for which the readonly state is set.
     * 
     * @param propertyName name of the property for which the readonly state is set. Must not be <code>null</code>.
     * @param isReadonly true to make the property readonly, false otherwise. Must not be <code>null</code>.
     */
    protected void setReadonly(String propertyName, Boolean isReadonly) {
        Boolean oldState = fReadonly.getState(propertyName);
        if (!oldState.equals(isReadonly)) {
            fReadonly.setState(propertyName, isReadonly);
            firePropertyChange(getReadonlyStatePropertyName(propertyName), oldState, isReadonly);
        }

    }

    /**
     * Adds a {@link PropertyChangeListener} for the enabled state of the given property.
     * 
     * @param propertyName name of the property for which the {@link PropertyChangeListener} is added on the enabled
     *            state. Must not be <code>null</code>.
     * @param enabledStateListener the PropertyChangeListener to be added.
     */
    public void addEnabledStateListener(String propertyName, PropertyChangeListener enabledStateListener) {
        addPropertyChangeListener(getEnabledStatePropertyName(propertyName), enabledStateListener);
    }

    /**
     * Adds a {@link PropertyChangeListener} for the readonly state of the given property.
     * 
     * @param propertyName name of the property for which the {@link PropertyChangeListener} is added on the readonly
     *            state. Must not be <code>null</code>.
     * @param readOnlyStateListener the PropertyChangeListener to be added.
     */
    public void addReadonlyStateListener(String propertyName, PropertyChangeListener readOnlyStateListener) {
        addPropertyChangeListener(getReadonlyStatePropertyName(propertyName), readOnlyStateListener);
    }

    /**
     * Adds a {@link PropertyChangeListener} for the mandatory state of the given property.
     * 
     * @param propertyName name of the property for which the {@link PropertyChangeListener} is added on the mandatory
     *            state. Must not be <code>null</code>.
     * @param mandatoryStateListener the PropertyChangeListener to be added.
     */
    public void addMandatoryStateListener(String propertyName, PropertyChangeListener mandatoryStateListener) {
        addPropertyChangeListener(getMandatoryStatePropertyName(propertyName), mandatoryStateListener);
    }

    /**
     * Adds a {@link PropertyChangeListener} for the error state of the given property.
     * 
     * @param propertyName name of the property for which the {@link PropertyChangeListener} is added on the error
     *            state. Must not be <code>null</code>.
     * @param errorStateListener the PropertyChangeListener to be added.
     */
    public void addErrorStateListener(String propertyName, PropertyChangeListener errorStateListener) {
        addPropertyChangeListener(getErrorStatePropertyName(propertyName), errorStateListener);
    }

    /**
     * Removes a {@link PropertyChangeListener} for the enabled state of the given property.
     * 
     * @param propertyName name of the property for which the enabled {@link PropertyChangeListener} is removed. Must
     *            not be <code>null</code>.
     * @param enabledStateListener the PropertyChangeListener to be removed.
     */
    public void removeEnabledStateListener(String propertyName, PropertyChangeListener enabledStateListener) {
        removePropertyChangeListener(getEnabledStatePropertyName(propertyName), enabledStateListener);
    }

    /**
     * Removes a {@link PropertyChangeListener} for the readonly state of the given property.
     * 
     * @param propertyName name of the property for which the readonly {@link PropertyChangeListener} is removed. Must
     *            not be <code>null</code>.
     * @param readOnlyStateListener the PropertyChangeListener to be removed.
     */
    public void removeReadonlyStateListener(String propertyName, PropertyChangeListener readOnlyStateListener) {
        removePropertyChangeListener(getReadonlyStatePropertyName(propertyName), readOnlyStateListener);
    }

    /**
     * Removes a {@link PropertyChangeListener} for the mandatory state of the given property.
     * 
     * @param propertyName name of the property for which the mandatory {@link PropertyChangeListener} is removed. Must
     *            not be <code>null</code>.
     * @param mandatoryStateListener the PropertyChangeListener to be removed.
     */
    public void removeMandatoryStateListener(String propertyName, PropertyChangeListener mandatoryStateListener) {
        removePropertyChangeListener(getMandatoryStatePropertyName(propertyName), mandatoryStateListener);
    }

    /**
     * Removes a {@link PropertyChangeListener} for the error state of the given property.
     * 
     * @param propertyName name of the property for which the error state {@link PropertyChangeListener} is removed.
     *            Must not be <code>null</code>.
     * @param errorStateListener the PropertyChangeListener to be removed.
     */
    public void removeErrorStateListener(String propertyName, PropertyChangeListener errorStateListener) {
        removePropertyChangeListener(getErrorStatePropertyName(propertyName), errorStateListener);
    }


    /**
     * Produces the property name of the error state for the given property.
     * 
     * @param propertyName name of the property for which the error state property name is returned. Must not be
     *            <code>null</code>.
     * @return the mapped property name of the error state of the given property.
     */
    static public String getErrorStatePropertyName(String propertyName) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        return FormModel.ERROR + "(" + propertyName + ")";
    }

    /**
     * Produces the property name of the enabled state for the given property.
     * 
     * @param propertyName name of the property for the mapped property for which the enabled state property name is
     *            returned. Must not be <code>null</code>.
     * @return the mapped property name of the enabled state of the given property.
     */
    static public String getEnabledStatePropertyName(String propertyName) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        return FormModel.ENABLED + "(" + propertyName + ")";
    }

    /**
     * Produces the property name of the mapped property for the readonly state for the given property.
     * 
     * @param propertyName name of the property for which the readonly state property name is returned. Must not be
     *            <code>null</code>.
     * @return the mapped property name of the readonly state of the given property.
     */
    static public String getReadonlyStatePropertyName(String propertyName) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        return FormModel.READONLY + "(" + propertyName + ")";
    }

    /**
     * Produces the property name of the mapped property for the mandatory state for the given property.
     * 
     * @param propertyName name of the property for which the mandatory state property name is returned. Must not be
     *            <code>null</code>.
     * @return the mapped property name of the mandatory state of the given property.
     */
    static public String getMandatoryStatePropertyName(String propertyName) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        return FormModel.MANDATORY + "(" + propertyName + ")";
    }

    /**
     * Produces the property name of the mapped property for the illegal input for the given property.
     * 
     * @param propertyName name of the property for which the illegal input property name is returned. Must not be
     *            <code>null</code>.
     * @return the mapped property name of the illegal input of the given property.
     */
    static public String getIllegalInputPropertyName(String propertyName) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        return FormModel.ILLEGAL_INPUT + "(" + propertyName + ")";
    }

    /**
     * Adds a {@link IValidator} that will get called in {@code validate()}.
     * 
     * @param validator the {@code IValidator} to add.
     */
    public void addValidator(IValidator validator) {
        getValidators().add(validator);
    }

    /**
     * @return the list of {@code IValidator}s that are added to the {@code FormModel}
     */
    protected List<IValidator> getValidators() {
        return fValidators;
    }

    /**
     * Returns whether the given property has changed.
     * 
     * @param propertyName name of the property
     * @return true if the property has been changed since the last call to setProperty(SetPropertyCommand) that really
     *         changed a property (i.e. the newValue was different from the property value).
     */
    protected boolean hasChanged(String propertyName) {
        return fChangedProperties.contains(propertyName);
    }

    private void clearChangedProperties() {
        fChangedProperties.clear();
    }

    /**
     * Returns the illegal input error object for the given property.
     * 
     * @param propertyName property name for which the illegal input ErrorObject is retrieved. Must not be
     *            <code>null</code>.
     * @return the ErrorObject with the illegal input or <code>null</code> if the property's last input was ok.
     */
    public ErrorObject getIllegalInput(String propertyName) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        return getIllegalInput().get(propertyName);
    }

    /**
     * Sets an {@code ErrorObject} for a property in case of an input that could not be converted into the property's
     * correct type. This can either happen within setProperty or from the Form if there is a TextField with a DataType
     * that detects the error on the client side. If the {@code ErrorObject} for the illegal input is changed a
     * {@link PropertyChangeEvent} is fired with the property name "illegalInput(<b>propertyName</b>)" where
     * <b>propertyName</b> is the name of the property for which the illegal input ErrorObject is set.
     * 
     * @param propertyName property name for which the illegal input {@code ErrorObject} is set. Must not be
     *            <code>null</code>.
     * @param error the {@code ErrorObject} to be set.
     */
    public void setIllegalInput(String propertyName, ErrorObject error) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        ErrorObject oldError = getIllegalInput(propertyName);
        if (((oldError == null) && (error != null)) || ((oldError != null) && !oldError.equals(error))) {
            if (error == null) {
                getIllegalInput().remove(propertyName);
            } else {
                getIllegalInput().put(propertyName, error);
            }
            firePropertyChange(getIllegalInputPropertyName(propertyName), oldError, error);
        }
    }

    /**
     * @return the map that stores illegal input {@code ErrorObject}s by property names.
     */
    protected Map<String, ErrorObject> getIllegalInput() {
        return fIllegalInput;
    }
}
