/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.easydeployment.server;

import com.ulcjava.applicationframework.application.Application;
import com.ulcjava.applicationframework.application.ILocaleProvider;
import com.ulcjava.applicationframework.application.ResourceManager;
import com.ulcjava.applicationframework.application.ResourceMap;
import com.ulcjava.applicationframework.application.StaticLocaleProvider;
import com.ulcjava.base.server.ApplicationConfiguration;
import com.ulcjava.base.shared.internal.ClassUtilities;
import com.ulcjava.base.shared.logging.Level;
import com.ulcjava.base.shared.logging.Logger;

import java.util.Locale;

public class DeploymentParameter {
    
    private static final Logger LOG = Logger.getLogger(DeploymentParameter.class.getName());
    
    public static final String DEPLOYMENT_PARAMETER_KEY_PREFIX = "Application";
    public static final String DEPLOYMENT_PARAMETER_TITLE = DEPLOYMENT_PARAMETER_KEY_PREFIX + ".title";
    public static final String DEPLOYMENT_PARAMETER_VENDOR = DEPLOYMENT_PARAMETER_KEY_PREFIX + ".vendor";
    public static final String DEPLOYMENT_PARAMETER_HOMEPAGE = DEPLOYMENT_PARAMETER_KEY_PREFIX + ".homepage";
    public static final String DEPLOYMENT_PARAMETER_DESCRIPTION = DEPLOYMENT_PARAMETER_KEY_PREFIX + ".description";
    public static final String DEPLOYMENT_PARAMETER_ICON = DEPLOYMENT_PARAMETER_KEY_PREFIX + ".icon";
    public static final String DEPLOYMENT_PARAMETER_SPLASH = DEPLOYMENT_PARAMETER_KEY_PREFIX + ".splash";
    public static final String DEPLOYMENT_PARAMETER_MENU_GROUP = DEPLOYMENT_PARAMETER_KEY_PREFIX + ".menu_group";
    
    private final Locale fLocale;
    private ResourceMap fResourceMap;
    
    public DeploymentParameter(Locale locale) {
        fLocale = locale;
        
    }
    
    /**
     * @return the title of the application used in jnlp and as the title of the web page in applet deployment.
     */
    public String getTitle() {
        return getResourceMap().getString(DEPLOYMENT_PARAMETER_TITLE);
    }
    
    /**
     * @return the vendor of the application used in jnlp.
     */
    public String getVendor() {
        return getResourceMap().getString(DEPLOYMENT_PARAMETER_VENDOR);
    }
    
    /**
     * @return a URL pointing to where more information on this application can be found used by jnlp.
     */
    public String getHomepage() {
        return getResourceMap().getString(DEPLOYMENT_PARAMETER_HOMEPAGE);
    }
    
    /**
     * @return a short description of the application used in jnlp.
     */
    public String getDescription() {
        return getResourceMap().getString(DEPLOYMENT_PARAMETER_DESCRIPTION);
    }
    
    
    /**
     * Contains the path to an image file relative to the document root that with a given size. For JNLP the icon must have either GIF and
     * JPEG format, Java 6 does support PNG and ICO formats too.
     * 
     * @param size of the image in pixel.
     * @return the path to an image file relative to the document root that is used to represent the application. <code>null</code> if no
     *         icon is defined for the given size.
     */
    public String getIcon(int size) {
        return toResourcePath(DEPLOYMENT_PARAMETER_ICON + "." + size);
    }
    
    /**
     * Contains the path to an image file relative to the document root. If a icon with size 16 is defined it is returned, otherwise the the
     * icon with size 32 and if this is not define either the path to the icon with size 64 is returned.
     * 
     * @return the path to an image file relative to the document root that is used to represent the application. <code>null</code> if no
     *         icon for the sizes 16,32 and 64 is defined.
     */
    public String getFavIcon() {
        String icon = getIcon(16);
        icon = icon != null ? icon : getIcon(32);
        icon = icon != null ? icon : getIcon(64);
        return icon;
    }
    
    private String toResourcePath(String iconKey) {
        ResourceMap resourceMap = getResourceMap();
        String iconPath = resourceMap.getString(iconKey);
        return iconPath != null ? (resourceMap.getResourcesDir() + iconPath) : null;
    }
    
    /**
     * Contains the path to an image file relative to the document root that is used as splash screen that is shown while the application
     * starts. For JNLP the image must have either GIF or JPEG format, Java 6 does support PNG and ICO formats too. If running the
     * application as an applet, any format that is supported by the HTML IMG Tag is allowed.
     * 
     * @return the path to an image file relative to the document root that is used as splash screen for the application.
     */
    public String getSplash() {
        return toResourcePath(DEPLOYMENT_PARAMETER_SPLASH);
    }
    
    /**
     * Defines a submenu under which an application shortcut is added to the start menu.
     * 
     * @return name of the start menu group where a shortcut for the application is created.
     */
    public String getMenuGroup() {
        return getResourceMap().getString(DEPLOYMENT_PARAMETER_MENU_GROUP);
    }
    
    /**
     * @return the locale for which the resources are loaded.
     */
    public Locale getLocale() {
        return fLocale;
    }
    
    /**
     * Accessor to the resource map. The resource map is lazily created on the first, using the createResourceMap() method.
     * 
     * @return the resource map that provides the localized resources.
     */
    protected ResourceMap getResourceMap() {
        if (fResourceMap == null) {
            fResourceMap = createResourceMap();
        }
        return fResourceMap;
    }
    
    /**
     * Creates the application default resource map with the DeploymentParameter's locale. The application class name is read from the
     * application configuration. Subclasses can overwrite this method to provide a custom resource map.
     * 
     * @return the resource map that provides the language dependent resources.
     */
    protected ResourceMap createResourceMap() {
        Class<? extends Application> applicationClass;
        try {
            applicationClass = ClassUtilities.forName(ApplicationConfiguration.getInstance().getApplicationClassName());
            ResourceManager resourceManager = new ResourceManager(applicationClass) {
                @Override
                protected ILocaleProvider getLocaleProvider() {
                    return new StaticLocaleProvider(getLocale());
                }
            };
            ResourceMap resourceMap = resourceManager.getResourceMap();
            return resourceMap;
        } catch (ClassNotFoundException e) {
            String msg = "Application class not found";
            LOG.log(Level.SEVERE, msg, e);
            throw new RuntimeException(msg, e);
        }
    }
    

}
