/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.easydeployment.server;

import com.ulcjava.base.server.ApplicationConfiguration;

import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;

public class EasyDeploymentUtilities {
    private static final String CLIENT_RESOURCE_PREFIX = "/WEB-INF/lib/";
    
    private static final String CLIENT_RESOURCE_POSTFIX = "-client.jar";
    
    private static String sJNLPLauncherLibName = "ulc-all5-client.jar";
    
    public static synchronized String getJNLPLauncherLibName() {
        return sJNLPLauncherLibName;
    }
    
    public static synchronized void setJNLPLauncherLibName(String launcherLibName) {
        sJNLPLauncherLibName = launcherLibName;
    }
    
    private EasyDeploymentUtilities() {
    }
    
    /**
     * @deprecated use DeploymentParameter.getTitle instead
     * @return a title string
     */
    public static String toTitle() {
        return "ULC Application Start";
    }
    
    public static String toAppletArchiveAttribute(ServletContext servletContext) {
        StringBuffer result = new StringBuffer();
        
        Set libPaths = servletContext.getResourcePaths(CLIENT_RESOURCE_PREFIX);
        for (Iterator i = libPaths.iterator(); i.hasNext();) {
            String resourcePath = (String)i.next();
            if (isClientResource(resourcePath)) {
                if (result.length() > 0) {
                    result.append(',');
                }
                result.append(toClientResource(resourcePath));
            }
        }
        
        return result.toString();
    }
    
    public static String toAppletLauncherClassName(ServletContext context, HttpServletRequest request) {
        ApplicationConfiguration config = ApplicationConfiguration.getInstance();
        return config == null || config.getAppletLauncherClassName() == null ? "com.ulcjava.environment.applet.client.DefaultAppletLauncher"
                : config.getAppletLauncherClassName();
    }
    
    public static String toUrlStringParam(HttpServletRequest request) {
        return toBaseURL(request) + "ulc";
    }
    
    public static String toBaseURL(HttpServletRequest request) {
        String baseURL = request.getScheme() + "://" + request.getServerName() + ":" + request.getServerPort() + request.getContextPath()
                + "/";
        return baseURL;
    }
    
    public static String toKeepAliveIntervalParam() {
        return Integer.toString(900);
    }
    
    public static String toApplicationAppletClassName(ServletContext context, HttpServletRequest request) {
        String result = toBaseName(request);
        if (result == null) {
            throw new RuntimeException("unable to get Application");
        }
        
        return result;
    }
    
    
    public static String toApplicationClassName(String classRoot) {
        String applicationClassName = null;
        if (classRoot.endsWith("Application") || classRoot.endsWith("Applet")) {
            applicationClassName = classRoot.replace('/', '.');
        }
        return applicationClassName;
    }
    
    public static String toMatchingClassName(JarInputStream jarResource, String classNamePostfix) throws IOException {
        for (JarEntry entry = jarResource.getNextJarEntry(); entry != null; entry = jarResource.getNextJarEntry()) {
            if (entry.getName().indexOf(classNamePostfix + ".class") >= 0) {
                String result = entry.getName();
                result = result.replace('/', '.');
                return result.substring(0, result.indexOf(".class"));
            }
        }
        
        return null;
    }
    
    private static String toClientResource(String resourcePath) {
        return resourcePath.substring(CLIENT_RESOURCE_PREFIX.length());
    }
    
    private static boolean isClientResource(String resourcePath) {
        return resourcePath.endsWith(CLIENT_RESOURCE_POSTFIX);
    }
    
    private static String toBaseName(HttpServletRequest request) {
        String servletPath = request.getServletPath();
        int resultEndIndex = servletPath.lastIndexOf('.');
        return servletPath.substring("/".length(), resultEndIndex);
    }
    
    
    private static class JarType {
        public static final JarType CLIENT = new JarType("-client.jar");
        public static final JarType SERVER = new JarType("-server.jar");
        
        private final String fPathPostfix;
        
        private JarType(String pathPostfix) {
            fPathPostfix = pathPostfix;
        }
        
        public JarInputStream toJarResource(ServletContext servletContext, HttpServletRequest request) {
            try {
                String jarPath = "/WEB-INF/lib/" + EasyDeploymentUtilities.toBaseName(request) + fPathPostfix;
                InputStream jarStream = servletContext.getResourceAsStream(jarPath);
                return jarStream == null ? null : new JarInputStream(jarStream);
            } catch (IOException e) {
                throw new RuntimeException("unable to load jar resource", e);
            }
        }
        
        public String toMatchingClassName(ServletContext context, HttpServletRequest request, String classNamePostfix) {
            try {
                String jarPath = "/WEB-INF/lib/" + EasyDeploymentUtilities.toBaseName(request) + fPathPostfix;
                InputStream jarStream = context.getResourceAsStream(jarPath);
                
                return jarStream == null ? null : EasyDeploymentUtilities.toMatchingClassName(new JarInputStream(jarStream),
                        classNamePostfix);
            } catch (IOException e) {
                throw new RuntimeException("unable to load jar resource", e);
            }
        }
    }
    
    
    /**
     * Looks up a relative URL from the applications resource bundle and returns the full URL if found, <code>null</code> otherwise.
     */
    public static String toFullURLIfNotNull(HttpServletRequest request, String resourceName) {
        String ifNull = null;
        return toFullURLIfNotNull(request, resourceName, ifNull);
    }
    
    /**
     * Looks up a relative URL from the applications resource bundle and returns the full URL if found, the given default value otherwise.
     */
    public static String toFullURLIfNotNull(HttpServletRequest request, String resourceName, String defaultValue) {
        return resourceName != null ? EasyDeploymentUtilities.toBaseURL(request) + resourceName : defaultValue;
    }
    
    
    public static String toJNLPJarTags(ServletContext servletContext) {
        String jarList = toAppletArchiveAttribute(servletContext);
        String[] jars = jarList.split(",");
        String result = "";
        for (int i = 0; i < jars.length; i++) {
            String jar = jars[i];
            result += "<jar href=\"" + jar + "\"";
            if (jar.endsWith(getJNLPLauncherLibName())) {
                result += " main=\"true\"";
            }
            result += " />";
        }
        return result;
    }
    
    public static String toJNLPLauncherClassName(ServletContext servletContext, HttpServletRequest request) {
        ApplicationConfiguration config = ApplicationConfiguration.getInstance();
        return config == null || config.getJnlpLauncherClassName() == null ? "com.ulcjava.environment.jnlp.client.DefaultJnlpLauncher"
                : config.getJnlpLauncherClassName();
    }
}
