/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */

package com.ulcjava.environment.applet.client;

import com.ulcjava.base.client.InputBlocker;
import com.ulcjava.base.client.UIComponent;
import com.ulcjava.base.client.UIMenuBar;
import com.ulcjava.base.client.UIProxy;
import com.ulcjava.base.client.UIRootPane;
import com.ulcjava.base.shared.internal.IllegalStateException;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Toolkit;
import java.awt.event.ComponentEvent;

import javax.swing.JApplet;
import javax.swing.JComponent;
import javax.swing.JMenuBar;
import javax.swing.JPanel;

/**
 * @ulc.extension
 */
public class UIAppletPane extends UIRootPane {
    private Container fContentPane = null;
    private JMenuBar fMenuBar;
    private JPanel fEmptyContentPane;
    private JMenuBar fEmptyMenuBar;
    
    public UIAppletPane() {
        fEmptyContentPane = new JPanel(new BorderLayout());
        fEmptyMenuBar = null;
    }
    
    /**
     * This method instantiates a <code>JApplet</code> object as the underlying basic widget of <code>UIAppletPane</code>.
     * <p>
     * </p>
     * This method returns a <code>JApplet</code> using the following code:
     * 
     * <pre>
     * return AbstractAppletLauncher.getApplet();
     * </pre>
     * <p>
     * </p>
     * <b>Note: </b> When you override this method in a subclass of <code>UIAppletPane</code>, then you must return an instance of
     * <code>JApplet</code> by invoking <code>{@link AbstractAppletLauncher#getApplet()}</code> and not by calling Swing's applet
     * constructor.
     * 
     * @see UIProxy#createBasicObject(Object[]) for further Javadoc information
     */
    protected Object createBasicObject(Object[] arguments) {
        JApplet applet = AbstractAppletLauncher.getApplet();
        
        if (applet == null) {
            throw new IllegalStateException("No applet available. Make sure the application is deployed in an Applet environment.");
        }
        
        return applet;
    }
    
    /**
     * This method returns the instance of the basic <code>JApplet</code> widget.
     * 
     * @return the instance of the basic <code>JApplet</code> widget
     */
    public JApplet getBasicApplet() {
        return (JApplet)getBasicObject();
    }
    
    protected void preInitializeState() {
        super.preInitializeState();
        
        getBasicApplet().getContentPane().removeAll();
    }
    
    protected void postInitializeState() {
        super.postInitializeState();
        
        // we need to trigger an initial component event so we are considered as dirty
        // when we are dirty we synchronize our location, size, and visible state back to the server
        // -> see dirty data behavior in UIRootPane
        ComponentEvent event = new ComponentEvent(getBasicComponent(), ComponentEvent.COMPONENT_RESIZED);
        Toolkit.getDefaultToolkit().getSystemEventQueue().postEvent(event);
        
        ((JComponent)getBasicRootPane().getContentPane()).revalidate();
        InputBlocker.addRootPaneToBlock(getSession(), getBasicRootPane());
    }
    
    public void registerParent(UIAppletFrame appletFrame) {
        // Register UIAppletFrame in registry. Explicitly uploaded in
        // ULCAppletPane. See UBA-1113.
    }
    
    public void setVisible(boolean visible) {
        boolean oldVisible = getBasicApplet().isVisible();
        super.setVisible(visible);
        
        if (visible == oldVisible) {
            return;
        }
        
        // replacing the JLayeredPane is not supported by Swing and leads
        // to strange painting errors. Thus, both the menu bar and the content pane
        // have to be replaced.
        if (visible) {
            getBasicApplet().setJMenuBar(fMenuBar);
            getBasicApplet().setContentPane(fContentPane);
        } else {
            fMenuBar = getBasicApplet().getJMenuBar();
            fContentPane = getBasicApplet().getContentPane();
            
            getBasicApplet().setJMenuBar(fEmptyMenuBar);
            getBasicApplet().setContentPane(fEmptyContentPane);
            
            ((JComponent)getBasicApplet().getContentPane()).revalidate();
        }
    }
    
    protected void addComponent(UIComponent component, Object constraints, int index) {
        super.addComponent(component, constraints, index);
        
        String constraintStr = (String)constraints;
        
        if (constraintStr.equals(MENUBAR)) {
            UIMenuBar uiMenuBar = (UIMenuBar)component;
            fMenuBar = uiMenuBar.getBasicMenuBar();
        } else if (constraintStr.equals(CONTENT_PANE)) {
            fContentPane = (Container)component.getBasicComponent();
        }
    }
    
    protected void removeComponent(UIComponent component) {
        super.removeComponent(component);
        if (component.getBasicComponent() == getBasicRootPane().getJMenuBar()) {
            fMenuBar = null;
        }
    }
    
    public JComponent getBasicComponentForBorder() {
        return getBasicRootPane();
    }
    
    public void dispose() {
        super.dispose();
        
        InputBlocker.removeRootPaneToBlock(getSession(), getBasicApplet().getRootPane());
        // Ensure the applet area greys out at the end of the proxy's
        // lifecycle.
        setVisible(false);
    }
}
