/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */

package com.ulcjava.environment.jnlp.client;

import com.ulcjava.base.client.FileContents;
import com.ulcjava.base.client.FilteringEventQueue;
import com.ulcjava.base.client.IFileService;
import com.ulcjava.base.client.ServiceException;
import com.ulcjava.base.client.UIComponent;
import com.ulcjava.base.shared.FileChooserConfig;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;
import java.util.TreeSet;

import javax.jnlp.FileOpenService;
import javax.jnlp.FileSaveService;
import javax.jnlp.ServiceManager;
import javax.jnlp.UnavailableServiceException;

/**
 * Implementation of a Jnlp File Service. This implementation assumes that there is only one <code>UISession</code> per code base (which
 * is the case in most Jnlp applications).
 * <p>
 * </p>
 * Alternatively you can sign your resources, give the application all permissions in the Jnlp file, and use the standalone file service.
 * 
 * @ulc.extension
 */
public class JnlpFileService implements IFileService {
    public FileContents[] chooseFiles(FileChooserConfig fileChooserConfig, UIComponent parent) throws ServiceException {
        throw new ServiceException("choose file not supported");
    }
    
    public FileContents openFile(String filePath) throws ServiceException {
        File file = new File(filePath);
        return openFileDialog(file.getParent(), new String[0], false)[0];
    }
    
    public FileContents[] openFiles(FileChooserConfig fileChooserConfig, UIComponent parent) throws ServiceException {
        checkFileSelectionMode(fileChooserConfig);
        
        String pathHint = fileChooserConfig.getCurrentDirectory();
        String[] extensions = getExtensions(fileChooserConfig);
        return openFileDialog(pathHint, extensions, fileChooserConfig.isMultiSelectionEnabled());
    }
    
    public FileContents storeFile(String filePath, InputStream in) throws ServiceException {
        File file = new File(filePath);
        return saveFileDialog(file.getParent(), new String[0], in, file.getName());
    }
    
    public FileContents storeFile(FileChooserConfig fileChooserConfig, UIComponent parent, InputStream in) throws ServiceException {
        checkFileSelectionMode(fileChooserConfig);
        
        String pathHint = fileChooserConfig.getCurrentDirectory();
        String[] extensions = getExtensions(fileChooserConfig);
        String name = fileChooserConfig.getSelectedFile() == null ? null : new File(fileChooserConfig.getSelectedFile()).getName();
        return saveFileDialog(pathHint, extensions, in, name);
    }
    
    // PR 595: workaround implemented
    private FileContents[] openFileDialog(String pathHint, String[] extensions, boolean multiFileSelection) throws ServiceException {
        FilteringEventQueue.ensureIsUninstalled();
        try {
            if (multiFileSelection == false) {
                javax.jnlp.FileContents jnlpFileContents = ((FileOpenService)ServiceManager.lookup("javax.jnlp.FileOpenService"))
                        .openFileDialog(pathHint, extensions);
                if (jnlpFileContents != null) {
                    return new FileContents[] {getFileContents(jnlpFileContents)};
                } else {
                    return null;
                }
            } else {
                javax.jnlp.FileContents[] jnlpFileContents = ((FileOpenService)ServiceManager.lookup("javax.jnlp.FileOpenService"))
                        .openMultiFileDialog(pathHint, extensions);
                if (jnlpFileContents == null || jnlpFileContents.length == 0) {
                    return null;
                }
                FileContents[] fileContents = new FileContents[jnlpFileContents.length];
                for (int i = 0; i < jnlpFileContents.length; i++) {
                    fileContents[i] = getFileContents(jnlpFileContents[i]);
                }
                return fileContents;
            }
        } catch (IOException e) {
            throw new ServiceException("could not load file", e);
        } catch (UnavailableServiceException e) {
            throw new ServiceException("could not load file", e);
        } finally {
            FilteringEventQueue.ensureIsInstalled();
        }
    }
    
    private FileContents getFileContents(javax.jnlp.FileContents jnlpFileContents) throws IOException, UnavailableServiceException {
        return new FileContents(null, jnlpFileContents.getInputStream(), // jnlp does not support file path!
                jnlpFileContents.getLength(), jnlpFileContents.getName());
    }
    
    // PR 595: workaround implemented
    private FileContents saveFileDialog(String pathHint, String[] extensions, InputStream in, String name) throws ServiceException {
        FilteringEventQueue.ensureIsUninstalled();
        try {
            javax.jnlp.FileContents jnlpFileContents = ((FileSaveService)ServiceManager.lookup("javax.jnlp.FileSaveService"))
                    .saveFileDialog(pathHint, extensions, in, name);
            if (jnlpFileContents != null) {
                return new FileContents(null, jnlpFileContents.getInputStream(), jnlpFileContents.getLength(), jnlpFileContents.getName()); 
                // jnlp does not support file path!
            } else {
                return null;
            }
        } catch (IOException e) {
            throw new ServiceException("could not store file", e);
        } catch (UnavailableServiceException e) {
            throw new ServiceException("could not store file", e);
        } finally {
            FilteringEventQueue.ensureIsInstalled();
        }
    }
    
    private String[] getExtensions(FileChooserConfig fileChooserConfig) {
        Set result = new TreeSet();
        for (int i = 0; i < fileChooserConfig.getFileFilterConfigs().length; i++) {
            String[] extensions = fileChooserConfig.getFileFilterConfigs()[i].getAllowedExtensions();
            for (int j = 0; j < extensions.length; j++) {
                result.add(extensions[j]);
            }
        }
        
        return (String[])result.toArray(new String[0]);
    }
    
    private void checkFileSelectionMode(FileChooserConfig fileChooserConfig) throws ServiceException {
        if (fileChooserConfig.getFileSelectionMode() != FileChooserConfig.FILES_ONLY) {
            throw new ServiceException("only FileChooserConfig.FILES_ONLY file selection mode supported");
        }
    }
}