/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import com.ulcjava.base.application.ClientContext;
import com.ulcjava.base.application.border.ULCAbstractBorder;
import com.ulcjava.base.application.border.ULCBevelBorder;
import com.ulcjava.base.application.border.ULCCompoundBorder;
import com.ulcjava.base.application.border.ULCEmptyBorder;
import com.ulcjava.base.application.border.ULCEtchedBorder;
import com.ulcjava.base.application.border.ULCLineBorder;
import com.ulcjava.base.application.border.ULCMatteBorder;
import com.ulcjava.base.application.border.ULCSoftBevelBorder;
import com.ulcjava.base.application.border.ULCTitledBorder;
import com.ulcjava.base.application.datatype.IDataType;
import com.ulcjava.base.application.datatype.ULCAbstractDataType;
import com.ulcjava.base.application.datatype.ULCDateDataType;
import com.ulcjava.base.application.datatype.ULCNumberDataType;
import com.ulcjava.base.application.datatype.ULCPercentDataType;
import com.ulcjava.base.application.datatype.ULCRegularExpressionDataType;
import com.ulcjava.base.application.datatype.ULCStringDataType;
import com.ulcjava.base.application.event.KeyEvent;
import com.ulcjava.base.application.util.Color;
import com.ulcjava.base.application.util.Dimension;
import com.ulcjava.base.application.util.Font;
import com.ulcjava.base.application.util.Insets;
import com.ulcjava.base.application.util.KeyStroke;
import com.ulcjava.base.application.util.Point;
import com.ulcjava.base.application.util.ULCIcon;
import com.ulcjava.base.server.ULCResourceBorder;

import java.lang.reflect.Field;
import java.net.URL;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Base class for a set of resource map utility classes that convert resource strings into a different type. Client should use the static
 * method forType(Class) to get the converter for the target class. It then should call {@code parseString(String,ResourceMap)} on the
 * converter to do the conversion. A developer can add more converters by calling {@code register(IResourceConverter)} in a static code
 * block, most conveniently in his Application subclass.
 * 
 * @param <T> The generic type representing the type to which a resource is converted to.
 */
public abstract class AbstractResourceConverter<T> implements IResourceConverter<T> {
    
    private static Map<Class<?>, IResourceConverter<?>> sConverters = new HashMap<Class<?>, IResourceConverter<?>>();
    private final Class<T> fType;
    
    /**
     * @param type in to which resource strings can be converted.
     */
    protected AbstractResourceConverter(Class<T> type) {
        if (type == null) {
            throw new IllegalArgumentException("type must not be null");
        }
        fType = type;
    }
    
    /**
     * @return the type in to which resource strings can be converted.
     */
    public Class<T> getType() {
        return fType;
    }
    
    /**
     * @param s string to convert.
     * @param r {@code ResourceMap} where the resource is taken from.
     * @return the converted object.
     * @throws ResourceConverterException if the conversion fails.
     */
    public T parseString(String s, ResourceMap r) throws ResourceConverterException {
        if (s == null) {
            throw new ResourceConverterException("string must not be null", s);
        }
        if (s.trim().length() == 0) {
            throw new ResourceConverterException("string must not be empty", s);
        }
        return doParseString(s.trim(), r);
    }
    
    /**
     * Converts a string into a resource object. The resource is taken from the given resource map.
     * 
     * @param s string to convert.
     * @param r {@code ResourceMap} where the resource is taken from.
     * @return the converted object.
     * @throws ResourceConverterException if the conversion fails.
     */
    public abstract T doParseString(String s, ResourceMap r) throws ResourceConverterException;
    
    /**
     * Registers a {@code IResourceConverter} in the resource converter store.
     * 
     * @param abstractResourceConverter the resource converter to register.
     */
    public static void register(IResourceConverter<?> abstractResourceConverter) {
        if (abstractResourceConverter == null) {
            throw new IllegalArgumentException("resourceConverter must not be null");
        }
        sConverters.put(abstractResourceConverter.getType(), abstractResourceConverter);
    }
    
    /**
     * Retrieves a converter for the given class from the resource converter store.
     * 
     * @param type for which a converter is requested.
     * @return the resource converter stored for the given type.
     */
    public static IResourceConverter<?> forType(Class<?> type) {
        if (type == null) {
            throw new IllegalArgumentException("type must not be null");
        }
        if (type.isPrimitive()) {
            type = getObjectType(type);
        }
        return sConverters.get(type);
    }
    
    private static Class<?> getObjectType(Class<?> type) {
        if (type.equals(java.lang.Boolean.TYPE)) {
            return Boolean.class;
        }
        if (type.equals(java.lang.Integer.TYPE)) {
            return Integer.class;
        }
        if (type.equals(java.lang.Double.TYPE)) {
            return Double.class;
        }
        if (type.equals(java.lang.Long.TYPE)) {
            return Long.class;
        }
        if (type.equals(java.lang.Short.TYPE)) {
            return Short.class;
        }
        if (type.equals(java.lang.Float.TYPE)) {
            return Float.class;
        }
        if (type.equals(java.lang.Byte.TYPE)) {
            return Byte.class;
        }
        if (type.equals(java.lang.Character.TYPE)) {
            return Character.class;
        }
        return type;
    }
    
    static {
        IResourceConverter<?>[] standardConverters = new IResourceConverter[] {new BooleanResourceConverter("true", "on", "yes"),
                new IntegerResourceConverter(), new ByteResourceConverter(), new ShortResourceConverter(), new FontResourceConverter(),
                new LongResourceConverter(), new FloatResourceConverter(), new DoubleResourceConverter(), new ColorResourceConverter(),
                new ULCIconConverter(), new PointResourceConverter(), new DimensionResourceConverter(), new KeyStrokeResourceConverter(),
                new ULCBorderConverter(), new ULCDataTypeConverter(), new InsetResourceConverter()};
        for (IResourceConverter<?> resourceConverter : standardConverters) {
            register(resourceConverter);
        }
    }
    
    /**
     * Exception to signal that an error occurred during the resource conversion.
     */
    public static class ResourceConverterException extends Exception {
        
        /**
         * Creates a {@code ResourceConverterException} for the given bad input, with the given message and root cause.
         * 
         * @param message plain text message explaining the error.
         * @param badString input that could not be converted.
         * @param cause exception that was thrown when the input was converted using some standard conversion methods.
         */
        public ResourceConverterException(String message, String badString, Throwable cause) {
            super(message + " string: \"" + shorten(badString) + "\"", cause);
        }
        
        private static String shorten(String string) {
            return string.length() <= 128 ? string : string.substring(0, 127) + "... (+" + (string.length() - 128) + " characters)";
        }
        
        /**
         * Creates a ResourceConverterException for the given bad input and the given message.
         * 
         * @param message plain text message explaining the error.
         * @param badString input that could not be converted.
         */
        public ResourceConverterException(String message, String badString) {
            this(message, badString, null);
        }
        
    }
    
    /**
     * Basic converter for number types that takes care of error handling. Subclasses must implement just the parsing step.
     * 
     * @param <T> The generic type representing a Number resource.
     */
    public static abstract class AbstractNumberResourceConverter<T extends Number> extends AbstractResourceConverter<T> {
        
        protected AbstractNumberResourceConverter(Class<T> numberClass) {
            super(numberClass);
        }
        
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public final T doParseString(String s, ResourceMap r) throws ResourceConverterException {
            T result = null;
            try {
                result = parseNumber(s);
            } catch (NumberFormatException e) {
                throw new ResourceConverterException("Could not convert to " + getType().getSimpleName(), s, e);
            }
            return result;
        }
        
        /**
         * Do the conversion form the String s into the converter's type.
         * 
         * @param s the input to convert.
         * @return the converted object of the converter's type.
         */
        protected abstract T parseNumber(String s);
        
    }
    
    /**
     * Converts a resource string into a Integer. Uses Integer.parseInt(String s) to parse and convert the string.
     */
    public static class IntegerResourceConverter extends AbstractNumberResourceConverter<Integer> {
        
        protected IntegerResourceConverter() {
            super(Integer.class);
        }
        
        @Override
        protected Integer parseNumber(String s) {
            return Integer.parseInt(s);
        }
    }
    /**
     * Converts a resource string into a Byte. Uses Byte.parseByte(String s) to parse and convert the string.
     */
    public static class ByteResourceConverter extends AbstractNumberResourceConverter<Byte> {
        
        protected ByteResourceConverter() {
            super(Byte.class);
        }
        
        @Override
        protected Byte parseNumber(String s) {
            return Byte.parseByte(s);
        }
    }
    
    /**
     * Converts a resource string into a Short. Uses Short.parseShort(String s) to parse and convert the string.
     */
    public static class ShortResourceConverter extends AbstractNumberResourceConverter<Short> {
        
        protected ShortResourceConverter() {
            super(Short.class);
        }
        
        @Override
        protected Short parseNumber(String s) {
            return Short.parseShort(s);
        }
    }
    /**
     * Converts a resource string into a Long. Uses Long.parseLong(String s) to parse and convert the string.
     */
    public static class LongResourceConverter extends AbstractNumberResourceConverter<Long> {
        
        protected LongResourceConverter() {
            super(Long.class);
        }
        
        @Override
        protected Long parseNumber(String s) {
            return Long.parseLong(s);
        }
    }
    /**
     * Converts a resource string into a Float. Uses Float.parseFloat(String s) to parse and convert the string.
     */
    public static class FloatResourceConverter extends AbstractNumberResourceConverter<Float> {
        
        protected FloatResourceConverter() {
            super(Float.class);
        }
        
        @Override
        protected Float parseNumber(String s) {
            return Float.parseFloat(s);
        }
    }
    /**
     * Converts a resource string into a Double. Uses Double.parseDouble(String s) to parse and convert the string.
     */
    public static class DoubleResourceConverter extends AbstractNumberResourceConverter<Double> {
        
        protected DoubleResourceConverter() {
            super(Double.class);
        }
        
        @Override
        protected Double parseNumber(String s) {
            return Double.parseDouble(s);
        }
    }
    /**
     * Converts a resource string into a Boolean. Returns true if the string is one of a list of strings for true (case insensitive). The
     * list that is defined for the registered default boolean converter is "true", "on", "yes".
     */
    public static class BooleanResourceConverter extends AbstractResourceConverter<Boolean> {
        private final List<String> fTrueStrings;
        
        /**
         * Creates a converter for the {@link Boolean} class. It uses the given strings to create a Boolean.TRUE.
         * 
         * @param trueStrings strings that are converted into Boolean.TRUE (case insensitive).
         */
        protected BooleanResourceConverter(String... trueStrings) {
            super(Boolean.class);
            fTrueStrings = new ArrayList<String>(trueStrings.length);
            for (String trueString : trueStrings) {
                fTrueStrings.add(trueString.toUpperCase());
            }
        }
        
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public Boolean doParseString(String s, ResourceMap r) throws ResourceConverterException {
            return fTrueStrings.contains(s.toUpperCase());
        }
        
    }
    
    /**
     * Base class to parse a fixed size comma separated list of integers. The integers are converted by the {@link IntegerResourceConverter}
     * .
     * 
     * @param <T> The generic type representing the type of converted resource object.
     */
    public static abstract class AbstractIntegerListResourceConverter<T> extends AbstractResourceConverter<T> {
        
        private final int[] fNumberOfParameters;
        
        protected AbstractIntegerListResourceConverter(Class<T> type, int... numberOfParameters) {
            super(type);
            fNumberOfParameters = numberOfParameters;
        }
        
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public T doParseString(String s, ResourceMap r) throws ResourceConverterException {
            String[] paramStrings = s.split(",");
            if (!isLegalNumberOfParameters(paramStrings.length)) {
                throw new ResourceConverterException(getType().getSimpleName() + " needs " + printLegalNumbers() + " parameter", s);
            }
            Integer[] parameters = new Integer[paramStrings.length];
            for (int i = 0; i < paramStrings.length; i++) {
                parameters[i] = intParameter(paramStrings[i], r);
            }
            return createObject(parameters);
        }
        
        private String printLegalNumbers() {
            String result = "";
            for (int i = 0; i < fNumberOfParameters.length; i++) {
                if (i < fNumberOfParameters.length - 2) {
                    result += fNumberOfParameters[i] + ",";
                } else if (i < fNumberOfParameters.length - 1) {
                    result += fNumberOfParameters[i] + " or ";
                } else {
                    result += fNumberOfParameters[i];
                }
            }
            return result;
        }
        
        private boolean isLegalNumberOfParameters(int number) {
            for (int i = 0; i < fNumberOfParameters.length; i++) {
                if (number == fNumberOfParameters[i]) {
                    return true;
                }
            }
            return false;
        }
        
        protected abstract T createObject(Integer[] parameters);
        
    }
    
    /**
     * Converts a resource string into a com.ulcjava.base.application.util.Point. Legal format for point is: <b>x,y</b> where x and y are
     * integers converted by the {@link IntegerResourceConverter}.
     */
    public static class PointResourceConverter extends AbstractIntegerListResourceConverter<Point> {
        
        protected PointResourceConverter() {
            super(Point.class, 2);
            
        }
        
        @Override
        protected Point createObject(Integer[] parameters) {
            return new Point(parameters[0], parameters[1]);
        }
        
    }
    
    /**
     * Converts a resource string into a com.ulcjava.base.application.util.Dimension. Legal format for dimension is: <b>x,y</b> where x and
     * y are integers converted by the {@link IntegerResourceConverter}.
     */
    public static class DimensionResourceConverter extends AbstractIntegerListResourceConverter<Dimension> {
        
        protected DimensionResourceConverter() {
            super(Dimension.class, 2);
            
        }
        
        @Override
        protected Dimension createObject(Integer[] parameters) {
            return new Dimension(parameters[0], parameters[1]);
        }
        
    }
    /**
     * Converts a resource string into a com.ulcjava.base.application.util.Insets. Legal format for insets are:
     * <ul>
     * <li><b>x</b> is assigned to all sides
     * <li><b>x,y</b> where <b>x</b> is assigned to top and bottom and <b>y</b> to left and right
     * <li><b>top,left,bottom,right</b>
     * </ul>
     * All integers are converted by the {@link IntegerResourceConverter}.
     */
    public static class InsetResourceConverter extends AbstractIntegerListResourceConverter<Insets> {
        
        protected InsetResourceConverter() {
            super(Insets.class, 4, 2, 1);
            
        }
        
        @Override
        protected Insets createObject(Integer[] parameters) {
            switch (parameters.length) {
                case 1:
                    return new Insets(parameters[0], parameters[0], parameters[0], parameters[0]);
                case 2:
                    return new Insets(parameters[0], parameters[1], parameters[0], parameters[1]);
                    
                default:
                    return new Insets(parameters[0], parameters[1], parameters[2], parameters[3]);
            }
            
        }
        
    }
    
    /**
     * Converts a string into a com.ulcjava.base.application.util.KeyStroke. It follows the same rules as
     * {@link KeyStroke#getKeyStroke(String s)} except it accepts an additional modifier <i>shortcut</i> which resolves to <i>control</i> or
     * <i>meta</i> depending on the client's system menu shortcut key mask.
     */
    public static class KeyStrokeResourceConverter extends AbstractResourceConverter<KeyStroke> {
        
        protected KeyStrokeResourceConverter() {
            super(KeyStroke.class);
            
        }
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public KeyStroke doParseString(String s, ResourceMap r) throws ResourceConverterException {
            try {
                s = s.replace("shortcut", ClientContext.getMenuShortcutKeyMask() == KeyEvent.META_MASK ? "meta" : "control");
                s = s.substring(0, s.length() - 1) + s.substring(s.length() - 1).toUpperCase();
                return KeyStroke.getKeyStroke(s);
            } catch (Exception e) {
                throw new ResourceConverterException("Could not convert to " + getType().getSimpleName(), s, e);
            }
        }
    }
    
    /**
     * Converts a string into a com.ulcjava.base.application.util.Font. Legal format for fonts resources are:
     * <ul>
     * <li><em>fontname style pointsize</em>
     * <li><em>fontname pointsize</em>
     * <li><em>fontname style</em>
     * <li><em>fontname</em>
     * <li><em>fontname-style-pointsize</em>
     * <li><em>fontname-pointsize</em>
     * <li><em>fontname-style</em>
     * <li><em>fontname</em>
     * <li><em>-style-pointsize</em>
     * <li><em>-pointsize</em>
     * <li><em>-style</em>
     * </ul>
     * where <i>fontname</i> is the name of the font or family, <i>style</i> is one of <b>PLAIN</b>,<b>BOLD</b>,<b>ITALIC</b> or
     * <b>BOLDITALIC</b> (case-insensitive, therefore <b>plain</b> or <b>Plain</b> is also allowed) and <i>pointsize</i> is a positive
     * integer. If <i>style</i> or <i>pointsize</i> is omitted the default values (<code>PLAIN</code> and <code>12</code>) are used. If
     * fontname is omitted "Default" is used.
     * 
     * @see java.awt.Font#decode(String)
     */
    public static class FontResourceConverter extends AbstractResourceConverter<Font> {
        
        private enum FontStyle {
            PLAIN(Font.PLAIN), BOLD(Font.BOLD), ITALIC(Font.ITALIC), BOLDITALIC(Font.BOLD | Font.ITALIC);
            private final int fStyle;
            
            private FontStyle(int style) {
                fStyle = style;
            }
            
            public int getStyle() {
                return fStyle;
            }
        }
        
        private static final int DEFAULT_STYLE = Font.PLAIN;
        private static final int DEFAULT_SIZE = 12;
        
        protected FontResourceConverter() {
            super(Font.class);
            
        }
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public Font doParseString(String s, ResourceMap r) throws ResourceConverterException {
            
            Font font = null;
            font = parse(s, ' ');
            if (font == null) {
                font = parse(s, '-');
            }
            if (font == null) {
                font = new Font(s, DEFAULT_STYLE, DEFAULT_SIZE);
            }
            return font;
        }
        
        private Font parse(String s, char delimiter) {
            Font font = null;
            String[] blankSplited = s.split(new Character(delimiter).toString());
            String name = null;
            int size = 0;
            int style = -1;
            if (blankSplited.length > 1) {
                size = parseSize(blankSplited[blankSplited.length - 1]);
                if (size > 0) {
                    if (blankSplited.length == 2) {
                        name = blankSplited[0];
                        style = DEFAULT_STYLE;
                    } else {
                        style = parseStyle(blankSplited[blankSplited.length - 2]);
                        if (style >= 0) {
                            name = startToSecondLast(s, delimiter);
                        } else {
                            style = DEFAULT_STYLE;
                            name = startToLast(s, delimiter);
                        }
                    }
                } else {
                    style = parseStyle(blankSplited[blankSplited.length - 1]);
                    if (style >= 0) {
                        name = startToLast(s, delimiter);
                        size = DEFAULT_SIZE;
                    }
                }
            }
            if (name != null) {
                if (name.trim().length() == 0) {
                    name = "Default";
                }
                font = new Font(name, style, size);
                
            }
            return font;
        }
        
        private String startToSecondLast(String s, char delimiter) {
            return startToLast(startToLast(s, delimiter), delimiter);
        }
        
        private String startToLast(String s, char delimiter) {
            return s.substring(0, s.lastIndexOf(delimiter));
        }
        
        private int parseStyle(String possibleStyle) {
            int style = -1;
            try {
                FontStyle styleCode = FontStyle.valueOf(possibleStyle.toUpperCase());
                style = styleCode.getStyle();
            } catch (Exception e) {
            }
            return style;
        }
        
        private int parseSize(String possibleSize) {
            int size = 0;
            try {
                size = Integer.parseInt(possibleSize);
            } catch (NumberFormatException nfe) {
                size = 0;
            }
            return size;
        }
    }
    
    /**
     * Converts a string into a com.ulcjava.base.application.util.Color. Legal format for color resources are:
     * <ul>
     * <li>#RRGGBB
     * <li>#AARRGGBB,
     * <li>R, G, B
     * <li>R, G, B, A
     * <li>Color constant names, for example "gray" or "red"
     * </ul>
     * In the formats starting with # the values for the red, green, blue and alpha are given as two hexadecimal digits, the comma separated
     * lists takes 3 or 4 integer values.
     * 
     * @see Color
     */
    public static class ColorResourceConverter extends AbstractResourceConverter<Color> {
        
        /**
         * Creates a Color Resource converter.
         */
        public ColorResourceConverter() {
            super(Color.class);
        }
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public Color doParseString(String s, ResourceMap r) throws ResourceConverterException {
            if (s.startsWith("#")) {
                return parseHexCode(s);
            } else if (s.contains(",")) {
                return parseCommaSeparated(s);
            } else {
                return parseConstant(s);
            }
        }
        
        private Color parseConstant(String s) throws ResourceConverterException {
            try {
                Field constantColor = Color.class.getDeclaredField(s);
                return (Color)constantColor.get(null);
            } catch (Exception e) {
                throw new ResourceConverterException("No color found for ", s, e);
            }
        }
        
        private Color parseHexCode(String s) throws ResourceConverterException {
            String hexDigit = "[0-9a-fA-F]";
            if (s.matches("#(?:" + hexDigit + hexDigit + "){3}(" + hexDigit + hexDigit + ")?")) {
                String numbers = s.substring(1);
                int paramCount = numbers.length() == 6 ? 3 : 4;
                int[] ints = new int[paramCount];
                for (int i = 0; i < paramCount; i++) {
                    try {
                        ints[i] = Integer.parseInt(numbers.substring(i * 2, i * 2 + 2), 16);
                    } catch (NumberFormatException e) {
                        throw new ResourceConverterException("Illegal characters in ", s, e);
                    }
                }
                
                if (paramCount == 4) {
                    ints = new int[] {ints[1], ints[2], ints[3], ints[0]};
                }
                return createColor(s, ints);
            }
            throw new ResourceConverterException("No color found for ", s);
        }
        
        private Color parseCommaSeparated(String s) throws ResourceConverterException {
            String[] numbers = s.split(",");
            if (numbers.length != 3 && numbers.length != 4) {
                throw new ResourceConverterException("Color needs three(r,g,b) or four (r,g,b,a) parameters ", s);
            }
            int[] ints = new int[numbers.length];
            for (int i = 0; i < ints.length; i++) {
                try {
                    ints[i] = Integer.parseInt(numbers[i]);
                } catch (NumberFormatException e) {
                    throw new ResourceConverterException("Illegal characters in ", s, e);
                }
            }
            return createColor(s, ints);
        }
        
        private Color createColor(String s, int... i) throws ResourceConverterException {
            try {
                if (i.length == 4) {
                    return new Color(i[0], i[1], i[2], i[3]);
                } else {
                    return new Color(i[0], i[1], i[2]);
                }
            } catch (IllegalArgumentException e) {
                throw new ResourceConverterException(
                        "Color needs three(r,g,b) or four (r,g,b,a) positive values in the range of 0 and 255 ", s, e);
            }
        }
    }
    
    /**
     * Converts a string into a com.ulcjava.base.application.util.ULCIcon. It uses the ResourceMap's resources directory to get the resource
     * path. If it is not found, the resource is searched in the ResourceMap hierarchy upwards.
     */
    public static class ULCIconConverter extends AbstractResourceConverter<ULCIcon> {
        
        protected ULCIconConverter() {
            super(ULCIcon.class);
        }
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public ULCIcon doParseString(String s, ResourceMap r) throws ResourceConverterException {
            if (r == null) {
                throw new ResourceConverterException("resourceMap must not be null", s);
            }
            URL iconURL = r.getClassLoader().getResource(r.getResourcesDir() + s);
            while (iconURL == null && r.getParent() != null) {
                r = r.getParent();
                iconURL = r.getClassLoader().getResource(r.getResourcesDir() + s);
            }
            if (iconURL == null) {
                throw new ResourceConverterException("Not a valid url", s);
            }
            ULCIcon icon = new ULCIcon(iconURL);
            return icon;
        }
    }
    
    private static final String COMMA_MASK = "qqqqqqqqqqqqqqqqqqwwwwwwwwwwwwwwwwwwwwwwwwwww";
    
    private static String[] splitIntoParameters(String parameterString) {
        parameterString = parameterString.replaceAll("\\\\,", COMMA_MASK);
        String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
        for (int i = 0; i < parameter.length; i++) {
            parameter[i] = parameter[i].trim().replaceAll(COMMA_MASK, ",");
        }
        return parameter;
    }
    
    /**
     * Converts a string into a com.ulcjava.base.application.datatype.ULCAbstractDataType. The strings are mapped directly to the
     * corresponding constructor of the ULCDDatatype classes. Additionally parameters are used to set some other properties. The following
     * formats are allowed :
     * <ul>
     * <li>For {@link ULCDateDataType}
     * <ul>
     * <li>Date
     * <li>Date()
     * <li>Date(formatPattern)
     * <li>Date(formatPattern,twoDigitYearStartYear)
     * </ul>
     * where <b>formatPattern</b> defines the date format pattern to use, <b>twoDigitYearStartYear</b> is an integer and defines the year
     * that is used with {@link ULCDateDataType.setTwoDigitYearStart}.
     * <li>For {@link ULCNumberDataType}
     * <ul>
     * <li>Number
     * <li>Number()
     * <li>Number(formatPattern)
     * <li>Number(formatPattern,roundingMode)
     * <li>Number(formatPattern,man,max)
     * <li>Number(formatPattern,man,max,roundingMode)
     * <li>Number(formatPattern,localeString)
     * <li>Number(formatPattern,localeString,roundingMode)
     * <li>Number(formatPattern,localeString,min,max)
     * <li>Number(formatPattern,localeString,min,max,roundingMode)
     * </ul>
     * where <b>formatPattern</b> defines the decimal format pattern to use with a {@link DecimalFormat}. The comma used as grouping
     * separator placeholder must be escaped with a backslash. <b>roundingMode</b> is an integer and can have the valid values for
     * {@link ULCNumberDataType.setRoundingMode}. <b>min</b> and <b>max</b> are numbers converted by a {@link DoubleResourceConverter}.
     * <b>localeString</b> is of one of the formats language, language-country or language-country-variant that are used to create a
     * {@link Locale}.
     * <li>For {@link ULCPercentDataType}
     * <ul>
     * <li>Percent
     * <li>Percent()
     * <li>Percent(fractionalDigits)
     * <li>Percent(fractionalDigits,groupingUsed)
     * <li>Percent(fractionalDigits,roundingMode)
     * <li>Percent(fractionalDigits,roundingMode,groupingUsed)
     * <li>Percent(localeString,fractionalDigits)
     * <li>Percent(localeString,fractionalDigits,groupingUsed)
     * <li>Percent(localeString,fractionalDigits,roundingMode)
     * <li>Percent(localeString,fractionalDigits,roundingMode,groupingUsed)
     * </ul>
     * where <b>fractionalDigits</b> is an integer, <b>roundingMode</b> is an integer and can have the valid values for
     * {@link ULCPercentDataType.setRoundingMode}, <b>groupingUsed</b> is a boolean converted with {@link BooleanResourceConverter}.
     * <li>For {@link ULCRegularExpressionDataType}
     * <ul>
     * <li>Regexp or RegularExpression
     * <li>Regexp()
     * <li>Regexp(validateExpression)
     * <li>Regexp(validateExpression, inputFilterExpression)
     * </ul>
     * where <b>validateExpression</b> and <b>validateExpression</b> are strings that are used to create the
     * {@link ULCRegularExpressionDataType}. Any commas must be escaped with a backslash \.
     * <li>For {@link ULCStringDataType}
     * <ul>
     * <li>String
     * <li>String()
     * <li>String(maxLength)
     * <li>String(maxLength,mode)
     * <li>String(maxLength,mode,ignoreLongStringCompeletely)
     * </ul>
     * where <b>maxLength</b> is an integer, defines the number format pattern to use. <b>mode</b> is an integer and can have one of the
     * valid values for {@link ULCStringDataType.setMode}. <b>ignoreLongStringCompeletely</b> is a boolean converted with
     * {@link BooleanResourceConverter}.
     * </ul>
     */
    public static class ULCDataTypeConverter extends AbstractResourceConverter<IDataType> {
        
        enum ULCDataTypeTypes {
            STRING, REGEXP, REGULAREXPRESSION, PERCENT, DATE, NUMBER
        }
        
        protected ULCDataTypeConverter() {
            super(IDataType.class);
        }
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public ULCAbstractDataType doParseString(String s, ResourceMap r) throws ResourceConverterException {
            s = s.trim();
            ULCDataTypeTypes type;
            final int openIndex = s.indexOf('(');
            String typeString;
            String parameterString;
            if (openIndex < 0) {
                typeString = s;
                parameterString = "";
            } else {
                typeString = s.substring(0, openIndex).trim();
                if (s.charAt(s.length() - 1) != ')') {
                    throw new ResourceConverterException("Missing closing ) ", s);
                }
                parameterString = s.substring(openIndex + 1, s.length() - 1).trim();
            }
            try {
                type = ULCDataTypeTypes.valueOf(typeString.toUpperCase());
            } catch (Exception e) {
                throw new ResourceConverterException("Unknown data type type ", s, e);
            }
            switch (type) {
                case STRING:
                    return createStringDataType(s, r, parameterString);
                case REGEXP:
                case REGULAREXPRESSION:
                    return createRegexpDataType(s, r, parameterString);
                case PERCENT:
                    return createPercentDataType(s, r, parameterString);
                case DATE:
                    return createDateDataType(s, r, parameterString);
                case NUMBER:
                    return createNumberDataType(s, r, parameterString);
                default:
                    throw new ResourceConverterException("Unmapped data type ", s);
            }
            
        }
        
        private ULCNumberDataType createNumberDataType(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = splitIntoParameters(parameterString);
            
            boolean secondParameterIsNumber = false;
            if (parameter.length > 1) {
                try {
                    doubleParameter(parameter[1], r);
                    secondParameterIsNumber = true;
                } catch (Exception e) {
                }
            }
            if (secondParameterIsNumber) {
                switch (parameter.length) {
                    case 0:
                        return new ULCNumberDataType();
                    case 1:
                        return new ULCNumberDataType(parameter[0], new DecimalFormatSymbols());
                    case 2:
                        ULCNumberDataType dataType = new ULCNumberDataType(parameter[0], new DecimalFormatSymbols());
                        dataType.setRoundingMode(intParameter(parameter[1], r));
                        return dataType;
                        
                    case 3:
                        dataType = new ULCNumberDataType(parameter[0], new DecimalFormatSymbols());
                        dataType.setMin(doubleParameter(parameter[1], r));
                        dataType.setMax(doubleParameter(parameter[2], r));
                        return dataType;
                    case 4:
                        dataType = new ULCNumberDataType(parameter[0], new DecimalFormatSymbols());
                        dataType.setMin(doubleParameter(parameter[1], r));
                        dataType.setMax(doubleParameter(parameter[2], r));
                        dataType.setRoundingMode(intParameter(parameter[3], r));
                        return dataType;
                        
                    default:
                        throw new ResourceConverterException("Illegal number of parameter", s);
                }
                
            } else {
                switch (parameter.length) {
                    case 0:
                        return new ULCNumberDataType();
                    case 1:
                        return new ULCNumberDataType(parameter[0], new DecimalFormatSymbols());
                    case 2:
                        return new ULCNumberDataType(parameter[0], new DecimalFormatSymbols(parseLocale(parameter[1])));
                    case 3:
                        ULCNumberDataType dataType = new ULCNumberDataType(parameter[0],
                                new DecimalFormatSymbols(parseLocale(parameter[1])));
                        dataType.setRoundingMode(intParameter(parameter[2], r));
                        return dataType;
                        
                    case 4:
                        dataType = new ULCNumberDataType(parameter[0], new DecimalFormatSymbols(parseLocale(parameter[1])));
                        dataType.setMin(doubleParameter(parameter[2], r));
                        dataType.setMax(doubleParameter(parameter[3], r));
                        return dataType;
                    case 5:
                        dataType = new ULCNumberDataType(parameter[0], new DecimalFormatSymbols(parseLocale(parameter[1])));
                        dataType.setMin(doubleParameter(parameter[2], r));
                        dataType.setMax(doubleParameter(parameter[3], r));
                        dataType.setRoundingMode(intParameter(parameter[4], r));
                        return dataType;
                        
                    default:
                        throw new ResourceConverterException("Illegal number of parameter", s);
                }
            }
        }
        
        private ULCStringDataType createStringDataType(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
            switch (parameter.length) {
                case 0:
                    return new ULCStringDataType();
                case 1:
                    return new ULCStringDataType(intParameter(parameter[0], r));
                case 2:
                    return new ULCStringDataType(intParameter(parameter[0], r), intParameter(parameter[1], r));
                case 3:
                    final ULCStringDataType dataType = new ULCStringDataType(intParameter(parameter[0], r), intParameter(parameter[1], r));
                    dataType.setIgnoreLongStringCompeletely(booleanParameter(parameter[2], r));
                    return dataType;
                default:
                    throw new ResourceConverterException("Illegal number of parameter", s);
            }
        }
        
        private ULCDateDataType createDateDataType(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
            switch (parameter.length) {
                case 0:
                    return new ULCDateDataType();
                case 1:
                    return new ULCDateDataType(parameter[0]);
                case 2:
                    ULCDateDataType dataType = new ULCDateDataType(parameter[0]);
                    dataType.setTwoDigitYearStart(new GregorianCalendar(intParameter(parameter[1], r), 0, 1).getTime());
                    return dataType;
                default:
                    throw new ResourceConverterException("Illegal number of parameter", s);
            }
        }
        
        private ULCPercentDataType createPercentDataType(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
            if (parameter.length == 0) {
                return new ULCPercentDataType();
            }
            Locale locale = null;
            try {
                intParameter(parameter[0], r);
            } catch (Exception e) {
                locale = parseLocale(parameter[0]);
            }
            if (locale == null) {
                switch (parameter.length) {
                    case 1:
                        return new ULCPercentDataType(intParameter(parameter[0], r));
                    case 2:
                        ULCPercentDataType dataType = new ULCPercentDataType(intParameter(parameter[0], r));
                        try {
                            dataType.setRoundingMode(intParameter(parameter[1], r));
                        } catch (Exception e) {
                            dataType.setGroupingUsed(booleanParameter(parameter[1], r));
                        }
                        return dataType;
                    case 3:
                        dataType = new ULCPercentDataType(intParameter(parameter[0], r));
                        dataType.setRoundingMode(intParameter(parameter[1], r));
                        dataType.setGroupingUsed(booleanParameter(parameter[2], r));
                        return dataType;
                    default:
                        throw new ResourceConverterException("Illegal number of parameter", s);
                }
            } else {
                switch (parameter.length) {
                    case 1:
                        return new ULCPercentDataType(locale);
                    case 2:
                        return new ULCPercentDataType(locale, intParameter(parameter[1], r));
                    case 3:
                        ULCPercentDataType dataType = new ULCPercentDataType(locale, intParameter(parameter[1], r));
                        try {
                            dataType.setRoundingMode(intParameter(parameter[2], r));
                        } catch (Exception e) {
                            dataType.setGroupingUsed(booleanParameter(parameter[2], r));
                        }
                        return dataType;
                    case 4:
                        dataType = new ULCPercentDataType(locale, intParameter(parameter[1], r));
                        dataType.setRoundingMode(intParameter(parameter[2], r));
                        dataType.setGroupingUsed(booleanParameter(parameter[3], r));
                        return dataType;
                    default:
                        throw new ResourceConverterException("Illegal number of parameter", s);
                }
                
            }
        }
        
        private Locale parseLocale(String parameter) throws ResourceConverterException {
            String[] localeParams = parameter.split("-");
            switch (localeParams.length) {
                case 1:
                    return new Locale(localeParams[0]);
                case 2:
                    return new Locale(localeParams[0], localeParams[1]);
                case 3:
                    return new Locale(localeParams[0], localeParams[1], localeParams[2]);
                default:
                    throw new ResourceConverterException("Cannot create Locale", parameter);
            }
        }
        
        private ULCRegularExpressionDataType createRegexpDataType(String s, ResourceMap r, String parameterString)
                throws ResourceConverterException {
            String[] parameter = splitIntoParameters(parameterString);
            
            switch (parameter.length) {
                case 0:
                    return new ULCRegularExpressionDataType();
                case 1:
                    return new ULCRegularExpressionDataType(parameter[0]);
                case 2:
                    return new ULCRegularExpressionDataType(parameter[0], parameter[1]);
                default:
                    throw new ResourceConverterException("Illegal number of parameter", s);
            }
        }
    }
    

    /**
     * Converts a string into a com.ulcjava.base.application.border.ULCAbstractBorder. The strings are mapped directly to the corresponding
     * constructor of the ULCBorder classes. The following formats are allowed :
     * <ul>
     * <li>For {@link ULCBevelBorder}
     * <ul>
     * <li>Bevel(bevelType)
     * <li>Bevel(bevelType, highlight, shadow)
     * <li>Bevel(bevelType, highlightOuter, highlightInner, shadowOuter, shadowInner)
     * </ul>
     * where <b>bevelType</b> is an integer either ULCBevelBorder.RAISED (0) or ULCBevelBorder.LOWERED (1). The other parameter are colors
     * specified by using one of the formats defined by {@link ColorResourceConverter}.
     * <li>For {@link ULCCompoundBorder}
     * <ul>
     * <li>Compound(outsideBorder, insideBorder)
     * </ul>
     * where <b>outsideBorder</b> and <b>insideBorder</b> are borders specified by using one of the formats defined by
     * {@link ULCBorderConverter}.
     * <li>For {@link ULCEmptyBorder}
     * <ul>
     * <li>Empty(insets)
     * </ul>
     * where <b>insets</b> are specified using one of the formats defined by {@link InsetResourceConverter}
     * <li>For {@link ULCEtchedBorder}
     * <ul>
     * <li>Etched or Etched()
     * <li>Etched(etchType)
     * <li>Etched(highlight, shadow)
     * <li>Etched(etchType, highlight, shadow)
     * </ul>
     * where <b>etchType</b> is an integer either ULCEtchedBorder.RAISED (0) or ULCEtchedBorder.LOWERED (1). The other parameter are colors
     * specified by using one of the formats defined by {@link ColorResourceConverter}.
     * <li>For {@link ULCLineBorder}
     * <ul>
     * <li>Line(color)
     * <li>Line(color, thickness)
     * <li>Line(color, thickness, roundedCorners)
     * </ul>
     * where <b>color</b> is specified by using one of the formats defined by {@link ColorResourceConverter}, <b>thickness</b> is an integer
     * and <b>roundedCorners</b> a boolean specified by using one of the formats defined by {@link BooleanResourceConverter}.
     * <li>For {@link ULCMatteBorder}
     * <ul>
     * <li>Matte(insets, color)
     * <li>Matte(insets, icon)
     * </ul>
     * where <b>insets</b> are specified using one of the formats defined by {@link InsetResourceConverter}, <b>color</b> by using one of
     * the formats defined by {@link ColorResourceConverter} and <b>icon</b> an ULCIcon converted using a {@link ULCIconConverter}
     * <li>For {@link ULCResourceBorder}
     * <ul>
     * <li>Resource(resourceKey)
     * </ul>
     * where <b>resourceKey</b> is a string that is used a resource key for the border.
     * <li>For {@link ULCSoftBevelBorder}
     * <ul>
     * <li>SoftBevel(bevelType)
     * <li>SoftBevel(bevelType, highlight, shadow)
     * <li>SoftBevel(bevelType, highlightOuter, highlightInner, shadowOuter, shadowInner)
     * </ul>
     * where <b>bevelType</b> is an integer either ULCBevelBorder.RAISED (0) or ULCBevelBorder.LOWERED (1). The other parameter are colors
     * specified by using one of the formats defined by {@link ColorResourceConverter}.
     * <li>For {@link ULCTitledBorder}
     * <ul>
     * <li>Titled(titleText)
     * <li>Titled(border,titleText)
     * <li>Titled(border,titleText,titleJustification,titlePosition)
     * <li>Titled(border,titleText,titleJustification,titlePosition,titleFont)
     * <li>Titled(border,titleText,titleJustification,titlePosition,titleFont,titleColor)
     * </ul>
     * where <b>titleText</b> is a string, that is set as title. Any commas must be escaped with a backslash \. <b>border</b> is a border
     * specified by using one of the formats defined by {@link ULCBorderConverter}. <b>titleJustification</b> and <b>titlePosition</b> are
     * integers that must match one of the corresponding constants defined in {@link ULCTitledBorder}. <b>titleFont</b> is converted into an
     * {@link Font} using a {@link FontResourceConverter} and <b>titleColor</b> is converted with a {@link ColorResourceConverter} into a
     * {@link Color}.
     * </ul>
     */
    public static class ULCBorderConverter extends AbstractResourceConverter<ULCAbstractBorder> {
        

        enum ULCBorderTypes {
            RESOURCE, EMPTY, LINE, ETCHED, MATTE, BEVEL, SOFTBEVEL, TITLED, COMPOUND
        }
        
        protected ULCBorderConverter() {
            super(ULCAbstractBorder.class);
        }
        
        /**
         * @see com.ulcjava.applicationframework.application.AbstractResourceConverter#doParseString(java.lang.String,
         *      com.ulcjava.applicationframework.application.ResourceMap)
         */
        @Override
        public ULCAbstractBorder doParseString(String s, ResourceMap r) throws ResourceConverterException {
            s = s.trim();
            ULCBorderTypes type;
            final int openIndex = s.indexOf('(');
            String typeString;
            String parameterString;
            if (openIndex < 0) {
                typeString = s;
                parameterString = "";
            } else {
                typeString = s.substring(0, openIndex).trim();
                if (s.charAt(s.length() - 1) != ')') {
                    throw new ResourceConverterException("Missing closing ) ", s);
                }
                parameterString = s.substring(openIndex + 1, s.length() - 1).trim();
            }
            try {
                type = ULCBorderTypes.valueOf(typeString.toUpperCase());
            } catch (Exception e) {
                throw new ResourceConverterException("Unknown border type ", s, e);
            }
            switch (type) {
                case RESOURCE:
                    if (parameterString.length() <= 0) {
                        throw new ResourceConverterException("Parameters missing", s);
                    }
                    return new ULCResourceBorder(parameterString);
                case EMPTY:
                    return new ULCEmptyBorder(insetsParameter(parameterString, r));
                case LINE:
                    return createLineBorder(s, r, parameterString);
                case ETCHED:
                    return createEtchedBorder(s, r, parameterString);
                case MATTE:
                    return createMatteBorder(s, r, parameterString);
                case BEVEL:
                    return createBevelBorder(s, r, parameterString);
                case SOFTBEVEL:
                    return createSoftBevelBorder(s, r, parameterString);
                case TITLED:
                    return createTitledBorder(s, r, parameterString);
                case COMPOUND:
                    return createCompoundBorder(s, r, parameterString);
                default:
                    throw new ResourceConverterException("Unmapped border type ", s);
            }
            
        }
        
        private ULCSoftBevelBorder createSoftBevelBorder(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
            switch (parameter.length) {
                case 1:
                    return new ULCSoftBevelBorder(intParameter(parameter[0], r));
                case 3:
                    return new ULCSoftBevelBorder(intParameter(parameter[0], r), colorParameter(parameter[1], r), colorParameter(
                            parameter[2], r));
                case 5:
                    return new ULCSoftBevelBorder(intParameter(parameter[0], r), colorParameter(parameter[1], r), colorParameter(
                            parameter[2], r), colorParameter(parameter[3], r), colorParameter(parameter[4], r));
                    
                default:
                    throw new ResourceConverterException("Illegal number of parameters ", s);
            }
        }
        
        private ULCBevelBorder createBevelBorder(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
            switch (parameter.length) {
                case 1:
                    return new ULCBevelBorder(intParameter(parameter[0], r));
                case 3:
                    return new ULCBevelBorder(intParameter(parameter[0], r), colorParameter(parameter[1], r), colorParameter(parameter[2],
                            r));
                case 5:
                    return new ULCBevelBorder(intParameter(parameter[0], r), colorParameter(parameter[1], r), colorParameter(parameter[2],
                            r), colorParameter(parameter[3], r), colorParameter(parameter[4], r));
                    
                default:
                    throw new ResourceConverterException("Illegal number of parameters ", s);
            }
        }
        
        private ULCTitledBorder createTitledBorder(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            if (parameterString == null || parameterString.trim().length() == 0) {
                throw new ResourceConverterException("No parameter", s);
            }
            
            final int openIndex = parameterString.indexOf('(');
            String borderParam = null;
            String followingParam = parameterString;
            ULCAbstractBorder borderParameter = null;
            String[] parameter;
            if (openIndex < 0 || s.indexOf(',') < openIndex) {
                String[] param = splitIntoParameters(followingParam);
                try {
                    borderParameter = borderParameter(param[0], r);
                    parameter = new String[param.length - 1];
                    System.arraycopy(param, 1, parameter, 0, parameter.length);
                } catch (Exception e) {
                    parameter = param;
                }
            } else {
                String typeString = parameterString.substring(0, openIndex).trim();
                int endIndex = -1;
                try {
                    ULCBorderTypes.valueOf(typeString.toUpperCase());
                    endIndex = findClosing(openIndex, parameterString);
                    borderParam = parameterString.substring(0, endIndex + 1);
                } catch (Exception e) {
                }
                if (borderParam != null) {
                    borderParameter = borderParameter(borderParam, r);
                    int k = parameterString.indexOf(',', endIndex + 1);
                    if (k < 0) {
                        throw new ResourceConverterException("Illegal number of parameter", s);
                    }
                    followingParam = parameterString.substring(k + 1, parameterString.length()).trim();
                }
                parameter = splitIntoParameters(followingParam);
            }
            
            switch (parameter.length) {
                case 1:
                    if (borderParameter == null) {
                        return new ULCTitledBorder(parameter[0]);
                    } else {
                        return new ULCTitledBorder(borderParameter, parameter[0]);
                    }
                case 3:
                    return new ULCTitledBorder(borderParameter, parameter[0], intParameter(parameter[1], r), intParameter(parameter[2], r));
                case 4:
                    return new ULCTitledBorder(borderParameter, parameter[0], intParameter(parameter[1], r), intParameter(parameter[2], r),
                            fontParameter(parameter[3], r));
                case 5:
                    return new ULCTitledBorder(borderParameter, parameter[0], intParameter(parameter[1], r), intParameter(parameter[2], r),
                            fontParameter(parameter[3], r), colorParameter(parameter[4], r));
                default:
                    throw new ResourceConverterException("Illegal number of parameter", s);
            }
            
        }
        
        private ULCCompoundBorder createCompoundBorder(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            if (parameterString == null || parameterString.trim().length() == 0) {
                throw new ResourceConverterException("No parameter", s);
            }
            
            final int openIndex = parameterString.indexOf('(');
            int commaIndex = parameterString.indexOf(',');
            if (commaIndex < 0) {
                throw new ResourceConverterException("no border parameter found ", s);
            }
            String firstParam;
            if (openIndex > 0 && openIndex < commaIndex) {
                int endIndex = findClosing(openIndex, parameterString);
                firstParam = parameterString.substring(0, endIndex + 1);
                commaIndex = parameterString.indexOf(',', endIndex + 1);
                if (commaIndex < 0) {
                    throw new ResourceConverterException("Illegal number of parameter", s);
                }
            } else {
                firstParam = parameterString.substring(0, commaIndex);
            }
            
            String secondParam = parameterString.substring(commaIndex + 1, parameterString.length()).trim();
            return new ULCCompoundBorder(borderParameter(firstParam, r), borderParameter(secondParam, r));
        }
        
        private int findClosing(int openIndex, String s) {
            int index = openIndex + 1;
            while (index < s.length() && s.charAt(index) != ')') {
                if (s.charAt(index) == '(') {
                    index = findClosing(index, s) + 1;
                } else {
                    index++;
                }
            }
            return index;
        }
        
        private ULCMatteBorder createMatteBorder(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
            switch (parameter.length) {
                case 2:
                case 3:
                case 5:
                    String insetsParamString = parameterString.substring(0, parameterString.lastIndexOf(','));
                    try {
                        return new ULCMatteBorder(insetsParameter(insetsParamString, r), colorParameter(parameter[parameter.length - 1], r));
                    } catch (Exception e) {
                        return new ULCMatteBorder(insetsParameter(insetsParamString, r), iconParameter(parameter[parameter.length - 1], r));
                    }
                default:
                    throw new ResourceConverterException("Illegal number of parameter", s);
            }
        }
        
        private ULCEtchedBorder createEtchedBorder(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            String[] parameter = parameterString.length() == 0 ? new String[0] : parameterString.split(",");
            switch (parameter.length) {
                case 0:
                    return new ULCEtchedBorder();
                case 1:
                    return new ULCEtchedBorder(intParameter(parameter[0], r));
                case 2:
                    return new ULCEtchedBorder(colorParameter(parameter[0], r), colorParameter(parameter[1], r));
                case 3:
                    return new ULCEtchedBorder(intParameter(parameter[0], r), colorParameter(parameter[1], r), colorParameter(parameter[2],
                            r));
                default:
                    throw new ResourceConverterException("Illegal number of parameter", s);
            }
        }
        
        private ULCLineBorder createLineBorder(String s, ResourceMap r, String parameterString) throws ResourceConverterException {
            final String[] parameter = parameterString.split(",");
            Color color = Color.black;
            int thickness = 1;
            boolean round = false;
            switch (parameter.length) {
                case 3:
                    round = booleanParameter(parameter[2], r);
                case 2:
                    thickness = intParameter(parameter[1], r);
                case 1:
                    try {
                        color = colorParameter(parameter[0], r);
                    } catch (ResourceConverterException e) {
                        if (parameter.length == 1) {
                            thickness = intParameter(parameter[0], r);
                        } else {
                            throw e;
                        }
                    }
                    break;
                default:
                    throw new ResourceConverterException("Illegal number of parameter", s);
            }
            
            return new ULCLineBorder(color, thickness, round);
        }
    }
    
    protected int intParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> intConverter = forType(Integer.class);
        return (Integer)intConverter.parseString(parameter, r);
    }
    
    protected double doubleParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> doubleConverter = forType(Double.class);
        return (Double)doubleConverter.parseString(parameter, r);
    }
    
    protected Insets insetsParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> insetsConverter = forType(Insets.class);
        return (Insets)insetsConverter.parseString(parameter, r);
    }
    
    protected ULCAbstractBorder borderParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> borderConverter = forType(ULCAbstractBorder.class);
        return (ULCAbstractBorder)borderConverter.parseString(parameter, r);
    }
    
    protected ULCIcon iconParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> iconConverter = forType(ULCIcon.class);
        return (ULCIcon)iconConverter.parseString(parameter, r);
    }
    
    protected Font fontParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> fontConverter = forType(Font.class);
        return (Font)fontConverter.parseString(parameter, r);
    }
    
    protected Color colorParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> colorConverter = forType(Color.class);
        return (Color)colorConverter.parseString(parameter, r);
    }
    
    protected Boolean booleanParameter(String parameter, ResourceMap r) throws ResourceConverterException {
        final IResourceConverter<?> booleanConverter = forType(Boolean.class);
        return (Boolean)booleanConverter.parseString(parameter, r);
    }
    
}
