/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotates a method to be called from the actionPerformed method of an {@link ApplicationAction}. The
 * {@link ApplicationAction} is created by an {@link ApplicationActionMap} which is retrieved by calling an
 * {@link ActionManager}'s <code>getActionMap(Class<? super T>, T)</code> method. The method's name acts as the name
 * for the action unless the annotation has a <code>name</code> parameter which specifies the name. The action's
 * defines the key in the {@link ApplicationActionMap}.
 * <p>
 * The {@link Application} typically uses an &#064;Action like this
 * 
 * <pre>
 * class MyApplication extends Application {
 * 
 *      &#064;Action 
 *      public void doit() {
 *           // do what has to be done.
 *      }
 *  
 *      ... 
 *      //within some method retrieve the action and set it on a button 
 *      ApplicationActionMap actions = getContext().getActionMap();
 *      ULCButton doitButton = new ULCButton(actions.get(&quot;doit&quot;));
 *      ...
 *      
 * }
 * </pre>
 * 
 * The name with the suffix <i>.Action</i> is used with a {@link ResourceMap} to initialize the properties of the
 * {@link ApplicationAction}. The resource key for the <i>doit</i> action's IAction.NAME property is
 * <code>doit.Action.text</code>. See {@link ApplicationAction} for the complete list of resources that are mapped to
 * action properties.
 * <p>
 * The annotated method can have zero or more parameters. The parameter must be of one of the following types:
 * <ul>
 * <li>com.ulcjava.base.application.event.ActionEvent</li>
 * <li>com.ulcjava.base.application.IAction</li>
 * <li>com.ulcjava.base.application.AbstractAction</li>
 * <li>com.ulcjava.applicationframework.application.ResourceMap</li>
 * <li>com.ulcjava.applicationframework.application.ApplicationContext</li>
 * <li>com.ulcjava.applicationframework.application.Application</li>
 * </ul>
 * 
 * @see ApplicationAction
 * @see ApplicationActionMap
 * @see ActionManager
 */
@Documented
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface Action {
    /**
     * specifies the action's name (instead of using the annotated method's name).
     */
    String name() default "";

    /**
     * specifies the boolean bound property from the annotated method's class that is used to enable and disable the
     * action.
     */
    String enabledProperty() default "";

}
