/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import java.io.Serializable;

/**
 * The ActionManager class knows how to create {@link ApplicationActionMap}s for a class hierarchy.
 */
public class ActionManager implements Serializable {

    private final ApplicationContext fContext;
    private ApplicationActionMap fApplicationActionMap;

    /**
     * Creates an ActionManager for the given {@link ApplicationContext}.
     * 
     * @param context {@link ApplicationContext} that provides the {@link ResourceMap}s for the created
     *            {@link ApplicationActionMap}s. Must not be <code>null</code>.
     */
    protected ActionManager(ApplicationContext context) {
        if (context == null) {
            throw new IllegalArgumentException("context must not be null");
        }
        fContext = context;
    }

    /**
     * The default {@link ApplicationActionMap} contains {@link ApplicationAction}s created with the annotated methods
     * for the class hierarchy from the class of the running application up to the Application.class.
     * 
     * @return the application's default action map.
     */
    public ApplicationActionMap getActionMap() {
        if (fApplicationActionMap == null) {
            fApplicationActionMap = createActionMapChain(getContext().getApplication(), getContext().getApplication()
                    .getClass(), Application.class, getContext().getResourceMap());
        }
        return fApplicationActionMap;
    }

    /**
     * Creates a {@link ApplicationActionMap} that contains {@link ApplicationAction}s created with the annotated
     * methods for the class hierarchy from the given actionsObject's class up to the given actionClass.
     * 
     * @param <T> The generic type representing the type of the action object.
     * @param actionsClass superclass of the <code>actionsObject</code> up to which &#064;Action annotated methods are
     *            added to the created {@link ApplicationActionMap}.
     * @param actionsObject the object on which methods are called when the actions are triggered.
     * @return the created {@link ApplicationActionMap}.
     * @see Action
     * @see ApplicationAction
     * @see ApplicationActionMap
     */
    public <T> ApplicationActionMap getActionMap(Class<? super T> actionsClass, T actionsObject) {

        Class<? extends T> class1 = (Class<? extends T>)actionsObject.getClass();
        ApplicationActionMap actionMapChain = createActionMapChain(actionsObject, actionsObject.getClass(),
                actionsClass, getContext().getResourceMap(class1, actionsClass));
        ApplicationActionMap rootMap = actionMapChain;
        while (rootMap.getParentMap() != null) {
            rootMap = rootMap.getParentMap();
        }
        rootMap.setParentMap(getActionMap());
        return actionMapChain;
    }

    private ApplicationActionMap createActionMapChain(Object actionsObject, Class<?> startClass, Class<?> endClass,
            ResourceMap resourceMap) {
        ;
        if (!endClass.isAssignableFrom(startClass)) {
            throw new IllegalArgumentException(endClass.toString() + " must be super class of " + startClass.toString());
        }

        ApplicationActionMap result;
        ApplicationActionMap childMap;
        Class<?> clazz = startClass;
        ApplicationActionMap applicationActionMap = result = childMap = new ApplicationActionMap(getContext(), clazz,
                actionsObject, resourceMap);
        while (clazz != endClass) {
            clazz = clazz.getSuperclass();
            applicationActionMap = new ApplicationActionMap(getContext(), clazz, actionsObject, resourceMap);
            childMap.setParentMap(applicationActionMap);
            childMap = applicationActionMap;
        }

        return result;
    }

    /**
     * @return the {@link ApplicationContext} that provides the {@link ResourceMap}s for the created
     *         {@link ApplicationActionMap}s.
     */
    protected ApplicationContext getContext() {
        return fContext;
    }
}
