/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import com.ulcjava.base.application.event.IRoundTripListener;

import java.io.Serializable;
import java.security.Principal;
import java.util.Locale;

/**
 * Helper class of the {@link Application} that manages and provides access to global objects like resources and actions. The
 * {@code ApplicationContext} can be accessed from anywhere in an {@link Application} by calling:
 * 
 * <pre>
 * Application.getInstance().getContext();
 * </pre>
 * 
 * <p>
 * Moreover this class provides convenience methods to access the application's runtime environment via
 * {@link com.ulcjava.base.application.ApplicationContext}.
 * </p>
 */
public class ApplicationContext extends AbstractBean implements Serializable {
    
    /**
     * The code that denotes that the server container type is not set properly and therefore undefined
     * 
     * @see #getServerContainerType
     */
    public static final int UNDEFINED = com.ulcjava.base.application.ApplicationContext.UNDEFINED;
    /**
     * The code that denotes the application is deployed in a Servlet container
     * 
     * @see #getServerContainerType
     */
    public static final int SERVLET = com.ulcjava.base.application.ApplicationContext.SERVLET;
    /**
     * The code that denotes the application is deployed in an EJB container
     * 
     * @see #getServerContainerType
     */
    public static final int EJB = com.ulcjava.base.application.ApplicationContext.EJB;
    /**
     * The code that denotes the application is deployed in a Local container
     * 
     * @see #getServerContainerType
     */
    public static final int LOCAL = com.ulcjava.base.application.ApplicationContext.LOCAL;
    /**
     * The code that denotes the application is deployed in a Development container
     * 
     * @see #getServerContainerType
     */
    public static final int DEVELOPMENT = com.ulcjava.base.application.ApplicationContext.DEVELOPMENT;
    

    private final Application fApplication;
    private ActionManager fActionManager;
    private ResourceManager fResourceManager;
    private Locale fLocale;
    
    /**
     * Creates an ApplicationContext for the given {@link Application}.
     * 
     * @param application the context is for.
     */
    protected ApplicationContext(Application application) {
        if (application == null) {
            throw new IllegalArgumentException("application must not be null");
        }
        fApplication = application;
    }
    
    /**
     * Creates the {@link ResourceManager} for this context. Subclasses should overwrite this if they want to use a {@link ResourceManager}
     * subclass.
     * 
     * @return the {@link ResourceManager} created for the context.
     */
    protected ResourceManager createResourceManager() {
        return new ResourceManager(this);
    }
    
    
    /**
     * Creates the {@link ActionManager} for this context. Subclasses should overwrite this if they want to use a {@link ActionManager}
     * subclass.
     * 
     * @return the {@link ActionManager} created for the context.
     */
    protected ActionManager createActionManager() {
        return new ActionManager(this);
    }
    
    /**
     * @return the {@link Application} which this is the context for.
     */
    public Application getApplication() {
        return fApplication;
    }
    
    /**
     * To retrieve an ApplicationActionMap, the <code>getActionMap</code> methods should be used.
     * 
     * @return the {@link ActionManager} that manages {@link ApplicationActionMap}s and {@link ApplicationAction}s.
     */
    public ActionManager getActionManager() {
        if (fActionManager == null) {
            fActionManager = createActionManager();
        }
        return fActionManager;
    }
    
    /**
     * Retrieves the application's default action map.
     * 
     * @return the {@link ApplicationActionMap} for the {@link Application} class hierarchy, from the running application's class up to
     *         Application.class.
     */
    public final ApplicationActionMap getActionMap() {
        return getActionManager().getActionMap();
    }
    
    /**
     * Retrieves the action map for the given class hierarchy.
     * 
     * @param <T> The generic type representing the type upto which the application's action map hierarchy is built
     * @param <S> The generic type representing the type of the Action object
     * @param actionsClass the class up to which the {@code ApplicationActionMap} hierarchy is built.
     * @param actionsObject the object on which the actions invoke the methods on.
     * @return the {@link ApplicationActionMap} for the <code>actionsClass</code> class hierarchy, from the <code>actionsObject</code>'s
     *         class up to <code>actionsClass</code>.
     */
    public final <T, S extends T> ApplicationActionMap getActionMap(Class<T> actionsClass, S actionsObject) {
        return getActionManager().getActionMap(actionsClass, actionsObject);
    }
    
    /**
     * Retrieves the action map for the given object.
     * 
     * @param <T> The generic type representing the type of the Action object
     * @param actionsObject the object that the actions are calling the methods on.
     * @return the {@link ApplicationActionMap} for the <code>actionsObject</code>'s class .
     */
    public final <T> ApplicationActionMap getActionMap(T actionsObject) {
        if (actionsObject == null) {
            throw new IllegalArgumentException("actionsObject must not be null");
        }
        Class<T> class1 = (Class<T>)actionsObject.getClass();
        return getActionManager().getActionMap(class1, actionsObject);
    }
    
    /**
     * Gets the resource manager. To retrieve a {@code ResourceMap}, the <code>getResourceMap</code> methods should be used.
     * 
     * @return the {@link ResourceManager} that manages {@link ResourceMap}s.
     */
    public ResourceManager getResourceManager() {
        if (fResourceManager == null) {
            fResourceManager = createResourceManager();
        }
        return fResourceManager;
    }
    
    /**
     * Retrieves the application's default resource map.
     * 
     * @return the {@link ResourceMap} for the {@link Application} class hierarchy, from the running application's class up to
     *         Application.class.
     * @see ResourceManager#getResourceMap()
     */
    public final ResourceMap getResourceMap() {
        return getResourceManager().getResourceMap();
    }
    
    /**
     * Retrieves the resource map for the given class.
     * 
     * @param clazz the class that the <code>ResourceMap</code> is retrieved for.
     * @return the {@link ResourceMap} for the given class. The application's default ResourceMap is set as parent of this ResourceMap.
     * @see ResourceManager#getResourceMap(Class)
     */
    public final ResourceMap getResourceMap(Class<?> clazz) {
        return getResourceManager().getResourceMap(clazz);
    }
    
    /**
     * Retrieves the resource map for the given class hierarchy.
     * 
     * @param <T> The generic type representing the type of the class for which resource map is retrieved.
     * @param startClass the class that the <code>ResourceMap</code> is retrieved for.
     * @param endClass the <code>ResourceMap</code> hierarchy is built up to this class.
     * @return the <code>ResourceMap</code> for the given class hierarchy from the <code>startClass</code> class up to the
     *         <code>endClass</code> class. The parent of the <code>endClass</code>'s resource map is set to the application default
     *         ResourceMap.
     * @see ResourceManager#getResourceMap(Class, Class)
     */
    public final <T> ResourceMap getResourceMap(Class<? extends T> startClass, Class<T> endClass) {
        return getResourceManager().getResourceMap(startClass, endClass);
    }
    
    /**
     * Retrieves the resource map for the given class hierarchy. The given resource map is set as parent to the ResourceMap of the
     * <code>endClass</code>.
     * 
     * @param <T> the base class of the class hierarchy.
     * @param startClass the class that the <code>ResourceMap</code> is retrieved for. Must not be <code>null</code>.
     * @param endClass the <code>ResourceMap</code> hierarchy is built up to this class. Must not be <code>null</code>.
     * @param parent the ResourceMap that is set as parent of the <code>endClass</code>'s resource map. If <code>null</code> the
     *            returned ResourceMap hierarchy has no parent ResourceMap
     * @return the <code>ResourceMap</code> for the given class hierarchy from the <code>startClass</code> class up to the
     *         <code>endClass</code> class. The parent of the <code>endClass</code>'s resource map is set to the given ResourceMap.
     * @throws IllegalArgumentException if <code>startClass</code> or <code>endClass</code> is <code>null</code>.
     * @see ResourceManager#getResourceMap(Class, Class, ResourceMap)
     */
    public <T> ResourceMap getResourceMap(Class<? extends T> startClass, Class<T> endClass, ResourceMap parent) {
        return getResourceManager().getResourceMap(startClass, endClass, parent);
    }
    
    /**
     * Returns a <code>Principal</code> object containing the name of the current authenticated user (optional operation). If the user has
     * not been authenticated, the method returns null. This methods forwards the call to the server container.
     * 
     * @return a <code>Principal</code> containing the name of the user making this request; null if the user has not been authenticated
     * @throws UnsupportedOperationException if the <code>getUserPrincipal</code> method is not supported by the server container.
     * @see Principal
     */
    public static Principal getUserPrincipal() {
        return com.ulcjava.base.application.ApplicationContext.getUserPrincipal();
    }
    
    /**
     * Returns a boolean indicating whether the authenticated user is included in the specified logical "role" (optional operation). If the
     * user has not been authenticated, the method returns false.
     * 
     * @param role a String specifying the name of the role
     * @return a boolean indicating whether the user making this request belongs to a given role; false if the user has not been
     *         authenticated
     * @throws UnsupportedOperationException if the <code>getUserPrincipal</code> method is not supported by the server container.
     */
    public static boolean isUserInRole(String role) {
        return com.ulcjava.base.application.ApplicationContext.isUserInRole(role);
    }
    
    /**
     * Returns the names of all init parameters specified for the server container.
     * 
     * @return the names of all init parameters
     */
    public static String[] getInitParameterNames() {
        return com.ulcjava.base.application.ApplicationContext.getInitParameterNames();
    }
    
    /**
     * Returns the init parameter indicated by the specified key specified the server container.
     * 
     * @param key the name of the init parameter
     * @return the string value of the init parameter, or <code>null</code> if there is no init parameter with that key
     */
    public static String getInitParameter(String key) {
        return com.ulcjava.base.application.ApplicationContext.getInitParameter(key);
    }
    
    /**
     * Returns the code denoting the server container type.
     * 
     * @return the code denoting the server container type
     */
    public static int getServerContainerType() {
        return com.ulcjava.base.application.ApplicationContext.getServerContainerType();
    }
    
    /**
     * Returns the names of all application attributes.
     * 
     * @return the names of all application attributes
     */
    public static String[] getAttributeNames() {
        return com.ulcjava.base.application.ApplicationContext.getAttributeNames();
    }
    
    /**
     * Returns the application attribute indicated by the specified key.
     * 
     * @param key the name of the application attribute
     * @return the application attribute, or <code>null</code> if there is no application attribute with that key
     */
    public static Object getAttribute(String key) {
        return com.ulcjava.base.application.ApplicationContext.getAttribute(key);
    }
    
    /**
     * Sets the application attribute indicated by the specified key. The attribute is added non-transient and thus will be serialized
     * whenever the <code>ULCSession</code> is serialized (e.g. during session replication).
     * 
     * @param key the name of the application attribute
     * @param value the new value of the application attribute
     * @see #setAttribute(String, Object, boolean)
     */
    public static void setAttribute(String key, Object value) {
        com.ulcjava.base.application.ApplicationContext.setAttribute(key, value);
    }
    
    /**
     * Sets the application attribute indicated by the specified key. If <code>isTransient</code> is <code>false</code>, the attribute
     * is added non-transient and thus will be serialized whenever the <code>ULCSession</code> is serialized (e.g. during session
     * replication). If <code>isTransient</code> is <code>true</code>, the attribute is added transient and thus will not be serialized
     * whenever the <code>ULCSession</code> is serialized. Therefore, the attribute no longer exists in the deserialized instance of
     * <code>ULCSession</code> and must be re-registered with the <code>ULCSession</code> manually.
     * 
     * @param key the name of the application attribute
     * @param value the new value of the application attribute
     * @param isTransient true if the attribute is transient, false if the attribute is persistent
     */
    public static void setAttribute(String key, Object value, boolean isTransient) {
        com.ulcjava.base.application.ApplicationContext.setAttribute(key, value, isTransient);
    }
    
    /**
     * Removes the application attribute indicated by the specified key.
     * 
     * @param key the name of the application attribute
     */
    public static void removeAttribute(String key) {
        com.ulcjava.base.application.ApplicationContext.removeAttribute(key);
    }
    
    /**
     * This method adds a round trip listener.
     * 
     * @param listener the listener to add
     */
    public static void addRoundTripListener(IRoundTripListener listener) {
        com.ulcjava.base.application.ApplicationContext.addRoundTripListener(listener);
    }
    
    /**
     * This method removes a round trip listener.
     * 
     * @param listener the listener to remove
     */
    public static void removeRoundTripListener(IRoundTripListener listener) {
        com.ulcjava.base.application.ApplicationContext.removeRoundTripListener(listener);
    }
    
    /**
     * Returns an array of all the round trip listeners registered on the current session.
     * 
     * @return all of this current session's <code>IRoundTripListener</code>s or an empty array if no round trip listeners are currently
     *         registered
     * @see #addRoundTripListener(IRoundTripListener)
     * @see #removeRoundTripListener(IRoundTripListener)
     */
    public static IRoundTripListener[] getRoundTripListeners() {
        return com.ulcjava.base.application.ApplicationContext.getRoundTripListeners();
    }
    
    /**
     * Returns the title of the implementation.
     * 
     * @return the title of the implementation
     */
    public static String getImplementationTitle() {
        return com.ulcjava.base.application.ApplicationContext.getImplementationTitle();
    }
    
    /**
     * Returns the vendor of the implementation.
     * 
     * @return the vendor of the implementation
     */
    public static String getImplementationVendor() {
        return com.ulcjava.base.application.ApplicationContext.getImplementationVendor();
    }
    
    /**
     * Returns the implementation version.
     * 
     * @return the implementation version
     */
    public static String getImplementationVersion() {
        return com.ulcjava.base.application.ApplicationContext.getImplementationVersion();
    }
    
    /**
     * Returns the title of the specification.
     * 
     * @return the title of the specification
     */
    public static String getSpecificationTitle() {
        return com.ulcjava.base.application.ApplicationContext.getSpecificationTitle();
    }
    
    /**
     * Returns the specification vendor.
     * 
     * @return the specification vendor
     */
    public static String getSpecificationVendor() {
        return com.ulcjava.base.application.ApplicationContext.getSpecificationVendor();
    }
    
    /**
     * Returns the specification version.
     * 
     * @return the specification version
     */
    public static String getSpecificationVersion() {
        return com.ulcjava.base.application.ApplicationContext.getSpecificationVersion();
    }
}
