/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import com.ulcjava.base.application.ClientContext;
import com.ulcjava.base.application.ULCFrame;
import com.ulcjava.base.application.ULCRootPane;

/**
 * View that uses a {@code ULCFrame}.
 * <p>
 * The frame's name is set to "main.Frame" thus allowing its properties to be injected with resource's prefixed by <code>main.Frame.</code>.
 * <p>
 * If the application is using a LookAndFeel that supports window decorations, the frame is configured to use the window decorations from
 * the L&F. This is not true for the "Metal" LookAndFeel, because this is the default on Windows and we assume that most of the users would
 * like to keep the Windows-style decoration.
 * 
 * @see SingleFrameApplication
 */
public class FrameView extends View {
    
    private ULCFrame fFrame;
    
    /**
     * Creates a frame for the given {@link Application}.
     * 
     * @param application that the frame view is for.
     */
    public FrameView(Application application) {
        super(application);
    }
    
    /**
     * @see com.ulcjava.applicationframework.application.View#getRootPane()
     */
    @Override
    public ULCRootPane getRootPane() {
        return getFrame();
    }
    
    /**
     * @return the {@link ULCFrame} that acts as root pane. The name of the frame is set to "<b>main.Frame</b>". This can be used to inject
     *         properties on the frame.
     */
    public ULCFrame getFrame() {
        if (fFrame == null) {
            String title = getContext().getResourceMap().getString("Application.title");
            ULCFrame frame = new ULCFrame(title);
            frame.setName("main.Frame");
            setFrame(frame);
            
            boolean metalLookAndFeel = "Metal".equals(ClientContext.getLookAndFeelName());
            if (!metalLookAndFeel && ClientContext.getLookAndFeelSupportsWindowDecorations()) {
                frame.setUndecorated(true);
                frame.setWindowDecorationStyle(ULCFrame.FRAME);
            }
        }
        return fFrame;
    }
    
    private void setFrame(ULCFrame frame) {
        fFrame = frame;
    }
    
}
