/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * The {@code ResourceManager} class knows how to create {@link ResourceMap}s for a class hierarchy.
 */
public class ResourceManager implements Serializable {
    
    private final ApplicationContext fContext;
    private ResourceMap fApplicationResourceMap;
    private Class<? extends Application> fApplicationClass;
    
    /**
     * Creates a {@code ResourceManager} for the given {@link ApplicationContext}.
     * 
     * @param applicationClass the class of the application. Must not be <code>null</code>.
     */
    public ResourceManager(Class<? extends Application> applicationClass) {
        if (applicationClass == null) {
            throw new IllegalArgumentException("applicationClass must not be null");
        }
        fContext = null;
        fApplicationClass = applicationClass;
    }
    
    /**
     * Creates a {@code ResourceManager} for the given {@link ApplicationContext}.
     * 
     * @param applicationContext the {@link ApplicationContext} of the running {@link Application}. Must not be <code>null</code>.
     */
    public ResourceManager(ApplicationContext applicationContext) {
        if (applicationContext == null) {
            throw new IllegalArgumentException("applicationContext must not be null");
        }
        fContext = applicationContext;
        fApplicationClass = applicationContext.getApplication().getClass();
    }
    
    /**
     * @return the {@link ApplicationContext} of the running {@link Application}.
     */
    protected ApplicationContext getContext() {
        return fContext;
    }
    
    /**
     * Retrieves the application's default map that is built for the {@link Application} class hierarchy, from the class of the running
     * application up to the Application.class.
     * 
     * @return the default resource map for the application.
     */
    public ResourceMap getResourceMap() {
        if (fApplicationResourceMap == null) {
            fApplicationResourceMap = getResourceMap(fApplicationClass, Application.class, null);
        }
        return fApplicationResourceMap;
    }
    
    /**
     * Retrieves the resource map for the given class. The application's default resource map is set as the parent map.
     * 
     * @param <T> The generic type representing the type of class for which the resource map is retrieved.
     * @param clazz for which the resource map is retrieved. Must not be <code>null</code>.
     * @return the resource map for the given class.
     * @throws IllegalArgumentException if <code>clazz</code> is <code>null</code>.
     */
    public <T> ResourceMap getResourceMap(Class<T> clazz) {
        
        return getResourceMap(clazz, clazz);
    }
    
    /**
     * * Retrieves the resource map for the given class hierarchy. The application default ResourceMap is set as parent to the ResourceMap
     * of the <code>endClass</code>.
     * 
     * @param <T> the base class of the class hierarchy.
     * @param startClass the class that the <code>ResourceMap</code> is retrieved for. Must not be <code>null</code>.
     * @param endClass the <code>ResourceMap</code> hierarchy is built up to this class. Must not be <code>null</code>.
     * @return the <code>ResourceMap</code> for the given class hierarchy from the <code>startClass</code> class up to the
     *         <code>endClass</code> class. The parent of the <code>endClass</code>'s resource map is set to the application default
     *         ResourceMap.
     * @throws IllegalArgumentException if <code>startClass</code> or <code>endClass</code> is <code>null</code>.
     */
    public <T> ResourceMap getResourceMap(Class<? extends T> startClass, Class<T> endClass) {
        ResourceMap resourceMapForClassTree = getResourceMap(startClass, endClass, getResourceMap());
        
        return resourceMapForClassTree;
    }
    
    /**
     * Collects the resource bundle names for the class hierarchy by calling getClassBundleNames(Class) for each class. The order of the
     * bundles goes from the start class's to the end class's bundles.
     * 
     * @param <T> The generic type representing the type of class for which the resource bundles are retrieved.
     * @param startClass first class in the hierarchy for that the bundle names are added to the list.
     * @param endClass last class in the hierarchy for that the bundle names are added to the list.
     * @return the list of bundle names for all the classes in the hierarchy.
     */
    public <T> List<String> getBundleNames(Class<? extends T> startClass, Class<T> endClass) {
        ArrayList<String> result = new ArrayList<String>();
        Class<?> clazz = startClass;
        while (clazz != endClass.getSuperclass()) {
            result.addAll(getClassBundleNames(clazz));
            clazz = clazz.getSuperclass();
        }
        return result;
    }
    
    /**
     * This implementation return a list containing single resource bundle name. The bundle has the same name as the class and is located in
     * the <code>"resources"</code> sub-package of the class's package.
     * <p>
     * For example, if the class is <code>com.ulc.java.MyClass</code> the bundle name is "<code>com.ulc.java.resources.MyClass</code>".
     * 
     * @param clazz that the resource bundles belong to.
     * @return the resource bundles for the given class.
     */
    public List<String> getClassBundleNames(Class<?> clazz) {
        String bundleName = clazz.getName();
        int insertionPoint = bundleName.lastIndexOf('.') + 1;
        bundleName = bundleName.substring(0, insertionPoint) + "resources." + bundleName.substring(insertionPoint);
        ArrayList<String> arrayList = new ArrayList<String>();
        arrayList.add(bundleName);
        return arrayList;
    }
    
    /**
     * Retrieves the resource map for the given class hierarchy. The given resource map is set as parent to the ResourceMap of the
     * <code>endClass</code>.
     * 
     * @param <T> the base class of the class hierarchy.
     * @param startClass the class that the <code>ResourceMap</code> is retrieved for. Must not be <code>null</code>.
     * @param endClass the <code>ResourceMap</code> hierarchy is built up to this class. Must not be <code>null</code>.
     * @param parent the ResourceMap that is set as parent of the <code>endClass</code>'s resource map. If <code>null</code> the
     *            returned ResourceMap hierarchy has no parent ResourceMap
     * @return the <code>ResourceMap</code> for the given class hierarchy from the <code>startClass</code> class up to the
     *         <code>endClass</code> class. The parent of the <code>endClass</code>'s resource map is set to the given ResourceMap.
     * @throws IllegalArgumentException if <code>startClass</code> or <code>endClass</code> is <code>null</code>.
     */
    public <T> ResourceMap getResourceMap(Class<? extends T> startClass, Class<T> endClass, ResourceMap parent) {
        if (startClass == null) {
            throw new IllegalArgumentException("startClass must not be null");
        }
        if (endClass == null) {
            throw new IllegalArgumentException("endClass must not be null");
        }
        List<String> bundleNames = getBundleNames(startClass, endClass);
        ResourceMap rMap = null;
        for (int i = bundleNames.size() - 1; i >= 0; i--) {
            rMap = new ResourceMap(parent, getLocaleProvider(), bundleNames.get(i));
            
            parent = rMap;
        }
        return rMap;
    }
    
    protected ILocaleProvider getLocaleProvider() {
        return ClientContextLocaleProvider.getInstance();
    }
    

}
