/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import com.ulcjava.base.application.ClientContext;
import com.ulcjava.base.application.IAction;
import com.ulcjava.base.application.ULCButton;
import com.ulcjava.base.application.ULCComponent;
import com.ulcjava.base.application.ULCFrame;
import com.ulcjava.base.application.ULCMenu;
import com.ulcjava.base.application.ULCMenuBar;
import com.ulcjava.base.application.ULCMenuItem;
import com.ulcjava.base.application.ULCRootPane;
import com.ulcjava.base.application.ULCSeparator;
import com.ulcjava.base.application.ULCToolBar;
import com.ulcjava.base.application.ULCWindow;
import com.ulcjava.base.application.event.IWindowListener;
import com.ulcjava.base.application.event.WindowEvent;
import com.ulcjava.environment.applet.application.ULCAppletPane;

import java.io.Serializable;

/**
 * Base class that defines a single main window with menuBar, tool bar, status-line and a content pane in the center. The view for the
 * application is determined by the environment that the client is running in. If it is running in an Applet an {@link AppletView} is used,
 * if it is running in an other environment a {@link FrameView} is used. This works from the same server for each client separately.
 * <p>
 * The applications's {@link ResourceMap} is used to inject the properties from the view's root pane down to the component tree.
 * <p>
 * The menu can be built using the action names form the application's {@link ApplicationActionMap}.
 * <p>
 * The most lightweight applications can just call show with the content pane in startup().
 */
public abstract class SingleFrameApplication extends Application {
    
    /**
     * Each String[] in <code>menus</code> defines a menu. The first entry is interpreted as key in the application's {@link ResourceMap}
     * that defines the text of the menu. The other entries are either an ActionName that is looked up in the application's
     * {@link ApplicationActionMap} or a String made out of dashes ( "-" or "------" ) that defines a {@link ULCSeparator}.
     */
    public class MenuBuilder implements Serializable {
        
        private final String[] fActions;
        private final String fTitle;
        
        /**
         * @param title the <code>title</code> is looked up in the applications {@code ResourceMap}, if the resource is found or else the
         *            <code>title</code> String itself is set as the Menus text, interpreting an '&' as indicator for the Menu's mnemonic
         *            character.
         * @param items each <code>item</code> is interpreted first as key of an action in the application's {@code ApplicationActionMap},
         *            if no action is found, the <code>item</code> is looked up in the application's {@code ResourceMap}, if the resource
         *            is found, else the <code>item</code> String itself is set as the MenuItems text, interpreting an '&' as indicator
         *            for the MenuItems mnemonic character. If the <code>item</code> consists only of '-' ( "-" or "------" ) a Separator
         *            is added to the menu. None of the <code>items</code> may be <code>null</code>.
         * @see MnemonicTextUtils
         */
        public MenuBuilder(String title, String... items) {
            if (title == null) {
                throw new IllegalArgumentException("title must not be null");
            }
            if (items == null) {
                throw new IllegalArgumentException("actions must not be null");
            }
            if (items.length == 0) {
                throw new IllegalArgumentException("actions must not be empty");
            }
            for (String action : items) {
                if (action == null) {
                    throw new IllegalArgumentException("action must not be null");
                }
            }
            fTitle = title;
            fActions = items;
        }
        
        /**
         * @return the built menu.
         */
        public ULCMenu getMenu() {
            String text = fromResourceOrKeep(fTitle);
            ULCMenu menu = new ULCMenu();
            MnemonicTextUtils.setMnemonicText(menu, fromResourceOrKeep(text));
            for (String item : fActions) {
                if (item.matches("-+")) {
                    menu.addSeparator();
                } else {
                    IAction action = getAction(item);
                    if (action == null) {
                        ULCMenuItem menuItem = new ULCMenuItem();
                        MnemonicTextUtils.setMnemonicText(menuItem, fromResourceOrKeep(item));
                        menu.add(menuItem);
                    } else {
                        menu.add(new ULCMenuItem(action));
                    }
                }
            }
            return menu;
        }
        
        private String fromResourceOrKeep(String key) {
            String text = getContext().getResourceMap().getString(key);
            if (text == null) {
                text = key;
            }
            return text;
        }
    }
    
    private View fMainView;
    private ToolBarFactory fToolBarFactory;
    
    
    @Override
    protected void startup() {
        setMenuBar(createStartupMenuBar());
        setToolBar(createStartupToolBar());
        setStatusBar(createStartupStatusBar());
        setMainContent(createStartupMainContent());
        
        show(getMainView());
    }
    
    
    protected ULCComponent createStartupMainContent() {
        return null;
    }
    
    protected ULCComponent createStartupStatusBar() {
        return null;
    }
    
    protected ULCComponent createStartupToolBar() {
        return null;
    }
    
    protected ULCMenuBar createStartupMenuBar() {
        return null;
    }
    
    public ULCComponent getMainContent() {
        return getMainView().getMainContent();
    }
    
    public ULCComponent getStatusBar() {
        return getMainView().getStatusBar();
    }
    
    public ULCMenuBar getMenuBar() {
        return getMainView().getMenuBar();
    }
    
    public ULCComponent getToolBar() {
        return getMainView().getToolBar();
    }
    
    public void setMainContent(ULCComponent component) {
        getMainView().setMainContent(component);
    }
    
    public void setStatusBar(ULCComponent statusBar) {
        getMainView().setStatusBar(statusBar);
    }
    
    public void setMenuBar(ULCMenuBar menuBar) {
        getMainView().setMenuBar(menuBar);
    }
    
    public void setToolBar(ULCComponent createStartupToolBar) {
        getMainView().setToolBar(createStartupToolBar);
    }
    
    /**
     * Sets the given component in the main view's main content area.
     * 
     * @param mainContent to set into the view's center area.
     * @deprecated use createStartupMainContent() instead
     */
    protected void show(ULCComponent mainContent) {
        if (mainContent == null) {
            throw new IllegalArgumentException("component must not be null");
        }
        View mainView = getMainView();
        mainView.setMainContent(mainContent);
        show(mainView);
    }
    
    /**
     * If the client runs in an applet the main view is a {@link AppletView}, otherwise a {@link FrameView}.
     * 
     * @return the applications main view.
     */
    public View getMainView() {
        if (fMainView == null) {
            fMainView = createMainView();
        }
        return fMainView;
    }
    
    /**
     * Creates the main view of the application. If the client runs in an applet the main view is a {@link AppletView}, otherwise a
     * {@link FrameView}.
     * 
     * @return the created {@link View}.
     */
    protected View createMainView() {
        if (ClientContext.getClientEnvironmentType() != ClientContext.APPLET) {
            FrameView frameView = createFrameView();
            initFrame(frameView.getFrame());
            return frameView;
        } else {
            AppletView appletView = createAppletView();
            initAppletView(appletView);
            return appletView;
        }
    }
    
    /**
     * @return the application's root pane. A ULCAppletPane if the client runs in an applet environment, an ULCFrame otherwise.
     */
    public ULCRootPane getRootPane() {
        return getMainView().getRootPane();
    }
    
    /**
     * @return the ULCFrame that serves as root pane of the application, <code>null</code> if the client runs in an applet environment.
     */
    public ULCFrame getFrame() {
        if (getRootPane() instanceof ULCFrame) {
            return (ULCFrame)getRootPane();
        }
        return null;
        
    }
    
    /**
     * @return the ULCAppletPane that serves as root pane of the application if the client runs in an applet environment, <code>null</code>
     *         otherwise
     */
    public ULCAppletPane getAppletPane() {
        if (getRootPane() instanceof ULCAppletPane) {
            return (ULCAppletPane)getRootPane();
        }
        return null;
        
    }
    
    
    /**
     * Factory method that creates a {@link AppletView} for clients that are running in an applet. Subclasses can overwrite this to create a
     * custom {@link AppletView} subclass.
     * 
     * @return the created {@link AppletView}.
     * @see #initAppletView(AppletView)
     */
    protected AppletView createAppletView() {
        return new AppletView(this);
    }
    
    /**
     * Factory method that creates a {@link FrameView} for clients that are not running in an applet. Subclasses can overwrite this to
     * create a custom {@link FrameView} subclass.
     * 
     * @return the created {@link FrameView}.
     * @see #initAppletView(AppletView)
     */
    protected FrameView createFrameView() {
        return new FrameView(this);
    }
    
    /**
     * Does nothing. Subclasses may overwrite this to initialize the applet view created as main view.
     * 
     * @param appletView that is going to be the main View.
     */
    protected void initAppletView(AppletView appletView) {
        initAppletPane(appletView.getAppletPane());
    }
    
    /**
     * Does nothing. Subclasses may overwrite this to initialize the applet pane created as main view.
     * 
     * @param appletPane that is going to be the main View root container.
     */
    protected void initAppletPane(ULCAppletPane appletPane) {
    }
    
    
    /**
     * Initializes the {@link ULCFrame} that will be used as the application's main frame. Sets the CloseOperation to
     * ULCWindow.DO_NOTHING_ON_CLOSE and installs a {@link IWindowListener} that calls exit(WindowEvent) on window closing. Subclasses may
     * overwrite this to initialize the frame created as main window.
     * 
     * @param frame that is going to be the main window.
     */
    protected void initFrame(ULCFrame frame) {
        frame.setDefaultCloseOperation(ULCWindow.DO_NOTHING_ON_CLOSE);
        frame.addWindowListener(new IWindowListener() {
            public void windowClosing(WindowEvent event) {
                exit(event);
            }
        });
    }
    
    /**
     * @return the {@link ToolBarFactory} that is used to create the tool bar and its buttons.
     */
    protected ToolBarFactory getToolBarFactory() {
        if (fToolBarFactory == null) {
            fToolBarFactory = createToolbarFactory();
        }
        
        return fToolBarFactory;
    }
    
    /**
     * Factory method that creates the {@link ToolBarFactory} for the application. Subclasses can overwrite this to return a custom
     * ToolbarFactory.
     * 
     * @return the created {@link ToolBarFactory}.
     * @deprecated use createToolBarFactory() instead
     */
    protected ToolBarFactory createToolbarFactory() {
        return createToolBarFactory();
    }

    /**
     * Factory method that creates the {@link ToolBarFactory} for the application. Subclasses can overwrite this to return a custom
     * tool bar factory.
     * 
     * @return the created {@link ToolBarFactory}.
     */
    public ToolBarFactory createToolBarFactory() {
        return new ToolBarFactory(getContext().getActionMap());
    }
    
    /**
     * Creates the application's tool bar out of strings that are used to lookup the toolbar's action in the Action Map.
     * 
     * @param actions String array that are creating the menus. Must not be <code>null</code> nor any element in it must be
     *            <code>null</code>
     * @deprecated override createStartupToolBar() instead
     */
    public void createAndAddToolBar(String... actions) {
        ULCToolBar toolBar = buildToolBar(actions);
        getMainView().setToolBar(toolBar);
    }
    
    /**
     * Creates a tool bar out of strings that are used to lookup the toolbar's action in the Action Map using the {@link ToolBarFactory}.
     * 
     * @param actions String array that are creating the menus. Must not be <code>null</code> nor any element in it must be
     *            <code>null</code>
     * @return the created tool bar
     * @deprecated use buildToolBar() instead
     */
    public ULCToolBar createToolbar(String... actions) {
        return buildToolBar(actions);
    }

    /**
     * Creates a tool bar out of strings that are used to lookup the toolbar's action in the Action Map using the {@link ToolBarFactory}.
     * 
     * @param actions String array that are creating the menus. Must not be <code>null</code> nor any element in it must be
     *            <code>null</code>
     * @return the created tool bar
     */
    public ULCToolBar buildToolBar(String... actions) {
        return getToolBarFactory().createToolBar(actions);
    }
    
    /**
     * Creates a ULCButton to be put into the tool bar. The button's name is set to "<i>actionName</i>.ToolBarButton".
     * 
     * @param actionName the name of the action that will be retrieved from the application's ActionMap and set on the button.
     * @return ULCButton configured with an action defined by actionName, to be put into the tool bar.
     * @deprecated use createToolBarButton() instead
     */
    public ULCButton createToolbarButton(String actionName) {
        return createToolBarButton(actionName);
    }

    /**
     * Creates a ULCButton to be put into the tool bar. The button's name is set to "<i>actionName</i>.ToolBarButton".
     * 
     * @param actionName the name of the action that will be retrieved from the application's ActionMap and set on the button.
     * @return ULCButton configured with an action defined by actionName, to be put into the tool bar.
     */
    public ULCButton createToolBarButton(String actionName) {
        return getToolBarFactory().createToolBarButton(actionName);
    }
    
    /**
     * Retrieves an action with the <code>actionName</code>. It is searched in the applications ActionMap.
     * 
     * @param actionName key of the action in the ActionMap.
     * @return the action if found, <code>null</code> otherwise.
     */
    protected IAction getAction(String actionName) {
        return getContext().getActionMap().get(actionName);
    }
    
    /**
     * Creates the applications menu bar out of string arrays. Each array defines a menu with the title set to the first item. The other
     * items are names defining actions that are used as menu entries or a String made of dashes to define a separator.
     * 
     * @see MenuBuilder
     * @param menus String array that are creating the menus. Must not be <code>null</code> nor any element in it must be
     *            <code>null</code>
     * @deprecated override createStartupMenuBar() instead
     */
    public void createMenuBar(String[]... menus) {
        if (menus == null) {
            throw new IllegalArgumentException("menues must not be null");
        }
        if (menus.length == 0) {
            throw new IllegalArgumentException("there must be at least one menu");
        }
        MenuBuilder[] builders = new MenuBuilder[menus.length];
        for (int i = 0; i < menus.length; i++) {
            String[] items = menus[i];
            if (items == null) {
                throw new IllegalArgumentException("a single menu must not be null");
            }
            if (items.length < 2) {
                throw new IllegalArgumentException("a single menu must have a at least two elements (title and one entry");
            }
            String[] entries = new String[items.length - 1];
            System.arraycopy(items, 1, entries, 0, entries.length);
            builders[i] = new MenuBuilder(items[0], entries);
        }
        createMenuBar(builders);
        
    }
    
    /**
     * Creates the application's menu bar with the menus that are built with the builders
     * 
     * @see MenuBuilder
     * @param menus MenuBuilders that are creating the menus. Must not be <code>null</code> nor any element in it must be
     *            <code>null</code>
     * @deprecated override createStartupMenuBar() instead
     */
    public void createMenuBar(MenuBuilder... menus) {
        if (menus == null) {
            throw new IllegalArgumentException("menues must not be null");
        }
        if (menus.length == 0) {
            throw new IllegalArgumentException("there must be at least one menu");
        }
        for (MenuBuilder menu : menus) {
            addMenu(menu);
        }
    }
    
    
    /**
     * Creates a menu bar out of string arrays. Each array defines a menu with the title set to the first item. The other items are names
     * defining actions that are used as menu entries or a String made of dashes to define a separator.
     * 
     * @see MenuBuilder
     * @param menus String array that are creating the menus. Must not be <code>null</code> nor any element in it must be
     *            <code>null</code>
     * @return the created menu bar
     */
    public ULCMenuBar buildMenuBar(String[]... menus) {
        if (menus == null) {
            throw new IllegalArgumentException("menues must not be null");
        }
        if (menus.length == 0) {
            throw new IllegalArgumentException("there must be at least one menu");
        }
        MenuBuilder[] builders = new MenuBuilder[menus.length];
        for (int i = 0; i < menus.length; i++) {
            String[] items = menus[i];
            if (items == null) {
                throw new IllegalArgumentException("a single menu must not be null");
            }
            if (items.length < 2) {
                throw new IllegalArgumentException("a single menu must have a at least two elements (title and one entry");
            }
            String[] entries = new String[items.length - 1];
            System.arraycopy(items, 1, entries, 0, entries.length);
            builders[i] = new MenuBuilder(items[0], entries);
        }
        return buildMenuBar(builders);
    }
    
    /**
     * Creates a menu bar with the menus that are built with the builders
     * 
     * @see MenuBuilder
     * @param menus MenuBuilders that are creating the menus. Must not be <code>null</code> nor any element in it must be
     *            <code>null</code>
     * @return the created menu bar
     */
    public ULCMenuBar buildMenuBar(MenuBuilder... menus) {
        ULCMenuBar menuBar = new ULCMenuBar();
        if (menus == null) {
            throw new IllegalArgumentException("menues must not be null");
        }
        if (menus.length == 0) {
            throw new IllegalArgumentException("there must be at least one menu");
        }
        for (MenuBuilder menuBuilder : menus) {
            menuBar.add(menuBuilder.getMenu());
        }
        return menuBar;
    }
    
    /**
     * Adds the menu built by the given {@link MenuBuilder} to the applications's menu bar.
     * 
     * @param menuBuilder
     * @deprecated use getMenuBar().add(menuBuilder.getMenu()) instead
     */
    public void addMenu(MenuBuilder menuBuilder) {
        if (menuBuilder == null) {
            throw new IllegalArgumentException("menues must not contain null");
        }
        ULCMenuBar menuBar = getMainView().getMenuBar();
        menuBar.add(menuBuilder.getMenu());
    }
    

}
