/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application;

import com.ulcjava.base.application.IAction;
import com.ulcjava.base.application.ULCButton;
import com.ulcjava.base.application.ULCToolBar;
import com.ulcjava.base.application.util.Insets;

import java.io.Serializable;

/**
 * Factory class that can create tool bars and the buttons for it.
 */
public class ToolBarFactory implements Serializable {


    private final ApplicationActionMap fActionMap;

    /**
     * Creates a ToolbarFactory with the given {@link ApplicationActionMap}.
     * 
     * @param actionMap used to create the actions set on the toolbar's buttons.
     */
    public ToolBarFactory(ApplicationActionMap actionMap) {
        fActionMap = actionMap;
    }

    /**
     * Creates a tool bar out of strings that are used to lookup the action in this ToolbarFactory's ActionMap.
     * 
     * @param actions names that are used to create the tool bar buttons. Must not be <code>null</code> nor any element
     *            in it must be <code>null</code>
     * @return the created tool bar
     */
    public ULCToolBar createToolBar(String... actions) {
        if (actions == null) {
            throw new IllegalArgumentException("actions must not be null");
        }
        if (actions.length == 0) {
            throw new IllegalArgumentException("there must be at least one actions");
        }
        ULCToolBar toolBar = createToolBar();
        for (String action : actions) {
            if (action == null) {
                throw new IllegalArgumentException("a single menu must not be null");
            }
            toolBar.add(createToolBarButton(action));
        }
        return toolBar;
    }

    /**
     * Creates a horizontal {@link ULCToolBar} and sets it to non floatable. Subclasses can overwrite this method to
     * create and configure a tool bar.
     * 
     * @return the created tool bar
     */
    protected ULCToolBar createToolBar() {
        ULCToolBar toolBar = new ULCToolBar(ULCToolBar.HORIZONTAL);
        toolBar.setFloatable(false);
        return toolBar;
    }

    /**
     * Creates a ULCButton to be put into the tool bar. The button's name is set to actionName<code>actionName</code>.ToolBarButton.
     * The button's verticalTextPosition is set to BOTTOM, the horizontalTextPosition to CENTER, focusable to
     * <code>false</code>, a margin of 15 is set on the left and the right and the border is not painted. Subclasses
     * can overwrite this method to create and configure the button differently.
     * 
     * @param actionName the name of the action that will be retrieved from the application's ActionMap.
     * @return ULCButton configured with an action defined by actionName, to be put into the tool bar.
     */
    protected ULCButton createToolBarButton(String actionName) {
        ULCButton button = new ULCButton();
        IAction action = fActionMap.get(actionName);
        button.setAction(action);
        button.setName(actionName + ".ToolBarButton");
        button.setVerticalTextPosition(ULCButton.BOTTOM);
        button.setHorizontalTextPosition(ULCButton.CENTER);
        button.setFocusable(false);
        button.setMargin(new Insets(0, 15, 0, 15));
        button.setBorderPainted(false);
        return button;
    }
}
