/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding;

import com.ulcjava.applicationframework.application.form.model.FormModel;
import com.ulcjava.applicationframework.application.form.model.PropertyAccessException;
import com.ulcjava.applicationframework.application.form.model.SetPropertyCommand;
import com.ulcjava.base.application.ULCCheckBox;
import com.ulcjava.base.application.event.IValueChangedListener;
import com.ulcjava.base.application.event.ValueChangedEvent;

/**
 * Binds a {@link ULCCheckBox} to a boolean {@link FormModel} property.
 */
public class CheckBoxBinder extends AbstractWidgetBinder<ULCCheckBox> {

    private final boolean fForTrue;

    /**
     * Creates a binding between the given {@link ULCCheckBox} and the given property.
     * 
     * @param model that holds the property. Must not be <code>null</code>.
     * @param propertyName name of the property to be bound. Must not be <code>null</code>.
     * @param checkBox to be bound. Must not be <code>null</code>.
     * @param forTrue if <code>true</code>, <code>true</code> is set on the property if the checkBox is checked, if
     *            <code>false</code>, <code>true</code> is set on the property if the checkBox is unchecked.
     */
    public CheckBoxBinder(FormModel<?> model, String propertyName, ULCCheckBox checkBox, boolean forTrue) {
        super(model, propertyName, checkBox);
        fForTrue = forTrue;
        checkBox.addValueChangedListener(new IValueChangedListener() {

            public void valueChanged(ValueChangedEvent event) {
                updateModel();
            }

        });

    }

    /**
     * @return <code>true</code> if <code>true</code> is set on the property if the checkBox is checked,
     *         <code>false</code> otherwise.
     */
    public boolean isForTrue() {
        return fForTrue;
    }

    /**
     * @return the {@link ULCCheckBox} that is bound.
     */
    protected ULCCheckBox getCheckBox() {
        return getWidget();
    }


    public void updateValueFromModel(FormModel<?> model) {
        try {
            Object value = model.getProperty(getPropertyName());
            if (value != null) {
                boolean booleanValue = ((Boolean)value).booleanValue();
                getCheckBox().setSelected(isForTrue() ? booleanValue : !booleanValue);
            }
        } catch (PropertyAccessException e) {
            throw new RuntimeException(e);
        }
    }

    private void updateModel() {
        try {
            boolean newValue = getCheckBox().isSelected();
            getModel().setProperty(
                    new SetPropertyCommand(getPropertyName(), new Boolean(isForTrue() ? newValue : !newValue)));
        } catch (PropertyAccessException e) {
            throw new RuntimeException(e);
        }
    }

}