/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.binding;

import com.ulcjava.applicationframework.application.form.model.FormModel;
import com.ulcjava.applicationframework.application.form.model.PropertyAccessException;
import com.ulcjava.applicationframework.application.form.model.SetPropertyCommand;
import com.ulcjava.base.application.ULCTextField;
import com.ulcjava.base.application.datatype.ULCAbstractErrorManager;
import com.ulcjava.base.application.event.IValueChangedListener;
import com.ulcjava.base.application.event.ValueChangedEvent;
import com.ulcjava.base.shared.ErrorObject;

import java.util.logging.ErrorManager;


/**
 * Binds a {@link ULCTextField} to a {@link FormModel} property.
 */
public class TextFieldBinder extends AbstractWidgetBinder<ULCTextField> {
    /**
     * Creates a binding between the given {@link ULCTextField} and the given property. If the {@link ErrorManager} is
     * not <code>null</code>, it is checked if it contains an error for the text field. If it does, the error is
     * propagated as illegal input to the model instead of the text field's value.
     * 
     * @param model that holds the property. Must not be <code>null</code>.
     * @param propertyName name of the property to be bound. Must not be <code>null</code>.
     * @param textField to be bound. Must not be <code>null</code>.
     * @param errorManager holds client side error state of the text field. May be <code>null</code>.
     * @see ULCAbstractErrorManager
     * @see FormModel#setIllegalInput(String, ErrorObject)
     */
    public TextFieldBinder(FormModel<?> model, String propertyName, ULCTextField textField, final ULCAbstractErrorManager errorManager) {
        super(model, propertyName, textField);
        getULCTextField().addValueChangedListener(new IValueChangedListener() {

            public void valueChanged(ValueChangedEvent event) {
                try {
                    Object value = getULCTextField().getValue();
                    if (value != null) {
                        SetPropertyCommand command = new SetPropertyCommand(getPropertyName(), value);
                        if (errorManager != null) {
                            ErrorObject error = errorManager.getError(getULCTextField());
                            if (error != null && value.equals(error.getIllegalValue())) {
                                command = new SetPropertyCommand(FormModel
                                        .getIllegalInputPropertyName(getPropertyName()), error);
                            }
                        }
                        getModel().setProperty(command);
                    }
                } catch (PropertyAccessException e) {
                    throw new RuntimeException(e);
                }
            }

        });
    }

    /**
     * @return the {@link ULCTextField} that is bound.
     */
    protected ULCTextField getULCTextField() {
        return getWidget();
    }

    /**
     * Propagates the value of the property to text field if the property does not have an illegalInput.
     * 
     * @param model the FormModel that wraps the bean containing the property bound to the text field
     * @see FormModel#setIllegalInput(String, ErrorObject)
     */
    public void updateValueFromModel(FormModel<?> model) {
        try {
            if (model.getIllegalInput(getPropertyName()) == null) {
                getULCTextField().setValue(model.getProperty(getPropertyName()));
            }
        } catch (PropertyAccessException e) {
            throw new RuntimeException(e);
        }
    }

}
