/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ULCLabel;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a {@link ULCLabel} that is added to the form.
 * It holds the parameters to configure the label, such as:
 * <dl>
 * <dt>text</dt>
 * <dd>Sets the text of the label.</dd>
 * <dt>horizontalAlignment</dt>
 * <dd>Sets the horizontal alignment on the label.</dd>
 * <dt>verticalAlignment</dt>
 * <dd>Sets the vertical alignment on the label.</dd>
 * <dt>horizontalTextPosition</dt>
 * <dd>Sets the horizontal text position of the label.</dd>
 * <dt>verticalTextPosition</dt>
 * <dd>Sets the vertical text position of the label.</dd>
 * </dl>
 */
public class LabelParameter extends LayoutParameter<LabelParameter, ULCLabel, WidgetFactory> {
    
    private boolean fHorizontalAlignmentIsSet;
    
    /**
     * Creates a {@code LabelParameter} object that uses the given {@code WidgetFactory} to create the {@link ULCLabel}.
     * 
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public LabelParameter(WidgetFactory widgetFactory) {
        super(widgetFactory);
    }
    
    /**
     * Creates a {@code LabelParameter} object that uses the given {@code WidgetFactory} to create the {@link ULCLabel}.
     * 
     * @param text of the label.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public LabelParameter(String text, WidgetFactory widgetFactory) {
        super(widgetFactory);
        getWidget().setText(text);
    }
    
    /**
     * Sets the horizontal alignment on the label.
     * 
     * @param alignment the alignment to be set. Should be one of LEFT, RIGHT, CENTER, LEADING or TRAILING defined in IDefaults
     * @return the {@code LabelParameter} object itself.
     * @see ULCLabel#setHorizontalAlignment(int)
     */
    public LabelParameter horizontalAlignment(int alignment) {
        fHorizontalAlignmentIsSet = true;
        getWidget().setHorizontalAlignment(alignment);
        return this;
    }
    
    /**
     * @return <code>true</code> if the horizontalAlignment is explicitly set by calling the horizontalAlignment method.
     */
    public boolean isHorizontalAlignmentSet() {
        return fHorizontalAlignmentIsSet;
    }
    
    /**
     * Sets the vertical alignment on the label.
     * 
     * @param alignment the alignment to be set. Should be one of TOP, BOTTOM, CENTER defined in IDefaults
     * @return the {@code LabelParameter} object itself.
     * @see ULCLabel#setVerticalAlignment(int)
     */
    public LabelParameter verticalAlignment(int alignment) {
        getWidget().setVerticalAlignment(alignment);
        return this;
    }
    
    /**
     * Sets the horizontal alignment on the label.
     * 
     * @param alignment the text postio to be set. Should be one of LEFT, RIGHT, CENTER, LEADING or TRAILING defined in IDefaults
     * @return the {@code LabelParameter} object itself.
     * @see ULCLabel#setHorizontalTextPosition(int)
     */
    public LabelParameter horizontalTextPosition(int textPosition) {
        getWidget().setHorizontalTextPosition(textPosition);
        return this;
    }
    
    /**
     * Sets the vertical alignment on the label.
     * 
     * @param textPosition the alignment to be set. Should be one of TOP, BOTTOM, CENTER defined in IDefaults
     * @return the {@code LabelParameter} object itself.
     * @see ULCLabel#setVerticalTextPosition(int)
     */
    public LabelParameter verticalTextPosition(int textPosition) {
        getWidget().setVerticalTextPosition(textPosition);
        return this;
    }
    
    /**
     * Sets the text on the label.
     * 
     * @param text the text to be set.
     * @return the {@code LabelParameter} object itself.
     * @see ULCLabel#setText(String)
     */
    public LabelParameter text(String text) {
        getWidget().setText(text);
        return this;
    }
    
    @Override
    protected ULCLabel createWidget() {
        return getWidgetFactory().createLabel();
    }
    
    @Override
    protected LabelParameter getThis() {
        return this;
    }
    
}
