/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ULCSpinner;
import com.ulcjava.base.application.ULCSpinnerNumberModel;

import java.text.DecimalFormat;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a {@link ULCSpinner} with
 * a number model that is added to the form.
 * 
 * @see ULCSpinner
 * @see ULCSpinnerNumberModel
 */
public class NumericSpinnerParameter extends LayoutParameter<NumericSpinnerParameter, ULCSpinner, WidgetFactory> {
    private String fDecimalFormatPattern;
    private ULCSpinnerNumberModel fSpinnerModel;

    /**
     * Creates a NumericSpinnerParameter object for the given property that uses the given {@code WidgetFactory} to
     * create the {@link ULCSpinner}.
     * 
     * @param propertyName name of the property the spinner is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public NumericSpinnerParameter(String propertyName, WidgetFactory widgetFactory) {
        super(propertyName, widgetFactory);
    }

    /**
     * @return <code>this</code>.
     */
    @Override
    protected NumericSpinnerParameter getThis() {
        return this;
    }

    /**
     * @return {@code Number} that is the spinner's value.
     */
    public Number getInitialValue() {
        return fSpinnerModel.getNumber();
    }

    /**
     * @return {@code Number} that is the spinner's lower bound.
     */
    public Number getMinimum() {
        return fSpinnerModel.getMinimum();
    }

    /**
     * @return {@code Number} that is the spinner's upper bound.
     */
    public Number getMaximum() {
        return fSpinnerModel.getMaximum();
    }

    /**
     * @return {@code Number} that is the step size of the spinner.
     */
    public Number getStepSize() {
        return fSpinnerModel.getStepSize();
    }

    /**
     * @return the format pattern that the spinner uses to display the numeric value.
     */
    public String getDecimalFormatPattern() {
        return fDecimalFormatPattern;
    }

    /**
     * Set the value on the spinner.
     * 
     * @param initialValue the value to set. If the given value is <code>null</code> then the minimum is set if the
     *            minimum is not <code>null</code>. If the minimum is <code>null</code> too, 0 is set.
     * @return the {@code NumericSpinnerParameter} object itself.
     */
    public NumericSpinnerParameter initialValue(Number initialValue) {
        Number initial = initialValue == null ? (getMinimum() == null ? 0 : getMinimum()) : initialValue;
        fSpinnerModel.setValue(initial);
        return this;
    }

    /**
     * Sets the spinner's lower bound.
     * 
     * @param minimum the lower bound to set.
     * @return the {@code NumericSpinnerParameter} object itself.
     */
    public NumericSpinnerParameter minimum(Number minimum) {
        fSpinnerModel.setMinimum(minimum);
        return this;
    }

    /**
     * Sets the spinner's upper bound.
     * 
     * @param maximum the upper bound to set.
     * @return the {@code NumericSpinnerParameter} object itself.
     */
    public NumericSpinnerParameter maximum(Number maximum) {
        fSpinnerModel.setMaximum(maximum);
        return this;
    }

    /**
     * Sets the spinner's step size.
     * 
     * @param stepSize the step size to set.
     * @return the {@code NumericSpinnerParameter} object itself.
     */
    public NumericSpinnerParameter stepSize(Number stepSize) {
        fSpinnerModel.setStepSize(stepSize);
        return this;
    }

    /**
     * Sets the format pattern that the spinner uses to display the numeric value.
     * 
     * @param decimalFormatPattern that defines how to display the value.
     * @return the {@code NumericSpinnerParameter} object itself
     * @see DecimalFormat
     */
    public NumericSpinnerParameter formatPattern(String decimalFormatPattern) {
        getWidget().setEditor(new ULCSpinner.ULCNumberEditor(getWidget(), decimalFormatPattern));
        fDecimalFormatPattern = decimalFormatPattern;
        return this;
    }

    /**
     * Creates a {@link ULCSpinner} with the {@link WidgetFactory} and sets a {@link ULCSpinnerNumberModel} on it.
     * 
     * @return the created {@code ULCSpinner}.
     */
    @Override
    protected ULCSpinner createWidget() {
        fSpinnerModel = new ULCSpinnerNumberModel();
        ULCSpinner spinner = getWidgetFactory().createSpinner();
        spinner.setModel(fSpinnerModel);
        return spinner;
    }
}