/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ULCPasswordField;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a {@link ULCPasswordField}
 * that is added to the form. It holds the parameters to configure the password field such as:
 * <dl>
 * <dt>columns</dt>
 * <dd>The number of columns on the {@link ULCPasswordField}.</dd>
 * <dt>echoCharacter</dt>
 * <dd>The character to used for echoing. The default is '*'.</dd>
 * </dl>
 */
public class PasswordFieldParameter extends LayoutParameter<PasswordFieldParameter, ULCPasswordField, WidgetFactory> {
    /**
     * Creates a {@code PasswordFieldParameter} object for the given property that uses the given {@code WidgetFactory}
     * to create the {@link ULCPasswordField}.
     * 
     * @param propertyName name of the property the password field is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public PasswordFieldParameter(String propertyName, WidgetFactory widgetFactory) {
        super(propertyName, widgetFactory);
    }

    /**
     * @return <code>this</code>.
     */
    @Override
    protected PasswordFieldParameter getThis() {
        return this;
    }

    /**
     * @return the number of columns on the {@link ULCPasswordField}.
     */
    public int getColumns() {
        return getWidget().getColumns();
    }

    /**
     * Sets the number of columns on the {@link ULCPasswordField}.
     * 
     * @param columns the number of columns to set.
     * @return the {@code PasswordFieldParameter} object itself.
     * @throws IllegalArgumentException if <code>columns</code> is less than 0.
     */
    public PasswordFieldParameter columns(int columns) {
        getWidget().setColumns(columns);
        return this;
    }

    /**
     * @return the character to used for echoing.
     */
    public Character getEchoCharacter() {
        return getWidget().getEchoChar();
    }

    /**
     * Sets the character to used for echoing on the {@link ULCPasswordField}.
     * 
     * @param echoCharacter the character to display.
     * @return the {@code PasswordFieldParameter} object itself.
     */
    public PasswordFieldParameter echoCharacter(char echoCharacter) {
        getWidget().setEchoChar(echoCharacter);
        return this;
    }

    /**
     * Creates a {@link ULCPasswordField} with the {@link WidgetFactory}.
     * 
     * @return the created {@code ULCPasswordField}.
     */
    @Override
    protected ULCPasswordField createWidget() {
        return getWidgetFactory().createPasswordField();
    }
}