/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form;

import com.ulcjava.base.application.ClientContext;
import com.ulcjava.base.application.ULCTextField;
import com.ulcjava.base.application.datatype.IDataType;
import com.ulcjava.base.application.datatype.ULCDateDataType;
import com.ulcjava.base.application.datatype.ULCNumberDataType;

import java.text.DecimalFormatSymbols;

/**
 * Helper class that is used by the {@link AbstractFormBuilder} to collect the parameters for a {@link ULCTextField} that is added to the
 * form. It holds the parameters to configure the text field, such as:
 * <dl>
 * <dt>columns</dt>
 * <dd>Sets the number of columns on the text field. The default is <code>0</code>.</dd>
 * <dt>horizontalAlignment</dt>
 * <dd>Sets the horizontal alignment on the text field.</dd>
 * <dt>integerFormatPattern</dt>
 * <dd>Sets the a {@link ULCNumberDataType} with the given pattern on the text field. The data type's integer property is set to
 * <code>true</code>.</dd>
 * <dt>decimalFormatPattern</dt>
 * <dd>Sets the a {@link ULCNumberDataType} with the given pattern on the text field.</dd>
 * <dt>dateFormatPattern</dt>
 * <dd>Sets the a {@link ULCDateDataType} with the given pattern on the text field.</dd>
 * <dt>dataType</dt>
 * <dd>Sets a data type on the text field.</dd>
 * </dl>
 */
public class TextFieldParameter extends LayoutParameter<TextFieldParameter, ULCTextField, WidgetFactory> {
    
    private String fIntegerFormatPattern;
    private String fDecimalFormatPattern;
    private String fDateFormatPattern;
    private boolean fHorizontalAlignmentIsSet = false;
    
    /**
     * Creates a {@code TextFieldParameter} object for the given property that uses the given {@code WidgetFactory} to create the
     * {@link ULCTextField}.
     * 
     * @param propertyName name of the property the text field is bound to.
     * @param widgetFactory factory that is used to create widgets. Must not be <code>null</code>.
     */
    public TextFieldParameter(String propertyName, WidgetFactory widgetFactory) {
        super(propertyName, widgetFactory);
    }
    
    /**
     * @return {@code this}.
     */
    @Override
    protected TextFieldParameter getThis() {
        return this;
    }
    
    
    /**
     * @return the integer format pattern if set, <code>null</code> otherwise.
     * @see #integerFormatPattern(String)
     */
    public String getIntegerFormatPattern() {
        return fIntegerFormatPattern;
    }
    
    /**
     * @return the decimal format pattern if set, <code>null</code> otherwise.
     * @see #decimalFormatPattern(String)
     */
    public String getDecimalFormatPattern() {
        return fDecimalFormatPattern;
    }
    
    /**
     * @return the date format pattern if set, <code>null</code> otherwise.
     * @see #dateFormatPattern(String)
     */
    public String getDateFormatPattern() {
        return fDateFormatPattern;
    }
    
    /**
     * Sets the number of columns on the text field.
     * 
     * @param columns the number of columns. Must be greater or equal to 0.
     * @return the {@code TextFieldParameter} object itself.
     * @throws IllegalArgumentException if <code>columns</code> is less than 0.
     * @see ULCTextField#setColumns(int)
     */
    public TextFieldParameter columns(int columns) {
        getWidget().setColumns(columns);
        return this;
    }
    
    
    /**
     * Sets the horizontal alignment on the text field.
     * 
     * @param alignment the alignment to be set. Should be one of LEFT, RIGHT, CENTER, LEADING or TRAILING defined in IDefaults
     * @return the {@code TextFieldParameter} object itself.
     * @see ULCTextField#setHorizontalAlignment(int)
     */
    public TextFieldParameter horizontalAlignment(int alignment) {
        fHorizontalAlignmentIsSet = true;
        getWidget().setHorizontalAlignment(alignment);
        return this;
    }
    
    /**
     * @return <code>true</code> if the horizontalAlignment is explicitly set by calling the horizontalAlignment method.
     */
    public boolean isHorizontalAlignmentSet() {
        return fHorizontalAlignmentIsSet;
    }
    
    /**
     * Sets the a {@link ULCNumberDataType} with the given pattern on the text field. The data type's integer property is set to
     * <code>true</code>.
     * 
     * @param formatPattern a valid format for the {@link ULCNumberDataType} class.
     * @return the {@code TextFieldParameter} object itself.
     * @see ULCNumberDataType
     */
    public TextFieldParameter integerFormatPattern(String formatPattern) {
        ULCNumberDataType numberDataType = new ULCNumberDataType(formatPattern, getDecimalFormatSymbols());
        numberDataType.setInteger(true);
        setDataType(numberDataType);
        fIntegerFormatPattern = formatPattern;
        return this;
    }
    
    /**
     * Sets the a {@link ULCNumberDataType} with the given pattern on the text field.
     * 
     * @param formatPattern a valid format for the {@link ULCNumberDataType} class.
     * @return {@code TextFieldParameter} object itself.
     * @see ULCNumberDataType
     */
    public TextFieldParameter decimalFormatPattern(String formatPattern) {
        setDataType(new ULCNumberDataType(formatPattern, getDecimalFormatSymbols()));
        fDecimalFormatPattern = formatPattern;
        return this;
    }
    
    /**
     * Sets the a {@link ULCDateDataType} with the given pattern on the text field.
     * 
     * @param formatPattern a valid format for the Java {@link java.text.SimpleDateFormat} class.
     * @return the {@code TextFieldParameter} object itself.
     * @see ULCDateDataType
     */
    public TextFieldParameter dateFormatPattern(String formatPattern) {
        setDataType(new ULCDateDataType(formatPattern));
        fDateFormatPattern = formatPattern;
        return this;
    }
    
    /**
     * Returns the decimal format symbols to be used to instantiate {@link ULCNumberDataType} objects.
     * 
     * @return {@link DecimalFormatSymbols} created with the client's locale.
     */
    protected DecimalFormatSymbols getDecimalFormatSymbols() {
        return new DecimalFormatSymbols(ClientContext.getLocale());
    }
    
    /**
     * @return the {@code IDataType} that is set on the text field.
     */
    public IDataType getDataType() {
        return getWidget().getDataType();
    }
    
    /**
     * Sets the given data type on the text field.
     * 
     * @param dataType to set on the text field.
     * @return the {@code TextFieldParameter} object itself.
     * @see ULCTextField#setDataType(IDataType)
     */
    public TextFieldParameter dataType(IDataType dataType) {
        setDataType(dataType);
        return this;
    }
    
    private void setDataType(IDataType dataType) {
        getWidget().setDataType(dataType);
    }
    
    /**
     * Creates a {@link ULCTextField} with the {@link WidgetFactory}.
     * 
     * @return the created {@code ULCTextField}.
     */
    @Override
    protected ULCTextField createWidget() {
        return getWidgetFactory().createTextField();
    }
    
}