/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form.model;

import com.ulcjava.base.application.ULCTextField;
import com.ulcjava.base.application.datatype.ULCNumberDataType;

import java.io.Serializable;
import java.math.BigDecimal;

/**
 * IMapper that converts model BigDecimal property to a Double to be used by a form's {@link ULCTextField} with a
 * {@link ULCNumberDataType}. Most conveniently used by adding it in a {@link FormModel} subclass's addMappers method:
 * 
 * <pre><code>
 * public class MyFromModel extends FormModel&lt;MyJavaBean&gt; {
 * ...
 *     &#064;Override
 *  protected void addMappers() {
 *      addMapper(BigDecimal.class, new BigDecimalToDoubleMapper());
 *  }
 * ...
 * }
 * </code></pre>
 */
public final class BigDecimalToDoubleMapper implements IMapper<BigDecimal, Double>, Serializable {

    /**
     * Maps Double to BigDecimal
     * 
     * @param fromForm Double value
     * @return BigDecimal value
     */
    public BigDecimal mapToModel(Double fromForm) {
        return fromForm == null ? null : new BigDecimal(fromForm.toString());
    }

    /**
     * Maps BigDecimal to Double
     * 
     * @param fromModel BigDecimal value
     * @return Double value
     */
    public Double mapToForm(BigDecimal fromModel) {
        return fromModel == null ? null : fromModel.doubleValue();
    }
}