/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form.model;

import com.ulcjava.base.shared.ErrorCodes;
import com.ulcjava.base.shared.ErrorObject;

/**
 * Performs a greater or equal check on one or more properties. If the check fails for a property, an
 * {@link ErrorObject} is created and set on the {@link FormModel} for the failed property. The error object has the
 * error code {@code ErrorCodes.ERROR_CODE_VALUE_TOO_SMALL}; the minimal value is set as additional parameter (index 1)
 * for the place holder in the error code. The first parameter at place holder index 0 on the error code is as always
 * the value that failed the check.
 * 
 * @param <T> Generic type for the type of property value on which the check is performed
 */
public class GreaterOrEqualValidator<T extends Comparable<? super T>> extends RangeValidator<T> {

    /**
     * Creates a validator that checks the values of the given properties against the given minimal value.
     * 
     * @param min minimal value that the property values are compared against.
     * @param propertyName names of all properties that are checked against the minimal value.
     */
    public GreaterOrEqualValidator(T min, String... propertyName) {
        super(min, null, propertyName);

    }


    @Override
    /**
     * @returns ErrorCodes.ERROR_CODE_VALUE_TOO_SMALL.
     */
    protected String createErrorCode(Comparable<? super T> value) {
        return ErrorCodes.ERROR_CODE_VALUE_TOO_SMALL;
    }

    @Override
    /**
     * Creates an {@link ErrorObject} with the given error code, the illegalValue, and the minimal value as additional
     * parameter (index 1).
     * 
     * @param propertyName property that is in error.
     * @param illegalValue the value that fails validation.
     * @param errorCode identifies the error type.
     * @return {@link ErrorObject} to be used for error feedback.
     */
    protected ErrorObject createError(String propertyName, java.lang.Comparable<? super T> illegalValue,
            String errorCode) {
        return new ErrorObject(errorCode, illegalValue, new Object[] { getMin() });
    }
}
