/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form.model;

import com.ulcjava.base.shared.ErrorObject;

/**
 * Base class for writing validators for properties of a given type in a form model.
 * <p>
 * </p>
 * <b>Example</b>
 * 
 * <pre><code>
 * formModel.addValidator(new PropertyValidator&lt;Integer&gt;(propertyNames) {
 *     &#064;Override
 *     public String validateValue(Integer value) {
 *         return value &lt; 0 ? VALUE_MUST_BE_GREATER_OR_EQUAL_TO_0 : null;
 *     }
 * });
 * </code></pre>
 * 
 * @param <T> The generic type representing the type of the properties to be validated
 */
public abstract class PropertyValidator<T> implements IValidator {

    private final String[] fProperties;


    /**
     * Creates a an instance of {@code PropertyValidator}
     * 
     * @param propertyNames The names of the properties that need to be validated
     */
    public PropertyValidator(String... propertyNames) {
        if (propertyNames == null) {
            throw new IllegalArgumentException("propertyNames must not be null");
        }
        for (String propertyName : propertyNames) {
            if (propertyName == null) {
                throw new IllegalArgumentException("propertyName must not be null");
            }
        }
        fProperties = propertyNames;
    }

    /**
     * @return a {@code String} array with names of the properties that need to be validated
     */
    public String[] getProperties() {
        return fProperties;
    }

    /**
     * An implementation of {@link IValidator}.
     * 
     * @param formModel The {@code FormModel} of a given bean type to be validated
     * @see IValidator
     */
    public void validate(FormModel<?> formModel) {
        String[] properties = getProperties();
        for (String propertyName : properties) {
            validate(propertyName, formModel);
        }
    }

    /**
     * Validate the property with the given name in the {@code FormModel} and add the error code if any to the
     * {@link ErrorObject} that will be attached to the {@code FormModel}.
     * 
     * @param propertyName The name of the property to be validated.
     * @param formModel The form model containing the bean with the given property name.
     */
    public void validate(String propertyName, FormModel<?> formModel) {
        T value = (T)formModel.getProperty(propertyName);
        String errorCode = validateValue(value);
        if (errorCode != null) {
            formModel.setErrorForProperty(propertyName, createError(propertyName, value, errorCode));
        }
    }

    /**
     * Creates an {@link ErrorObject} with the given code and the illegal value. Subclasses should overwrite this to
     * create an ErrorObject with additional parameters.
     * 
     * @param propertyName property that is in error.
     * @param illegalValue the value that fails validation.
     * @param errorCode identifies the error type.
     * @return {@link ErrorObject} to be used for error feedback.
     */
    protected ErrorObject createError(String propertyName, T illegalValue, String errorCode) {
        return new ErrorObject(errorCode, illegalValue);
    }

    /**
     * Validates the value. If the value is not valid an appropriate error message is returned, If the value is valid
     * <code>null</code> is returned.
     * 
     * @param value the value to be validated
     * @return if the value is valid <code>null</code> otherwise a string describing the error
     */
    public abstract String validateValue(T value);

}
