/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form.model;

import com.ulcjava.base.shared.ErrorCodes;
import com.ulcjava.base.shared.ErrorObject;

/**
 * Performs a range check on one or more properties. If the check fails for a property, an {@link ErrorObject} is
 * created and set on the {@link FormModel} for the failed property. The error object has the error code
 * {@code ErrorCodes.ERROR_CODE_VALUE_NOT_IN_RANGE}; the minimal (index 1) and the maximal value (index 2) are set as
 * additional parameters for the place holders in the error code. The first parameter at place holder index 0 on the
 * error code is as always the value that failed the check.
 * 
 * @param <T> Generic type for the type of property value on which the range check is performed
 */
public class RangeValidator<T extends Comparable<? super T>> extends PropertyValidator<Comparable<? super T>> {
    private T fMax;
    private T fMin;

    /**
     * Creates a validator that checks if the values of the given properties are between the given minimal and maximal
     * value (inclusive). Either one of the given bounds may be <code>null</code>, but not both. If one value is
     * <code>null</code> then this bound is not checked.
     * 
     * @param min lower bound that the property values are compared against.
     * @param max higher bound that the property values are compared against.
     * @param propertyName names of all properties that are checked.
     */
    public RangeValidator(T min, T max, String... propertyName) {
        super(propertyName);
        if (min == null & max == null) {
            throw new IllegalArgumentException("min and max cannot both be null");
        }
        if (min == null || max == null || min.compareTo(max) <= 0) {
            fMin = min;
            fMax = max;
        } else {
            fMin = max;
            fMax = min;
        }
    }

    /**
     * @see PropertyValidator#validateValue(java.lang.Object)
     */
    @Override
    public String validateValue(Comparable<? super T> value) {
        if (value != null && (isOutsideLowerBound(value) || isOutsideUpperBound(value))) {
            return createErrorCode(value);
        }
        return null;
    }

    /**
     * Creates error code for the value that was out of range.
     * 
     * @param value the value that failed the check
     * @return ErrorCodes.ERROR_CODE_VALUE_NOT_IN_RANGE
     */
    protected String createErrorCode(Comparable<? super T> value) {
        return ErrorCodes.ERROR_CODE_VALUE_NOT_IN_RANGE;
    }

    /**
     * Creates an {@link ErrorObject} with the given error code, the illegalValue and an array containing the minimum
     * (index 1) and the maximum values (index 2).
     * 
     * @param propertyName property that is in error.
     * @param illegalValue the value that fails validation.
     * @param errorCode identifies the error type.
     * @return {@link ErrorObject} to be used for error feedback.
     */
    @Override
    protected ErrorObject createError(String propertyName, Comparable<? super T> illegalValue, String errorCode) {
        return new ErrorObject(errorCode, illegalValue, new Object[] { getMin(), getMax() });
    }

    /**
     * Checks if the given value is less than the lower bound.
     * 
     * @param value the value to check.
     * @return true if there is a lower bound and the value is less than it, otherwise false.
     */
    protected boolean isOutsideLowerBound(Comparable<? super T> value) {
        return getMin() != null && value.compareTo(getMin()) < 0;
    }

    /**
     * Checks if the given value is bigger than the upper bound.
     * 
     * @param value the value to check.
     * @return true if there is a higher bound and the value is greater than it, otherwise false.
     */
    protected boolean isOutsideUpperBound(Comparable<? super T> value) {
        return getMax() != null && value.compareTo(getMax()) > 0;
    }

    /**
     * @return the value of the upper bound. It may be <code>null</code> signifying that there is no upper bound.
     */
    public T getMax() {
        return fMax;
    }

    /**
     * @return the value of the lower bound. It may be <code>null</code> signifying that there is no lower bound.
     */
    public T getMin() {
        return fMin;
    }

}
