/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.applicationframework.application.form.model;

import java.io.Serializable;

/**
 * Command to set a Property on a form model. It keeps the old and the new values and supports undo operation.
 */
public class SetPropertyCommand implements Serializable {
    private final String fPropertyName;
    private final Object fNewValue;
    private Object fOldValue;

    /**
     * Sets value on a property.
     * 
     * @param propertyName the property name to be set. Must not be <code>null</code>
     * @param newValue the new value to be set.
     */
    public SetPropertyCommand(String propertyName, Object newValue) {
        if (propertyName == null) {
            throw new IllegalArgumentException("propertyName must not be null");
        }
        fPropertyName = propertyName;
        fNewValue = newValue;
        fOldValue = null;
    }

    /**
     * @return the new value that the property should have after the command is executed.
     */
    public Object getNewValue() {
        return fNewValue;
    }

    /**
     * @return the property name on which the value is to be set.
     */
    public String getPropertyName() {
        return fPropertyName;
    }

    /**
     * @return the value of the property before the command is executed.
     */
    public Object getOldValue() {
        return fOldValue;
    }

    /**
     * Sets the value of the property before processing the command.
     * 
     * @param oldValue the value of the property before processing the command.
     */
    protected void setOldValue(Object oldValue) {
        fOldValue = oldValue;
    }

    /**
     * Undoes the command by setting the old value on the property again.
     * 
     * @param model the FormModel on which the property is set.
     */
    void undo(FormModel<?> model) {
        model.setProperty(fPropertyName, fOldValue);
        model.update(true);
    }
}