/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.easydeployment.server;

import sun.security.tools.JarSigner;

import com.ulcjava.base.server.ApplicationConfiguration;
import com.ulcjava.base.server.SessionLogConfigProvider;
import com.ulcjava.base.shared.logging.Level;
import com.ulcjava.base.shared.logging.Logger;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.jar.Pack200.Unpacker;
import java.util.zip.GZIPOutputStream;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.ServletException;

/**
 * Subclass of JnlpDownloadServlet that signs the client jars during initialization and packs them with pack200.
 */
public class ClientJarPreparationListener implements ServletContextListener {
    private static final Logger LOG = Logger.getLogger(ClientJarPreparationListener.class.getName());
    
    static {
        SessionLogConfigProvider.initializeLogManager();
    }
    
    public void contextDestroyed(ServletContextEvent event) {
    }
    
    public void contextInitialized(ServletContextEvent event) {
        ServletContext servletContext = event.getServletContext();
        try {
            new ClientJarPreparation(servletContext).prepareClientJars();
        } catch (Exception e) {
            LOG.log(Level.SEVERE, "Could not prepare client jar files", e);
        }
    }
    
    private static class ClientJarPreparation {
        private static final String SIGNJAR_PROPERTIES = "/WEB-INF/signjar.properties";
        private Properties fJarSigningProperties;
        private JarSigner fJarSigner;
        private final ServletContext fServletContext;
        
        public ClientJarPreparation(ServletContext servletContext) {
            fServletContext = servletContext;
        }
        
        public ServletContext getServletContext() {
            return fServletContext;
        }
        
        public void prepareClientJars() throws ServletException, IOException {
            String jnlpLauncherClassName = ApplicationConfiguration.getInstance().getJnlpLauncherClassName();
            String jnlpLauncherClassFileName = "com/ulcjava/environment/jnlp/client/DefaultJnlpLauncher.class";
            if (jnlpLauncherClassName != null) {
                jnlpLauncherClassFileName = jnlpLauncherClassName.replace('.', '/') + ".class";
            }
            boolean launcherLibFound = false;
            fJarSigningProperties = getJarSigningProperties();
            fJarSigner = new JarSigner();
            ServletContext servletContext = getServletContext();
            String jarList = EasyDeploymentUtilities.toAppletArchiveAttribute(servletContext);
            LOG.log(Level.INFO, "client libs -> " + jarList);
            if (getKeystoreName() == null) {
                LOG.log(Level.INFO, "No keystore defined - the jar files wont be signed");
            }
            String[] jars = jarList.split(",");
            for (int i = 0; i < jars.length; i++) {
                String jar = jars[i];
                String sourceJarFileName = getServletContext().getRealPath("/WEB-INF/lib/" + jar);
                String destJarFileName = getServletContext().getRealPath("/" + jar);
                copyFile(sourceJarFileName, destJarFileName);
                if (!launcherLibFound && jarContainsJnlpLauncher(sourceJarFileName, jnlpLauncherClassFileName)) {
                    launcherLibFound = true;
                    EasyDeploymentUtilities.setJNLPLauncherLibName(jar);
                }
            }
        }
        
        private boolean jarContainsJnlpLauncher(String sourceJarFileName, String jnlpLauncherClassFileName) throws IOException {
            JarFile jar = new JarFile(sourceJarFileName);
            return jar.getEntry(jnlpLauncherClassFileName) != null;
        }
        
        private Properties getJarSigningProperties() {
            Properties properties = new Properties();
            try {
                String propFilePath = getServletContext().getRealPath(SIGNJAR_PROPERTIES);
                properties.load(new FileInputStream(propFilePath));
            } catch (IOException e) {
                LOG.log(Level.WARNING, "Could not read jarsigning.properties", e);
            }
            return properties;
        }
        
        private void copyFile(String sourceJarFileName, String destJarFileName) {
            
            File source = new File(sourceJarFileName);
            File destination = new File(destJarFileName);
            if (!destination.exists() || source.lastModified() > destination.lastModified()) {
                doPack(sourceJarFileName, destJarFileName);
            }
            
        }
        
        private void doPack(String sourceJarFileName, String destJarFileName) {
            try {
                String packedJarFileName = destJarFileName + ".pack.gz";
                LOG.log(Level.INFO, "packing file " + sourceJarFileName + " to " + destJarFileName);
                JarFile jar = new JarFile(sourceJarFileName);
                if (containsNoFiles(jar)) {
                    return;
                }
                Pack200.Packer packer = Pack200.newPacker();
                FileOutputStream fileOutputStream = new FileOutputStream(packedJarFileName);
                packer.pack(jar, fileOutputStream);
                fileOutputStream.close();
                

                Unpacker unpacker = Pack200.newUnpacker();
                File repackedJar = new File(destJarFileName);
                FileOutputStream out = new FileOutputStream(repackedJar);
                File firstPackedFile = new File(packedJarFileName);
                JarOutputStream jarOut = new JarOutputStream(out);
                unpacker.unpack(firstPackedFile, jarOut);
                jarOut.close();
                sign(destJarFileName);
                
                GZIPOutputStream gzout = new GZIPOutputStream(new FileOutputStream(packedJarFileName));
                packer.pack(new JarFile(destJarFileName), gzout);
                gzout.close();
                

            } catch (Exception e) {
                LOG.log(Level.WARNING, "Packing of " + sourceJarFileName + " failed", e);
            }
        }
        
        private boolean containsNoFiles(final JarFile jarFile) {
            for (Enumeration<JarEntry> entries = jarFile.entries(); entries
                    .hasMoreElements();) {
                if (!entries.nextElement().isDirectory()) {
                    return false;
                }
            }
            return true;
        }

        
        private void sign(String unpackedJarFileName) {
            
            String keystoreName = getKeystoreName();
            if (keystoreName != null) {
                try {
                    LOG.log(Level.INFO, "signing " + unpackedJarFileName);
                    fJarSigner.run(new String[] {"-keystore", keystoreName, "-storepass", getStorePass(), "-keypass", getKeyPass(),
                            unpackedJarFileName, fJarSigningProperties.getProperty("signjar.alias")});
                } catch (Exception e) {
                    LOG.log(Level.WARNING, "Signing of " + unpackedJarFileName + " failed", e);
                }
            } else {
                LOG.log(Level.WARNING, "signjar.keystore not defined - jar files wont be signed");
            }
            
        }
        
        private String getStorePass() {
            return fJarSigningProperties.getProperty("signjar.storepass");
        }
        
        private String getKeyPass() {
            String keypass = fJarSigningProperties.getProperty("signjar.keypass");
            return keypass == null ? getStorePass() : keypass;
        }
        
        private String getKeystoreName() {
            String keystorename = fJarSigningProperties.getProperty("signjar.keystore");
            return keystorename != null ? getServletContext().getRealPath("/WEB-INF/" + keystorename) : null;
        }
    }
    
}
