/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.easydeployment.server;

import com.ulcjava.base.server.ApplicationConfiguration;

import java.io.IOException;
import java.io.PrintWriter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


public class JNLPDownloader extends javax.servlet.http.HttpServlet implements javax.servlet.Servlet {
    
    private static class Attribute {
        private final String fValue;
        private final String fKey;
        
        public Attribute(String key, String value) {
            fKey = key;
            fValue = value;
        }
        
        public String getKey() {
            return fKey;
        }
        
        public String getValue() {
            return fValue;
        }
        
        public boolean hasValue() {
            return getValue() != null;
        }
        
        @Override
        public String toString() {
            return getKey() + "=\"" + getValue() + "\"";
        }
    }
    
    
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        DeploymentParameter parameter = new DeploymentParameter(request.getLocale());
        
        response.setContentType("application/x-java-jnlp-file");
        
        ApplicationConfiguration appConfig = ApplicationConfiguration.getInstance();
        PrintWriter writer = response.getWriter();
        writer.println("<jnlp spec=\"1.0+\"");
        writer.println("      codebase=\"" + EasyDeploymentUtilities.toBaseURL(request) + "\"");
        String jnlpFileName = request.getServletPath();
        jnlpFileName = jnlpFileName.substring(jnlpFileName.lastIndexOf('/') + 1);
        writer.println("      href=\"" + jnlpFileName + "\">");
        writer.println("    <information>");
        writer.println("        <title>" + parameter.getTitle() + "</title>");
        String vendor = parameter.getVendor();
        writeTagIfNotNull(writer, "vendor", vendor != null ? vendor : "Canoo Engineering AG");
        String homepage = parameter.getHomepage();
        writeTagIfNotNull(writer, "homepage", new Attribute("href", homepage != null ? homepage : "http://www.canoo.com/ulc/"));
        writeTagIfNotNull(writer, "description", parameter.getDescription());
        String icon32 = parameter.getIcon(32);
        if (icon32 != null) {
            writeTagIfNotNull(writer, "icon", new Attribute[] {
                    new Attribute("href", EasyDeploymentUtilities.toFullURLIfNotNull(request, icon32)), new Attribute("width", "32"),
                    new Attribute("height", "32")});
        }
        String icon64 = parameter.getIcon(64);
        if (icon64 != null) {
            writeTagIfNotNull(writer, "icon", new Attribute[] {
                    new Attribute("href", EasyDeploymentUtilities.toFullURLIfNotNull(request, icon64)), new Attribute("width", "64"),
                    new Attribute("height", "64")});
        }
        String splashURL = EasyDeploymentUtilities.toFullURLIfNotNull(request, parameter.getSplash());
        if (splashURL != null) {
            writeTagIfNotNull(writer, "icon", new Attribute[] {new Attribute("href", splashURL), new Attribute("kind", "splash")});
        }
        writeTagIfNotNull(writer, "association", new Attribute[] {new Attribute("mime-type", appConfig.getMimeType()),
                new Attribute("extensions", appConfig.getExtensions())});
        writeShortCutTag(writer, parameter);
        writer.println("    </information>");
        writeSecurityAllPermissions(writer);
        writer.println("    <resources>");
        String javaVersion = appConfig.getJavaVersion();
        if (javaVersion == null) {
            javaVersion = "1.4+";
        }
        writeTagIfNotNull(writer, "j2se", new Attribute[] {new Attribute("version", javaVersion),
                new Attribute("href", "http://java.sun.com/products/autodl/j2se"), new Attribute("java-vm-args", appConfig.getVmArgs()),
                new Attribute("initial-heap-size", asMegaByte(appConfig.getInitialHeapSize())),
                new Attribute("max-heap-size", asMegaByte(appConfig.getMaxHeapSize())),});
        writer.println("        " + EasyDeploymentUtilities.toJNLPJarTags(getServletContext()));
        writer.println("    </resources>");
        writer.println("    <application-desc main-class=\""
                + EasyDeploymentUtilities.toJNLPLauncherClassName(getServletContext(), request) + "\">");
        writer.println("        <argument>url-string=" + EasyDeploymentUtilities.toUrlStringParam(request) + "</argument>");
        writer.println("    </application-desc>");
        writer.println("</jnlp> ");
        
    }
    
    private String asMegaByte(Integer heapSize) {
        return heapSize == null ? null : heapSize + "m";
    }
    
    private void writeSecurityAllPermissions(PrintWriter writer) {
        ApplicationConfiguration appConfig = ApplicationConfiguration.getInstance();
        if (appConfig.needsAllPermissions()) {
            writer.println("      <security><all-permissions/></security>");
        }
    }
    
    private void writeShortCutTag(PrintWriter writer, DeploymentParameter parameter) {
        ApplicationConfiguration appConfig = ApplicationConfiguration.getInstance();
        String menuGroup = parameter.getMenuGroup();
        if (appConfig.hasDesktopShortcut() || appConfig.hasMenuShortcut() || menuGroup != null) {
            writer.println("    <shortcut online=\"true\">");
            if (appConfig.hasDesktopShortcut()) {
                writer.println("    <desktop/>");
            }
            if (appConfig.hasMenuShortcut() || menuGroup != null) {
                if (menuGroup != null && menuGroup.length() > 0) {
                    writeTagIfNotNull(writer, "menu", new Attribute("submenu", menuGroup));
                } else {
                    writer.println("    <menu/>");
                }
            }
            writer.println("    </shortcut>");
            
        }
    }
    
    private void writeTagIfNotNull(PrintWriter writer, String tag, String value) {
        if (value != null) {
            writer.println("        <" + tag + ">" + value + "</" + tag + ">");
        }
    }
    
    private void writeTagIfNotNull(PrintWriter writer, String tag, Attribute attrib) {
        writeTagIfNotNull(writer, new Attribute(tag, null), new Attribute[] {attrib});
    }
    
    private void writeTagIfNotNull(PrintWriter writer, String tag, Attribute[] attrib) {
        writeTagIfNotNull(writer, new Attribute(tag, null), attrib);
    }
    
    private void writeTagIfNotNull(PrintWriter writer, Attribute tag, Attribute attrib) {
        writeTagIfNotNull(writer, tag, new Attribute[] {attrib});
    }
    
    private void writeTagIfNotNull(PrintWriter writer, Attribute tag, Attribute[] attrib) {
        String attributes = "";
        for (int i = 0; i < attrib.length; i++) {
            if (attrib[i].hasValue()) {
                attributes += attrib[i] + " ";
            }
        }
        if (attributes.length() > 0) {
            
            writer.print("        <" + tag.getKey() + " " + attributes);
            if (tag.hasValue()) {
                writer.println(">" + tag.getValue() + "</" + tag.getKey() + ">");
            } else {
                writer.println("/>");
            }
        }
    }
}